package com.bxm.localnews.news.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.mts.model.v20140618.SubmitJobsRequest;
import com.aliyuncs.mts.model.v20140618.SubmitJobsResponse;
import com.aliyuncs.profile.DefaultProfile;
import com.bxm.localnews.news.config.OssProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.vo.TranscodeMap;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * oss视频转码工具类
 *
 * @author stephen
 */
@Component
@Slf4j
public class OssTransCodeUtils {
    private static final String MP4_SUFFIX = ".mp4";

    private static final String MP4_FOLDER = "/mp4/";

    private static final String H264 = "h264";

    @Resource
    private OssProperties ossProperties;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    /**
     * 根据原视频地址转码，并缓存原视频对应的转码后视频地址
     *
     * @param originUrl 视频地址
     * @param bizId     归属的业务ID
     * @param callback  处理完成后的回调函数
     */
    public void transcodeByOriginUrl(String originUrl, Long bizId, TranscodeCallback callback) {
        log.info("执行转码操作，业务ID：{},原始地址：{}", bizId, originUrl);

        // 若转码开关未打开，或者原视频地址为空，或者不包含指定前缀，则直接返回
        if (!ossProperties.getTranscodeSwitch()
                || StringUtils.isBlank(originUrl)
                || !originUrl.contains(ossProperties.getUrlPrefix())) {
            log.info("不进行转码处理，转码开关：{},地址前缀：{},原始地址：{}",
                    ossProperties.getTranscodeSwitch(),
                    ossProperties.getUrlPrefix(),
                    originUrl);
            return;
        }

        log.debug("正常执行转码任务，地址：{}，bizId：{}", originUrl, bizId);

        KeyGenerator videoTranscodeKey = RedisConfig.VIDEO_TRANSCODE;
        // 若缓存中已存在，说明已经转码或者无需转码
        if (redisSetAdapter.exists(videoTranscodeKey, originUrl)) {
            log.info("缓存中已经存在:{}，不重复处理", originUrl);
            return;
        }

        String dealOriginUrl = originUrl.replace(ossProperties.getUrlPrefix(), "");
        int index = originUrl.lastIndexOf("/");
        if (index < 0) {
            log.info("视频地址错误：{}", originUrl);
            return;
        }

        // 生成转码目标地址
        index = originUrl.lastIndexOf(".");
        if (index < 0) {
            log.info("视频地址错误：{}", originUrl);
            return;
        }

        String fileSuffix = originUrl.substring(index + 1);
        String destUrl = dealOriginUrl.replace("/" + fileSuffix + "/", MP4_FOLDER)
                .replace("." + fileSuffix, MP4_SUFFIX);

        // 开始转码，并缓存原视频地址和转码后地址
        boolean result = transCode(dealOriginUrl, destUrl);

        if (result) {
            destUrl = ossProperties.getUrlPrefix() + destUrl;
            TranscodeMap transcodeMap = new TranscodeMap();
            transcodeMap.setOriginUrl(originUrl);
            transcodeMap.setDestUrl(destUrl);

            redisSetAdapter.add(videoTranscodeKey, originUrl);

            log.info("转码成功，业务ID：{},原始地址：{}，处理后地址：{}",
                    bizId, originUrl, destUrl);

            callback.apply(destUrl);
        } else {
            log.info("调用第三方服务失败，请求地址：{}", originUrl);
        }
    }

    /**
     * 提交oss转码任务
     */
    private boolean transCode(String ossInputObject, String ossOutputObject) {
        IAcsClient client = getiAcsClient();

        // 创建API请求并设置参数
        SubmitJobsRequest request = new SubmitJobsRequest();
        // Input
        JSONObject input = new JSONObject();
        input.put("Location", ossProperties.getOssLocation());
        input.put("Bucket", ossProperties.getOssBucket());

        try {
            input.put("Object", URLEncoder.encode(ossInputObject, "utf-8"));
        } catch (UnsupportedEncodingException e) {
            log.error(e.getMessage(), e);
            return false;
        }

        request.setInput(input.toJSONString());

        // Output
        String outputObject;
        try {
            outputObject = URLEncoder.encode(ossOutputObject, "utf-8");
        } catch (UnsupportedEncodingException e) {
            log.error(e.getMessage(), e);
            return false;
        }

        JSONArray imageWatermarkArray = buildImageWatermark();

        JSONObject output = new JSONObject();
        output.put("OutputObject", outputObject);
        // Ouput->TemplateId
        output.put("TemplateId", ossProperties.getTemplateId());
        // 设置水印
        output.put("WaterMarks", imageWatermarkArray.toJSONString());
        JSONArray outputs = new JSONArray();
        outputs.add(output);
        request.setOutputs(outputs.toJSONString());
        request.setOutputBucket(ossProperties.getOssBucket());
        request.setOutputLocation(ossProperties.getOssLocation());
        // PipelineId
        request.setPipelineId(ossProperties.getPipelineId());
        // 发起请求并处理应答或异常
        SubmitJobsResponse response;
        try {
            response = client.getAcsResponse(request);
            if (response.getJobResultList().get(0).getSuccess()) {
                return true;
            }

            log.error("提交转码任务失败：{},视频地址：{}",
                    JSON.toJSONString(response),
                    ossInputObject);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return false;
    }

    private JSONArray buildImageWatermark() {
        JSONObject videoWatermarkInput = new JSONObject();
        videoWatermarkInput.put("Location", ossProperties.getOssLocation());
        videoWatermarkInput.put("Bucket", ossProperties.getOssBucket());
        try {
            videoWatermarkInput.put("Object", URLEncoder.encode(ossProperties.getWaterMarkImagePath(), "utf-8"));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("videoWatetmark Input URL encode failed");
        }

        JSONObject videoWatermark = new JSONObject();
        videoWatermark.put("WaterMarkTemplateId", ossProperties.getWaterMarkTemplateId());
        videoWatermark.put("Type", "Image");
        videoWatermark.put("InputFile", videoWatermarkInput);
        videoWatermark.put("ReferPos", ossProperties.getWaterMarkImagePosition());
        videoWatermark.put("Height", ossProperties.getWaterMarkImageHeight());
        videoWatermark.put("Dx", ossProperties.getWaterMarkDx());
        videoWatermark.put("Dy", ossProperties.getWaterMarkDy());

        JSONArray watermarks = new JSONArray();
        watermarks.add(videoWatermark);

        return watermarks;
    }

    /**
     * 创建DefaultAcsClient实例并初始化
     */
    private IAcsClient getiAcsClient() {
        DefaultProfile profile = DefaultProfile.getProfile(
                ossProperties.getMpsRegionId(),
                ossProperties.getAccessKeyId(),
                ossProperties.getAccessKeySecret());
        return new DefaultAcsClient(profile);
    }

    private Boolean download(String path, String filePath) {
        try {
            OSSClient ossClient = this.getOssClient();
            GetObjectRequest getObjectRequest = new GetObjectRequest(ossProperties.getOssBucket(), path);
            ossClient.getObject(getObjectRequest, new File(filePath));
            ossClient.shutdown();
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    private OSSClient getOssClient() {
        return new OSSClient(ossProperties.getEndpoint(), ossProperties.getAccessKeyId(), ossProperties.getAccessKeySecret());
    }
}
