package com.bxm.localnews.news.action.impl;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.integration.MessageService;
import com.bxm.localnews.integration.MissionIntegrationService;
import com.bxm.localnews.news.action.PostClickService;
import com.bxm.localnews.news.activity.ForumTeachService;
import com.bxm.localnews.news.constant.AppConst;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.ForumPostRecordMapper;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.enums.TaskEnum;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.image.ImageHelper;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.ForumPostRecord;
import com.bxm.localnews.news.model.vo.ForumPostStatistic;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.post.PostCountService;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.newidea.component.redis.HyperLogLogAdapter;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.tools.IPUtil;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

/**
 * @author liujia
 * @date 1/18/21 4:06 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class PostClickServiceImpl implements PostClickService {

    private PostCountService postCountService;

    private ForumPostMapper forumPostMapper;

    private ForumPostRecordMapper forumPostRecordMapper;

    private SequenceCreater sequenceCreater;

    private MissionIntegrationService missionIntegrationService;

    private ImageHelper imageHelper;

    private MessageService messageService;

    private HyperLogLogAdapter hyperLogLogAdapter;

    private ForumTeachService forumTeachService;

    /**
     * 设置帖子的阅读数信息
     *
     * @param adminForumPost        要更新or新增的数据
     * @param existedAdminForumPost 已入库的数据
     */
    @Override
    public void setClickCountInfo(AdminForumPost adminForumPost, AdminForumPost existedAdminForumPost) {
        // 编辑 判断是否 爆料，热，精华，现金奖励
        byte yes = 1;
        boolean isUpdate = nonNull(existedAdminForumPost);

        boolean isBroke = isUpdate ? (!Objects.equals(adminForumPost.getIsBroke(), existedAdminForumPost.getIsBroke())
                && Objects.equals(adminForumPost.getIsBroke(), yes)) : Objects.equals(adminForumPost.getIsBroke(), yes);

        boolean isBrilliant = isUpdate ? (!Objects.equals(adminForumPost.getIsBrilliant(), existedAdminForumPost.getIsBrilliant())
                && Objects.equals(adminForumPost.getIsBrilliant(), yes)) : Objects.equals(adminForumPost.getIsBrilliant(), yes);

        boolean isRed = isUpdate ? (!Objects.equals(adminForumPost.getIsRed(), existedAdminForumPost.getIsRed())
                && Objects.equals(adminForumPost.getIsRed(), yes)) : Objects.equals(adminForumPost.getIsRed(), yes);

        boolean isRecommende = isUpdate ? (!Objects.equals(adminForumPost.getIsRecommend(), existedAdminForumPost.getIsRecommend())
                && Objects.equals(adminForumPost.getIsRecommend(), yes)) : Objects.equals(adminForumPost.getIsRecommend(), yes);

        boolean isCash = isUpdate ? (!Objects.equals(adminForumPost.getIsCash(), existedAdminForumPost.getIsCash())
                && Objects.equals(adminForumPost.getIsCash(), yes)) : Objects.equals(adminForumPost.getIsCash(), yes);

        // 是否是推荐的帖子
        boolean isRecommend = isBroke || isBrilliant || isRed || isRecommende || isCash;

        if (isNull(existedAdminForumPost)
                || isNull(existedAdminForumPost.getInitialBasicNum())
                || isNull(existedAdminForumPost.getReviewCount())) {
            // 第一次添加帖子 或者历史帖子编辑过，则初始化帖子的数据

            // 初始基数
            long initialBasicNum = postCountService.getInitialBasicNum(adminForumPost.getAreaCode(), isRecommend);

            adminForumPost.setInitialBasicNum(initialBasicNum);
            // 该帖子最终需要达到的用户阅读数量
            adminForumPost.setFinalClickCount(postCountService.getFinalClickCount(
                    // 真实阅读数量 在历史帖子里应该是有值的 要用上
                    isNull(existedAdminForumPost) || isNull(existedAdminForumPost.getReviewCount())
                            ? 0 : existedAdminForumPost.getReviewCount(),

                    initialBasicNum));

        } else if (isRecommend) {
            // 编辑时，当前帖子第一次设置为精华帖（其实重复设置也会） 则重新计算要达到的阅读数

            // 初始基数重新计算
            long initialBasicNum = postCountService.getInitialBasicNum(adminForumPost.getAreaCode(), true);
            // 取上一次和这次重新计算的 拿最大的 避免重复生成时，基数变小了
            if (initialBasicNum > existedAdminForumPost.getInitialBasicNum()) {
                // 变大了 需要更新基数
                adminForumPost.setInitialBasicNum(initialBasicNum);
            } else {
                initialBasicNum = existedAdminForumPost.getInitialBasicNum();
            }
            // 以及真实阅读数
            long reviewCount = existedAdminForumPost.getReviewCount();

            // 该帖子最终需要达到的用户阅读数量
            adminForumPost.setFinalClickCount(postCountService.getFinalClickCount(reviewCount, initialBasicNum));
        }
    }

    @Override
    public void doConsumeClickCount(Long postId, Long userId, ForumPostVo forumPostVo) {
        if (Objects.isNull(forumPostVo)) {
            forumPostVo = forumPostMapper.selectWithoutContent(postId);
            if (Objects.isNull(forumPostVo)) {
                return;
            }
        }

        // 获取帖子真实阅读数
        long reviewCount = Objects.isNull(forumPostVo.getReviewCount()) ? 0 : forumPostVo.getReviewCount();
        // 加上这次的
        reviewCount += 1;

        // 是否初始化过基数
        boolean isUnInit = Objects.isNull(forumPostVo.getInitialBasicNum())
                || forumPostVo.getInitialBasicNum().intValue() == 0;

        // 获取基数 如果未曾计算过 则进行初始化
        long initialBasicNum;
        if (isUnInit) {
            // 编辑 判断是否 爆料，热，精华，现金奖励
            byte yes = 1;
            boolean isRecommend = Objects.equals(forumPostVo.getIsBroke(), yes)
                    || Objects.equals(forumPostVo.getIsBrilliant(), (int) yes)
                    || Objects.equals(forumPostVo.getIsRed(), (int) yes)
                    || Objects.equals(forumPostVo.getIsRecommend(), yes) ||
                    Objects.equals(forumPostVo.getIsCash(), (int) yes);

            initialBasicNum = postCountService.getInitialBasicNum(forumPostVo.getAreaCode(), isRecommend);
        } else {
            // 否则使用之前初始化的基数
            initialBasicNum = forumPostVo.getInitialBasicNum();
        }


        long finalClickCount = Objects.isNull(forumPostVo.getFinalClickCount()) ? 0 : forumPostVo.getFinalClickCount();
        // 计算应有达到的阅读量
        long currentClickCount = postCountService.getFinalClickCount(reviewCount, initialBasicNum);
        // 取最大的
        currentClickCount = currentClickCount > finalClickCount ? currentClickCount : finalClickCount;

        ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
        // 用户看到的阅读数
        forumPostStatistic.setClickCount(1);
        // 帖子真实阅读数
        forumPostStatistic.setReviewCount(1);
        // 计算要增加的数量
        forumPostStatistic.setFinalClickCount(currentClickCount - finalClickCount);
        // 初始化基数
        if (isUnInit) {
            forumPostStatistic.setInitialBasicNum(initialBasicNum);
        }
        forumPostStatistic.setId(postId);

        if (log.isDebugEnabled()) {
            log.debug("帖子: {} 增加阅读数, 原有数据: {} 新增后的阅读数据: {}", postId, toJSONString(forumPostVo),
                    toJSONString(forumPostStatistic));
        }

        this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);
    }

    @Override
    public void doConsumeActiveViewCount(Long postId, Long userId) {
        ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
        forumPostStatistic.setReviewCount(1);
        forumPostStatistic.setId(postId);
        this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);
    }

    @Override
    public void removePostReadLimit(Long postId) {
        if (null != postId) {
            hyperLogLogAdapter.remove(RedisConfig.FORUM_RECORD_FROM_H5.copy().appendKey(postId));
        }
    }

    @Override
    public void doRecordForumPost(Long userId, Long postId, Long shareUserId, Integer platform, ForumPostVo forumPostVo) {
        ForumPostRecord pRecord = this.forumPostRecordMapper.selectByIds(postId, userId, null);
        Date d = new Date();
        if (pRecord != null) {
            pRecord.setUpdateTime(d);
            this.forumPostRecordMapper.updateByPrimaryKeySelective(pRecord);
        } else {
            ForumPostRecord nr = new ForumPostRecord();
            nr.setId(sequenceCreater.nextLongId());
            nr.setPostId(postId);
            nr.setUserId(userId);
            nr.setUpdateTime(d);
            nr.setCheckTime(0);
            nr.setGetGold(AppConst.UN_GOLD);
            nr.setAddTime(d);
            nr.setLastLocation("1");
            nr.setAdViewType((byte) 1);
            this.forumPostRecordMapper.insertSelective(nr);
        }

        this.doConsumeClickCount(postId, userId, forumPostVo);
    }

    @Async
    @Override
    public void doAsyncReadPost(Long userId, Long postId, Long shareUserId, Integer platform, ForumPostVo forumPostVo, String ip) {
        //添加帖子阅读记录
        this.doRecordForumPost(userId, postId, shareUserId, platform, forumPostVo);
        //在小程序中每日首次打开,分享人随机获得20-50的小红花
        if (null != shareUserId && shareUserId != 0) {
            if (PlatformEnum.APPLET.getCode() == platform && !shareUserId.equals(userId)) {
                //用户首次打开小程序完成
                String content = "转发【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】被阅读";
                NewsCompleTaskDTO firstBrowse = missionIntegrationService.compleTask(shareUserId, TaskEnum.TASK_FIRST_BROWSE.name(), userId.toString(), content);

                log.info("用户={}在小程序中打开帖子,新闻id={},分享人的shareUserId={},完成任务接口返回={}", userId, forumPostVo.getId(), shareUserId, toJSONString(firstBrowse));

                if (Objects.nonNull(firstBrowse) && firstBrowse.getGoldNum() > 0) {
                    //提取必要信息
                    String extendContent = null;
                    List<PostImgVo> imgList = imageHelper.getDetailFromPost(forumPostVo.getPostImgList());
                    String extendUrl = CollectionUtils.isNotEmpty(imgList) ? null : imgList.get(0).getImgUrl();
                    if (StringUtils.isEmpty(extendUrl)) {
                        extendContent = ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField());
                    }
                    //发送消息
                    messageService.addMessage(
                            shareUserId,
                            userId,
                            firstBrowse,
                            forumPostVo.getId(),
                            ReplyTypeEnum.POST_REPLY.getCode(),
                            extendUrl,
                            extendContent,
                            forumPostVo.getTitle());
                }
            }
        }

        //如果是[站外]访问[待审核]的帖子，记录访问次数
        if (null != platform && PlatformEnum.WEB.getCode() == platform
                && Objects.equals(PostStatusEnum.APPROVING.getCode(), forumPostVo.getStatus())) {
            KeyGenerator keyGenerator = RedisConfig.FORUM_RECORD_FROM_H5.copy().appendKey(postId);
            //将ip从字符串转为数值，便于存储
            long ipNum;
            if (StringUtils.isNotBlank(ip)) {
                ipNum = IPUtil.ip2long(ip);
            } else {
                ipNum = RandomUtils.nextLong();
            }
            hyperLogLogAdapter.add(keyGenerator, ipNum);
        }

        // 站内访问 增加教育完成的信息
        if (Objects.equals(PlatformEnum.ANDROID.getCode(), platform)
                || Objects.equals(PlatformEnum.IOS.getCode(), platform)) {
            try {
                forumTeachService.updateTeachInfo(userId, postId);
            } catch (Exception e) {
                log.error("更新用户的教育缓存失败, userId: {} postId: {}", userId, postId, e);
            }

        }
    }
}
