package com.bxm.localnews.news.action.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.news.action.PostLikeService;
import com.bxm.localnews.news.config.ClientConfigProperties;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.detail.helper.ForumPostImageHelper;
import com.bxm.localnews.news.domain.AdminForumPostLikeMapper;
import com.bxm.localnews.news.domain.AdminForumPostMapper;
import com.bxm.localnews.news.domain.ForumPostLikeMapper;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.model.dto.ForumPostLikeDTO;
import com.bxm.localnews.news.model.dto.PostLikeDto;
import com.bxm.localnews.news.model.param.VirtualUserRandomQueryParam;
import com.bxm.localnews.news.model.vo.*;
import com.bxm.localnews.news.util.FormPostContentUtil;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.localnews.vo.VirtualUserInfo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import java.util.*;
import java.util.concurrent.ThreadLocalRandom;

import static com.alibaba.fastjson.JSON.toJSONString;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author liujia
 * @date 1/15/21 9:38 AM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class PostLikeServiceImpl implements PostLikeService {

    private ForumPostLikeMapper forumPostLikeMapper;

    private UserIntegrationService userIntegrationService;

    private UserProperties userProperties;

    private SequenceCreater sequenceCreater;

    private ForumPostMapper forumPostMapper;

    private LocationIntegrationService locationIntegrationService;

    private UserAccountIntegrationService userAccountIntegrationService;

    private AdminForumPostLikeMapper adminForumPostLikeMapper;

    private AdminForumPostMapper adminForumPostMapper;

    private VirtualUserIntegrationService virtualUserIntegrationService;

    private ClientConfigProperties clientConfigProperties;

    private RedisStringAdapter redisStringAdapter;

    private ForumPostImageHelper forumPostImageHelper;

    private MessageService messageService;

    private BizLogIntegrationService bizLogIntegrationService;

    @Override
    public void doLikeForumPost(Long userId, Long postId, int type, String areaCode, Integer platform) {
        log.debug("[旧版]用户点赞帖子，参数->postId:{},userId:{},type:{},platform:{}", postId, userId, type, platform);
        ForumPostLike postLike = forumPostLikeMapper.selectByUserIdAndPostId(userId, postId);
        UserBean userBean = userIntegrationService.selectUserFromCache(userId);
        //生成一条帖子点赞记录
        this.generateForumPostLike(postLike, type, postId, userId, userBean);
        //更新帖子点赞数量
        doUpdateForumInfo(type, postId);
        //帖子点赞流程
        getSelfBean().doAsyncLikePost(userId, postId, type, areaCode, (byte) 1, platform);
    }

    private PostLikeService getSelfBean() {
        return SpringContextHolder.getBean(PostLikeService.class);
    }

    @Override
    public PostLikeDto doLikeForumPost(Long postId, Long userId, String areaCode, Integer platform) {
        PostLikeDto postLikeDto = new PostLikeDto();
        UserBean userBean = userIntegrationService.selectUserFromCache(userId);
        if (isBlank(userBean.getHeadImg()) || isBlank(userBean.getNickname())) {
            log.info("帖子点赞用户头像或昵称为空,userInfo={}", toJSONString(userBean));
        }
        ForumPostLike postLike = forumPostLikeMapper.selectByUserIdAndPostId(userId, postId);
        //判断帖子是否被用户点赞,或者为是取消点赞状态
        if (isNull(postLike) || postLike.getType() == 0) {
            //生成帖子点赞记录
            long id = this.generateForumPostLike(postLike, 1, postId, userId, userBean);

            //帖子概览数据的更新
            doUpdateForumInfo(1, postId);

            //判断是否是自己给自己点赞，如果是则只加点赞数
            ForumPostVo postVo = forumPostMapper.selectWithoutContent(postId);
            boolean flag = Objects.nonNull(userId) && Objects.nonNull(postVo.getUserId()) && userId.equals(postVo.getUserId());
            if (flag) {
                postLikeDto.setCode(PostLikeDto.SUCCESS_CODE);
                postLikeDto.setCopyWriter(null);
                postLikeDto.setShowFlower(PostLikeDto.FAILURE_CODE);
            } else {
                postLikeDto = calculateFlower(postVo, userId, id, areaCode, platform);
            }

            //  调用用户增加点赞数接口
            userIntegrationService.updateUserLikeNumByUserId(postVo.getUserId());
        } else {
            postLikeDto = null;
        }
        return postLikeDto;
    }

    /**
     * 点赞之后对鲜花的增减以及返回包装类
     *
     * @param postVo   帖子
     * @param userId   用户id
     * @param id       帖子id
     * @param areaCode 区域编码
     * @param platform 平台
     * @return 包装类
     */
    private PostLikeDto calculateFlower(ForumPostVo postVo, Long userId, Long id, String areaCode, int platform) {
        //根据是否精华帖获得赞赏鲜花的数量
        final int postLikeConsume = getPostLikeConsume(postVo.getIsBrilliant());
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        //判断帖子是否开启占位符
        FormPostContentUtil.replace(postVo, location);
        //点赞用户减少红花
        String consumeContent = "点赞【" + ExtendFactory.getPostContent(postVo.getTitle(), postVo.getTextField()) + "】 消耗小红花";
        Boolean consume = userAccountIntegrationService.addGold(AccountGoldParam.buildPostLikeExpendParam(userId, postLikeConsume, id, consumeContent));

        PostLikeDto postLikeDto = new PostLikeDto();
        if (consume) {
            //帖子作者增加红花
            String addContent = "你发布的【" + ExtendFactory.getPostContent(postVo.getTitle(), postVo.getTextField()) + "】被赞";
            userAccountIntegrationService.addGold(AccountGoldParam.buildPostLikeAddParam(postVo.getUserId(), postLikeConsume, id, addContent));
            //红花增减成功，填充返回实体
            postLikeDto.setAmount(postLikeConsume);
            postLikeDto.setCode(PostLikeDto.SUCCESS_CODE);
            //自己给自己点赞没有文案
            fillPostLikeDto(postLikeDto, userId, postVo.getUserId(), postLikeConsume);
            //异步推送信息
            getSelfBean().doAsyncLikePost(userId, postVo.getId(), 1, areaCode, (byte) 0, platform);
        } else {
            //扣除红花失败,回滚数据
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            //扣款失败
            postLikeDto.setCode(PostLikeDto.FAILURE_CODE);
            postLikeDto.setShowFlower(PostLikeDto.FAILURE_CODE);
            postLikeDto.setCopyWriter("你的小红花余额不足，快转发文章赚小红花");
        }
        return postLikeDto;
    }

    /**
     * 生成一条帖子点赞记录
     *
     * @param postLike 点赞信息
     * @param type     类型
     * @param postId   帖子id
     * @param userId   用户id
     * @param userBean 用户信息
     */
    private Long generateForumPostLike(ForumPostLike postLike, int type, Long postId, Long userId, UserBean userBean) {
        String updateHeadImg = isBlank(userBean.getHeadImg()) ? userProperties.getDefaultHeadImgUrl() : userBean.getHeadImg();
        String updateNickname = isBlank(userBean.getNickname()) ? null : userBean.getNickname();
        long id;
        if (isNull(postLike)) {
            id = sequenceCreater.nextLongId();
            ForumPostLike forumPostLike = ForumPostLike.buildPostLike(postId, userId);
            forumPostLike.setId(id);
            forumPostLike.setType((byte) type);
            forumPostLike.setHeadImg(updateHeadImg);
            forumPostLike.setUserNickname(updateNickname);
            //插入点赞表
            forumPostLikeMapper.insertSelective(forumPostLike);
        } else {
            id = postLike.getId();
            postLike.setType((byte) type);
            postLike.setHeadImg(updateHeadImg);
            postLike.setUserNickname(updateNickname);
            forumPostLikeMapper.updateByPrimaryKeySelective(postLike);
        }
        return id;
    }

    @Override
    public List<ForumPostLikeDTO> getPostLikeDetail(Long id, Long userId, String devcId) {
        ArrayList<ForumPostLikeDTO> result = Lists.newArrayList();
        List<ForumPostLike> forumPostLikes = forumPostLikeMapper.selectPostById(id);
        forumPostLikes.forEach(e -> {
            ForumPostLikeDTO postLikeDTO = new ForumPostLikeDTO();
            BeanUtils.copyProperties(e, postLikeDTO);
            if (isBlank(postLikeDTO.getHeadImg())) {
                postLikeDTO.setHeadImg(userProperties.getDefaultHeadImgUrl());
            }
            result.add(postLikeDTO);
        });
        return result;
    }

    @Override
    public Message doAddPostLike(Long postId, Date startTime, Date endTime, Integer num) {
        AdminForumPost adminForumPost = adminForumPostMapper.selectByPrimaryKey(postId);
        if (Objects.nonNull(adminForumPost)) {
            Long difference = endTime.getTime() - startTime.getTime();
            addPostLike(adminForumPost, num, startTime, difference);
        }
        return Message.build(true);
    }

    /**
     * 帖子添加虚拟用户点赞
     *
     * @param adminForumPost 帖子
     * @param num            点赞数
     */
    private void addPostLike(AdminForumPost adminForumPost, Integer num, Date startTime, Long difference) {
        //获取足够多的虚拟用户id
        List<Long> exitUserId = adminForumPostLikeMapper.selectVirtualUserIdByPostId(adminForumPost.getId());
        //正式环境屏蔽本地万事通马甲号
        exitUserId.add(29211L);

        String areaCode = isNotBlank(adminForumPost.getAreaCode()) ? adminForumPost.getAreaCode() : "";

        //当帖子多地区投放时取第一个投放地区码
        areaCode = Splitter.on(",").split(areaCode).iterator().next();

        VirtualUserRandomQueryParam randomQueryParam = new VirtualUserRandomQueryParam();
        randomQueryParam.setNum(num);
        randomQueryParam.setType(null);
        randomQueryParam.setExitUserId(exitUserId);
        randomQueryParam.setAreaCode(areaCode);

        List<VirtualUser> virtualUsers = virtualUserIntegrationService.getRandom(randomQueryParam);
        ArrayList<AdminForumPostLike> list = Lists.newArrayList();
        for (VirtualUser virtualUser : virtualUsers) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(startTime);
            //随机时间生成马甲号点赞
            int amount = (int) (1 + Math.random() * (difference));
            calendar.add(Calendar.MILLISECOND, amount);
            AdminForumPostLike postLike = AdminForumPostLike.builder()
                    .addTime(calendar.getTime())
                    .headImg(virtualUser.getHeadImg())
                    .id(sequenceCreater.nextLongId())
                    .isShow((byte) 0)
                    .isVest((byte) 1)
                    .postId(adminForumPost.getId())
                    .type((byte) 1)
                    .userId(virtualUser.getId())
                    .userNickname(virtualUser.getNickname())
                    .build();

            list.add(postLike);
        }
        //批量插入虚拟用户点赞信息
        MybatisBatchBuilder.create(AdminForumPostLikeMapper.class, list).run(AdminForumPostLikeMapper::insertSelective);
    }


    @Override
    public void simulateData(AdminForumPost adminForumPost) {
        //审核通过帖子增加虚拟用户点赞
        int vestCount = adminForumPostLikeMapper.selectByPostIdAndIsVest(adminForumPost.getId());
        if (vestCount < 5) {
            //添加的数量
            ThreadLocalRandom random = ThreadLocalRandom.current();
            int num = random.nextInt(10, 20);
            //添加点赞开始时间
            Date date = new Date();
            //添加时间距离开始时间的毫秒值
            long difference = 7200000L;

            this.addPostLike(adminForumPost, num, date, difference);
        }
    }

    @Async
    @Override
    public void doAsyncLikePost(Long userId, Long postId, int type, String areaCode, byte isOld, Integer platform) {
        if (1 == type) {
            ForumPostVo forumPostVo = forumPostMapper.selectWithoutContent(postId);
            //添加马甲号点赞(旧版没有马甲号点赞)
            this.generateVirtualLike(forumPostVo, isOld);
            //数据埋点(马甲号点赞时没有platform)
            if (Objects.nonNull(platform)) {
                bizLogIntegrationService.forumLikeSuccessed(userId, postId, areaCode, platform);
            } else {
                //马甲号点赞调用新版文文案接口 isOld = 0为新版文案接口
                isOld = 0;
            }
            //消息推送(新旧版文案不一)
            FormPostContentUtil.replace(forumPostVo, null);
            forumPostImageHelper.exchangeDetailPost(forumPostVo);
            this.messageService.pushPostLikeMessage(forumPostVo, userId, isOld, getPostLikeConsume(forumPostVo.getIsBrilliant()));
        }
    }

    /**
     * 生成机器人点赞信息
     *
     * @param forumPostVo 帖子信息
     * @param isOld       是否是旧版点赞 1：旧版 0：新版
     */
    private void generateVirtualLike(ForumPostVo forumPostVo, int isOld) {
        if (isOld != 0) {
            return;
        }
        //当用户已经在该条帖子发布评论时,或者获得虚拟用户为空时 重新获得虚拟用户
        List<VirtualUserInfo> virtualUserList = userIntegrationService.getVirtualUserListNew(20, Lists.newArrayList(1, 2, 3), forumPostVo.getAreaCode());
        if (CollectionUtils.isEmpty(virtualUserList)) {
            return;
        }
        for (VirtualUserInfo virtualUserInfo : virtualUserList) {
            ForumPostLike like = forumPostLikeMapper.selectByUserIdAndPostId(virtualUserInfo.getId(), forumPostVo.getId());
            if (Objects.isNull(like)) {
                Calendar calendar = Calendar.getInstance();
                calendar.add(Calendar.MINUTE, 3);
                ForumPostLike postLike = ForumPostLike.buildVestPostLike(sequenceCreater.nextLongId(),
                        forumPostVo.getId(),
                        virtualUserInfo.getId());
                postLike.setAddTime(calendar.getTime());
                postLike.setHeadImg(virtualUserInfo.getHeadImg());
                postLike.setUserNickname(virtualUserInfo.getNickname());
                forumPostLikeMapper.insertSelective(postLike);

                log.debug("添加马甲点赞成功,点赞信息:{}", toJSONString(postLike));

                break;
            }
        }
    }

    /**
     * 返回包装类，如已经点赞过两篇文章，则不返回文案
     */
    @Override
    public void fillPostLikeDto(PostLikeDto postLikeDto, Long userId, Long authorUserId, int postLikeConsume) {
        String dateKey = DateFormatUtils.format(new Date(), "yyyy-MM-dd");
        KeyGenerator forumPostLikeTimeKey = RedisConfig.FORUMPOST_LIEK_TIME.copy().appendKey(dateKey).appendKey(userId);
        //点赞次数
        long likeCount = redisStringAdapter.incrementWithDefault(forumPostLikeTimeKey, 1, 1, 86400);
        if (likeCount <= 2L) {
            postLikeDto.setCopyWriter("已赞赏楼主" + postLikeConsume + "朵小红花");
            postLikeDto.setShowFlower(PostLikeDto.SUCCESS_CODE);
        } else {
            postLikeDto.setCopyWriter(null);
            postLikeDto.setShowFlower(PostLikeDto.FAILURE_CODE);
        }
    }

    @Async
    @Override
    public void asyncSendPostLikeMessage(List<ForumPostLike> postLikes) {
        if (CollectionUtils.isNotEmpty(postLikes)) {
            postLikes.forEach(e -> {
                ForumPostVo postVo = forumPostMapper.selectWithoutContent(e.getPostId());
                //给用户添加赞数
                userIntegrationService.updateUserLikeNumByUserId(postVo.getUserId());

                final int postLikeConsume = getPostLikeConsume(postVo.getIsBrilliant());
                //用户已删除的帖子不发送推送
                boolean flag = !Objects.equals(postVo.getStatus(), PostStatusEnum.USER_DELETE.getCode());
                if (flag) {
                    //用户添加红花，帖子作者增加红花
                    String addContent = "你发布的【" + ExtendFactory.getPostContent(postVo.getTitle(), postVo.getTextField()) + "】被赞";
                    AccountGoldParam usableParam = AccountGoldParam.buildPostLikeAddParam(postVo.getUserId(), postLikeConsume, e.getId(), addContent);
                    this.userAccountIntegrationService.addGold(usableParam);
                    //推送点赞信息(机器人点赞没有platform)
                    getSelfBean().doAsyncLikePost(e.getUserId(), e.getPostId(), 1, postVo.getAreaCode(), (byte) 1, null);
                }
            });
        }
    }

    @Override
    public void doSyncPostLikeInfo() {
        List<Long> list = forumPostLikeMapper.selectNotExitsUserInfo();
        List<ForumPostLike> paramList = Lists.newArrayList();
        list.forEach(e -> {
            ForumPostLike postLike = new ForumPostLike();
            postLike.setUserId(e);
            UserBean userInfo = userIntegrationService.selectUserFromCache(e);
            if (Objects.nonNull(userInfo)) {
                postLike.setUserNickname(userInfo.getNickname());
                postLike.setHeadImg(userInfo.getHeadImg());
                paramList.add(postLike);
            }
        });
        MybatisBatchBuilder.create(ForumPostLikeMapper.class, paramList).run(ForumPostLikeMapper::updateUserInfoByUserId);
    }

    @Override
    public int getPostLikeConsume(Integer isBrilliant) {
        if (Objects.isNull(isBrilliant) || isBrilliant == 0) {
            return clientConfigProperties.getPostCommonLikeConsume();
        } else {
            return clientConfigProperties.getPostBrilliantLikeConsume();
        }
    }

    @Override
    public void doRefreshLikeInfo(Long postId) {
        //根据时间查询要修改的帖子点赞信息
        List<ForumPostLike> postLikes = forumPostLikeMapper.selectNotShowPostLike(postId);
        postLikes.forEach(e -> {
            e.setIsShow((byte) 1);
            forumPostLikeMapper.updateByPrimaryKeySelective(e);
            //帖子概览数据的更新
            doUpdateForumInfo(1, e.getPostId());
        });
        //异步发送消息推送
        asyncSendPostLikeMessage(postLikes);
        if (!CollectionUtils.isEmpty(postLikes)) {
            postLikes.size();
        }
    }

    @Async
    @Override
    public void doTriggerUpdateForumPostLikeInfo(Long userId, String nickname, String headImg) {
        if (StringUtils.isBlank(nickname) && StringUtils.isBlank(headImg)) {
            return;
        }
        forumPostLikeMapper.updateNickNameAndHeadImgByUserId(userId, nickname, headImg);
    }

    @Override
    public void doUpdateForumInfo(int type, Long postId) {

        //更新帖子点赞数量
        ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
        forumPostStatistic.setId(postId);
        forumPostStatistic.setLikeCount(type == 0 ? -1 : 1);

        //维护帖子点赞概览信息
        List<ForumPostLike> forumPostLikes = forumPostLikeMapper.selectByPostIdAndOrderByAddtime(postId);
        if (CollectionUtils.isNotEmpty(forumPostLikes)) {
            forumPostStatistic.setLikeInfo(toJSONString(forumPostLikes));
        }
        this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);
    }

    @Async
    @Override
    public void addPostLike(ForumPostVo forumPostVo, Integer num) {
        //获取足够多的虚拟用户id
        List<VirtualUserInfo> virtualUserList = userIntegrationService.getVirtualUserListNew(num * 2, Lists
                .newArrayList(1, 2, 3), forumPostVo.getAreaCode());

        if (log.isDebugEnabled()) {
            log.debug("发布帖子 异步获取点赞马甲数据 ： {}", JSON.toJSONString(virtualUserList));
        }

        ArrayList<ForumPostLike> list = Lists.newArrayList();
        for (VirtualUserInfo virtualUser : virtualUserList) {
            ForumPostLike like = forumPostLikeMapper.selectByUserIdAndPostId(virtualUser.getId(), forumPostVo.getId());
            //该虚拟用户信息不存在时,添加虚拟用户点赞信息
            if (Objects.isNull(like)) {
                Calendar calendar = Calendar.getInstance();
                //随机时间生成马甲号点赞
                int amount = (int) (1 + Math.random() * (120));
                calendar.add(Calendar.MINUTE, amount);
                ForumPostLike postLike = ForumPostLike.builder()
                        .addTime(calendar.getTime())
                        .headImg(virtualUser.getHeadImg())
                        .id(sequenceCreater.nextLongId())
                        .isShow((byte) 0)
                        .isVest((byte) 1)
                        .postId(forumPostVo.getId())
                        .type((byte) 1)
                        .userId(virtualUser.getId())
                        .userNickname(virtualUser.getNickname())
                        .build();
                list.add(postLike);
            }
            //达到规定数量跳出循环
            if (list.size() >= num) {
                break;
            }
        }
        //批量插入虚拟用户点赞信息
        MybatisBatchBuilder.create(ForumPostLikeMapper.class, list).run(ForumPostLikeMapper::insertSelective);
    }
}
