package com.bxm.localnews.news.action.impl;

import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.BizLogIntegrationService;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.MissionIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.action.PostShareService;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.ForumPostShareMapper;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.NewsShareEnum;
import com.bxm.localnews.news.event.ShareForumActionEvent;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.model.vo.ForumPostShare;
import com.bxm.localnews.news.model.vo.ForumPostStatistic;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.util.FormPostContentUtil;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.eventbus.EventBus;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;

import static com.bxm.localnews.news.constant.RedisConfig.USER_SHARE_NEWS_RECORD;

/**
 * @author liujia
 * @date 1/15/21 9:59 AM
 **/
@Slf4j
@Service
@AllArgsConstructor
public class PostShareServiceImpl implements PostShareService {

    private BizLogIntegrationService bizLogIntegrationService;

    private MissionIntegrationService missionIntegrationService;

    private UserIntegrationService userIntegrationService;

    private SequenceCreater sequenceCreater;

    private ForumPostShareMapper forumPostShareMapper;

    private ForumPostMapper forumPostMapper;

    private EventBus userActionEventBus;

    private RedisStringAdapter redisStringAdapter;

    private LocationIntegrationService locationIntegrationService;

    @Override
    public NewsCompleTaskDTO doShareForumPost(Long userId, Long postId, String areaCode, Byte type, Integer platform) {
        log.debug("用户分享帖子，参数->postId:{},userId:{},type:{},platform:{}", postId, userId, type, platform);

        bizLogIntegrationService.shareForumSuccessed(userId, postId, areaCode, platform);
        return saveShareForums(postId, userId, type, areaCode);
    }

    /**
     * 帖子分享成功金币赠送
     */
    private NewsCompleTaskDTO saveShareForums(Long postId, Long userId, Byte type, String areaCode) {
        UserBean user = this.userIntegrationService.selectUserFromCache(userId);

        //用户未正式注册前不能分享新闻获得金币
        if (user == null || user.getState() != 1) {
            return new NewsCompleTaskDTO();
        }

        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        if (NewsShareEnum.SIGN_SHARE.getCode() == type) {
            newsCompleTaskDTO = missionIntegrationService.signCompleTaskDTO(userId, "TASK_SIGN_SHARE", postId.toString(), "");
        } else if (NewsShareEnum.SUPPLEMENT_SHARE.getCode() == type) {
            //补签分享不发送任务
        } else {
            newsCompleTaskDTO = completePostShareTask(userId, postId, areaCode);
        }

        //分享成功获取金币记录
        ForumPostShare forumPostShare = new ForumPostShare();
        forumPostShare.setAddTime(new Date());
        forumPostShare.setId(sequenceCreater.nextLongId());
        forumPostShare.setPostId(postId);
        forumPostShare.setReward(Math.toIntExact(newsCompleTaskDTO.getGoldNum()));
        forumPostShare.setUserId(userId);
        forumPostShare.setType(type);
        forumPostShareMapper.insertSelective(forumPostShare);

        //更新分享数量
        ForumPostStatistic forumPostStatistic = ForumPostStatistic.buildShares(postId, 1);
        this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);

        userActionEventBus.post(ShareForumActionEvent.of().setInviteTypeEnum(InviteTypeEnum.FORUM).setTargetId(postId)
                .setUserId(userId));

        return newsCompleTaskDTO;
    }

    private NewsCompleTaskDTO completePostShareTask(Long userId, Long postId, String areaCode) {
        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        KeyGenerator shareRecordKey = getShareRecordKey(userId, postId);
        if (!redisStringAdapter.hasKey(shareRecordKey)) {
            //发放分享文章奖励
            ForumPostVo forumPostVo = forumPostMapper.selectWithoutContent(postId);
            LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
            FormPostContentUtil.replace(forumPostVo, location);
            String content = "转发【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】成功";
            newsCompleTaskDTO = missionIntegrationService.compleTask(userId, "TASK_NEWS_SHARE", postId.toString(), content);
            //记录是否分享过该新闻
            redisStringAdapter.set(shareRecordKey, "exist", DateUtils.getCurSeconds());
        }
        return newsCompleTaskDTO;
    }

    private KeyGenerator getShareRecordKey(Long userId, Long newsId) {
        return USER_SHARE_NEWS_RECORD.copy().appendKey(userId.toString())
                .appendKey(DateUtils.formatDate(new Date()))
                .appendKey(newsId.toString());
    }
}
