package com.bxm.localnews.news.activity.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.activity.ForumTeachService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.model.dto.UserTeach;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.Optional;

import static com.alibaba.fastjson.JSON.toJSONString;
import static com.bxm.localnews.news.constant.RedisConfig.FORUM_TEACH_CACHE;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

/**
 * @author gonzo
 * @date 2020-09-28 18:54
 **/
@Slf4j
@Service
@AllArgsConstructor
public class ForumTeachServiceImpl implements ForumTeachService {

    private final ForumProperties forumProperties;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserIntegrationService userIntegrationService;

    @Override
    public Optional<Long> getTeachForumPostId(Long userId,
                                              BasicParam basicParam) {
        if (log.isDebugEnabled()) {
            log.debug("用户: {} 获取教育信息, basicParam: {}",
                    userId,
                    toJSONString(basicParam));
        }

        if (isNull(userId)) {
            return Optional.empty();
        }

        if (CollectionUtils.isEmpty(forumProperties.getTeachIds())) {
            if (log.isDebugEnabled()) {
                log.debug("暂未配置教育id，跳过教育");
            }
            return Optional.empty();
        }

        // 查询用户是否已经被教育过
        UserTeach teach = redisHashMapAdapter.get(FORUM_TEACH_CACHE, Objects.toString(userId), UserTeach.class);

        if (log.isDebugEnabled()) {
            log.debug("用户: {} 的教育信息: {}", userId, JSON.toJSONString(teach));
        }

        if (Objects.nonNull(teach) && (Objects.equals(teach.getFinished(), Boolean.TRUE))) {
            // 已结束
            return Optional.empty();
        }

        if (isNull(teach) && !checkUserIs370VersionUpUser(userId)) {
            // 教育信息为空 且是老版本用户 则结束教育
            return Optional.empty();
        }

        // 新版本注册用户 且未教育完成
        if (isNull(teach) || isNull(teach.getForumIndex())) {
            if (log.isDebugEnabled()) {
                log.debug("用户: {}第一次被教育，获取到的教育id: {}", userId, forumProperties.getTeachIds().get(0));
            }

            // 第一次教育获取第0个id
            return Optional.of(forumProperties.getTeachIds().get(0));
        } else {
            // 上一次的角标 + 1
            int index = isNull(teach.getForumIndex()) ? 0 : teach.getForumIndex() + 1;

            // 避免中途修改过配置 导致角标溢出 需要判断角标 小于总条数
            if (index < forumProperties.getTeachIds().size()) {
                // 未教育完成 获取本次需要教育的帖子

                if (log.isDebugEnabled()) {
                    log.debug("用户: {} 第: {}次 获取到的教育id: {}", userId, index + 1, forumProperties.getTeachIds().get(index));
                }

                return Optional.of(forumProperties.getTeachIds().get(index));
            }
        }

        return Optional.empty();
    }

    @Async
    @Override
    public void updateTeachInfo(Long userId, Long newsId) {
        if (log.isDebugEnabled()) {
            log.debug("更新用户: {} 的教育信息，帖子: {}", userId, newsId);
        }

        if (isNull(userId) || isNull(newsId)) {
            return;
        }

        if (CollectionUtils.isEmpty(forumProperties.getTeachIds())
                || !forumProperties.getTeachIds().contains(newsId)) {
            // 教育功能不存在 或者帖子不是教育帖子
            return;
        }

        // 属于教育帖子

        // 查询用户是否已经被教育过
        UserTeach teach = redisHashMapAdapter.get(FORUM_TEACH_CACHE, Objects.toString(userId), UserTeach.class);

        if (log.isDebugEnabled()) {
            log.debug("用户: {} 的教育信息: {}", userId, JSON.toJSONString(teach));
        }

        if (Objects.nonNull(teach) && (Objects.equals(teach.getFinished(), Boolean.TRUE))) {
            // 已结束
            return;
        }

        if (isNull(teach)) {
            // 没有用户的教育信息
            // 双重判断一下，其实在列表查询的时候就已经过滤过了

            if (!checkUserIs370VersionUpUser(userId)) {
                // 不是新版本用户
                return;
            }

            // 教育缓存为空，且是新版本用户 则初始化数据 角标为0
            teach = new UserTeach();
            teach.setForumIndex(0);
            teach.setFinished(false);
            teach.setFinishedType((byte) 0);
        } else {
            // 有历史教育数据

            // 上一次的角标 + 1
            int index = isNull(teach.getForumIndex()) ? 0 : teach.getForumIndex() + 1;

            // 避免中途修改过配置 导致角标溢出
            if (index >= forumProperties.getTeachIds().size()) {
                teach.setFinished(true);
                teach.setFinishedType((byte) 1);
            } else {
                // 更新角标
                teach.setForumIndex(index);

                // 判断是否教育结束
                if (index == forumProperties.getTeachIds().size() - 1) {

                    if (log.isDebugEnabled()) {
                        log.debug("用户: {} 已完成教育", userId);
                    }

                    teach.setFinished(true);
                    teach.setFinishedType((byte) 1);
                }
            }
        }

        // 更新教育缓存
        redisHashMapAdapter.put(FORUM_TEACH_CACHE, Objects.toString(userId), teach);
    }

    /**
     * 校验用户是否是3.7.0版本以上的用户 如果不是则初始化教育信息
     *
     * @param userId 用户id
     * @return true 是新版本用户
     */
    private boolean checkUserIs370VersionUpUser(Long userId) {
        // 没有用户的教育信息 查询用户的注册信息 是否是新版本用户
        UserBean userBean = userIntegrationService.selectUserFromCache(userId);

        if (log.isDebugEnabled()) {
            log.debug("用户信息: {}", JSON.toJSONString(userBean));
        }

        boolean is370VersionUpUser = nonNull(userBean) && nonNull(userBean.getJudgeMarker())
                && BitOperatorUtil.getBitAsBoolean(userBean.getJudgeMarker(), LocalNewsUserJudgeMarkerEnum.VER_3_7_0.getIndex());

        if (!is370VersionUpUser) {

            if (log.isDebugEnabled()) {
                log.debug("用户: {} 属于老版本用户设置教育结束缓存", userId);
            }

            // 老版本用户 置为已结束
            UserTeach teach = new UserTeach();
            teach.setForumIndex(0);
            teach.setFinished(true);
            teach.setFinishedType((byte) 2);
            // 更新教育缓存
            redisHashMapAdapter.put(FORUM_TEACH_CACHE, Objects.toString(userId), teach);
        }

        return is370VersionUpUser;
    }

}
