package com.bxm.localnews.news.activity.impl;

import com.bxm.localnews.integration.PushMsgIntegService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.news.activity.PostAwardService;
import com.bxm.localnews.news.config.BizConfigProperties;
import com.bxm.localnews.news.domain.PostAwardMapper;
import com.bxm.localnews.news.model.param.CashAccountParam;
import com.bxm.localnews.news.model.param.EarningsSaveOrUpdateParam;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.PostAwardBean;
import com.bxm.localnews.news.util.SecurityUtils;
import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.UserEarningsTypeEnum;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;

/**
 * @author liujia
 * @date 2020-08-08 10:32
 **/
@Service
@Slf4j
public class PostAwardServiceImpl implements PostAwardService {

    private final PostAwardMapper postAwardMapper;

    private final PushMsgIntegService pushMsgIntegService;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final SequenceCreater sequenceCreater;

    private final BizConfigProperties bizConfigProperties;

    @Autowired
    public PostAwardServiceImpl(PostAwardMapper postAwardMapper,
                                PushMsgIntegService pushMsgIntegService,
                                UserAccountIntegrationService userAccountIntegrationService,
                                SequenceCreater sequenceCreater,
                                BizConfigProperties bizConfigProperties) {
        this.postAwardMapper = postAwardMapper;
        this.pushMsgIntegService = pushMsgIntegService;
        this.userAccountIntegrationService = userAccountIntegrationService;
        this.sequenceCreater = sequenceCreater;
        this.bizConfigProperties = bizConfigProperties;
    }

    @Override
    public void execGrantAward(AdminForumPost post, String postTitle) {
        if (null == post) {
            return;
        }

        if (Objects.equals(post.getIsCash(), (byte) 0) || Objects.isNull(post.getCashReward())) {
            log.info("帖子不存在，或者不是现金帖，或者无现金奖励金额");
            return;
        }

        if (postAwardMapper.existsPostId(post.getId()) > 0) {
            log.info("帖子[{}]已经发放过奖励，不重复发放", post.getId());
            return;
        }

        // 记录发放历史，防止重复发放
        saveAwardRecord(post);

        EarningsSaveOrUpdateParam userEarningsParam = new EarningsSaveOrUpdateParam();
        userEarningsParam.setEarningsType(UserEarningsTypeEnum.POST_REWARD.getEarningsType());
        userEarningsParam.setEarningsTitle("爆料有奖奖励金");
        userEarningsParam.setOrderTime(new Date());
        userEarningsParam.setImg(bizConfigProperties.getPlatFromImgUrl());

        // 调用用户服务，给用户增加可用余额
        CashAccountParam param = new CashAccountParam();
        param.setExtData(userEarningsParam);
        param.setAccountAction(AccountActionEnum.POST_REWARD);
        param.setCashFlowType(CashFlowTypeEnum.OTHER);
        param.setAmount(post.getCashReward());
        param.setUserId(post.getUserId());
        param.setRelationId(post.getId());
        param.setRemark("您的帖子《" + postTitle + "》被奖励" + post.getCashReward().stripTrailingZeros().toPlainString() + "元");

        if (userAccountIntegrationService.cashAccountOperation(param)) {
            // 发送用户推送
            pushMsgIntegService.pushAwardMsg(post, postTitle);
        }

    }

    private void saveAwardRecord(AdminForumPost post) {
        PostAwardBean entity = new PostAwardBean();
        entity.setId(sequenceCreater.nextLongId());
        entity.setAwardType(1);
        entity.setAwardAmount(post.getCashReward());
        entity.setUserId(post.getUserId());
        entity.setOperator(SecurityUtils.getAdminUserId());
        entity.setPostId(post.getId());
        entity.setCreateTime(new Date());
        postAwardMapper.insert(entity);
    }
}
