package com.bxm.localnews.news.approve.impl;

import com.bxm.localnews.facade.UserActivityService;
import com.bxm.localnews.news.action.PostClickService;
import com.bxm.localnews.news.action.PostLikeService;
import com.bxm.localnews.news.approve.PostApproveService;
import com.bxm.localnews.news.approve.context.PostApproveContext;
import com.bxm.localnews.news.constant.LogicGroupConstant;
import com.bxm.localnews.news.domain.AdminForumPostMapper;
import com.bxm.localnews.news.domain.ForumPostApproveHistoryMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.AdminForumPostApproveHistory;
import com.bxm.localnews.param.InviteUserPostCashAwardParam;
import com.bxm.newidea.component.filter.FilterChainExecutor;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 1/18/21 2:37 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class PostApproveServiceImpl implements PostApproveService {

    private final UserActivityService userActivityService;

    private final AdminForumPostMapper adminForumPostMapper;

    private final SequenceCreater sequenceCreater;

    private final PostLikeService postLikeService;

    private final ForumPostApproveHistoryMapper forumPostApproveHistoryMapper;

    private final FilterChainExecutor filterChainExecutor;

    private final PostClickService postClickService;

    private List<Long> parseIds(String ids) {
        if (StringUtils.isBlank(ids)) {
            return Lists.newArrayList();
        }

        return Arrays.stream(ids.split(",")).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
    }

    @Override
    public void batchApprove(String ids, Integer status, String comment, Long operator) {
        List<Long> postIdList = parseIds(ids);
        if (CollectionUtils.isEmpty(postIdList)) {
            return;
        }

        PostStatusEnum statusEnum = PostStatusEnum.getByCode(status);
        if (null == statusEnum) {
            return;
        }

        for (Long postId : postIdList) {
            approve(postId, statusEnum, comment, operator);
        }
    }

    @Override
    public void approve(Long postId, PostStatusEnum status, String comment, Long operator) {
        log.info("运营审核帖子，postID:{},status:{},comment:{},operator:{}",
                postId, status, comment, operator);

        AdminForumPost adminForumPost = this.adminForumPostMapper.selectByPrimaryKey(postId);
        if (null == adminForumPost) {
            return;
        }

        AdminForumPost updatePost = new AdminForumPost();
        updatePost.setId(postId);

        PostApproveContext context = new PostApproveContext();
        context.setComment(comment);
        context.setOperator(operator);
        context.setModifyStatus(status);
        context.setPostInfo(adminForumPost);
        context.setUpdatePost(updatePost);

        filterChainExecutor.doFilter(LogicGroupConstant.ADMIN_POST_APPROVE_FILTER, context);

        if (this.adminForumPostMapper.updateByIdPartSelective(updatePost) > 0) {
            // 记录审核历史
            AdminForumPostApproveHistory approveHistory = new AdminForumPostApproveHistory();
            approveHistory.setId(sequenceCreater.nextLongId());
            approveHistory.setPostId(postId);
            approveHistory.setStatus(status.getCode());
            approveHistory.setComment(comment);
            approveHistory.setCreator(operator);
            approveHistory.setCreateTime(new Date());

            forumPostApproveHistoryMapper.addApproveHistory(approveHistory);

            // 审核后置处理
            afterApprove(adminForumPost);
        }
    }

    /**
     * 审核通过的后续处理
     *
     * @param post 帖子信息
     */
    private void afterApprove(AdminForumPost post) {
        // 非运营发帖，进行通知处理
        if (post.getCreator() == null) {
            userActivityService.postPass(InviteUserPostCashAwardParam.builder()
                    .postId(post.getId())
                    .userId(post.getUserId())
                    .build());
        }

        //为帖子增加模拟数据（点赞数）
        postLikeService.simulateData(post);
        // 删除站外阅读的缓存限制
        postClickService.removePostReadLimit(post.getId());
    }

    @Override
    public List<AdminForumPostApproveHistory> getApproveHistoryList(Long id, Integer action) {
        return forumPostApproveHistoryMapper.getApproveHistoryList(id, action);
    }
}
