package com.bxm.localnews.news.create.filter;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.constant.OrderConstant;
import com.bxm.localnews.news.create.context.AdminPostContext;
import com.bxm.localnews.news.domain.AdminNewsReplyMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.AdminNewsReply;
import com.bxm.localnews.news.util.DealContentUtil;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.newidea.component.annotations.FilterBean;
import com.bxm.newidea.component.filter.IFilter;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;

import java.util.Date;
import java.util.List;

import static com.bxm.localnews.news.constant.LogicGroupConstant.ADMIN_POST_CREATE_FILTER;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 填充运营发帖的基础信息
 *
 * @author liujia
 * @date 1/18/21 3:26 PM
 **/
@FilterBean(group = ADMIN_POST_CREATE_FILTER)
@AllArgsConstructor
public class AdminPostFillAttributeFilter implements IFilter<AdminPostContext> {

    private UserIntegrationService userIntegrationService;

    private DealContentUtil dealContentUtil;

    private AdminNewsReplyMapper adminNewsReplyMapper;

    @Override
    public void doFilter(AdminPostContext context) {
        AdminForumPost savePost = context.getSavePost();
        AdminForumPost beforePost = context.getBeforePost();

        if (null == savePost) {
            savePost = new AdminForumPost();
            BeanUtils.copyProperties(context.getRequestPost(), savePost);
            context.setSavePost(savePost);
        }

        //字段填充
        UserBean userInfo = userIntegrationService.selectUserFromCache(savePost.getUserId());
        if (null != userInfo) {
            savePost.setUserImg(userInfo.getHeadImg());
            savePost.setUserName(userInfo.getNickname());
        }

        // 对内容的Html进行处理，提取图片信息和视频信息
        dealContentUtil.analyzeContent(savePost);

        savePost.setAreaCode(isNotBlank(savePost.getAreaCode())
                ? savePost.getAreaCode()
                : null);
        savePost.setCoverList(isNotBlank(savePost.getCoverList()) ?
                JSON.toJSONString(savePost.getCoverList().split(","))
                : null);
        savePost.setEditorMessage(isNotBlank(savePost.getEditorMessage())
                ? savePost.getEditorMessage()
                : null);
        savePost.setModifyTime(new Date());

        if (context.isUpdatePost()) {
            if (isNull(savePost.getStatus())) {
                savePost.setStatus(beforePost.getStatus());
            }
        } else {
            // 仅新增时需要设置的参数
            savePost.setCreateTime(new Date());
            savePost.setCreator(context.getOperator());
            // 运营后台发布的帖子默认为审核通过
            if (isNull(savePost.getStatus())) {
                savePost.setStatus(PostStatusEnum.NORMAL.getCode());
            }

            //防止运营在帖子未生成时添加评论,并且评论已被扫描
            savePost.setCommentCount(adminNewsReplyMapper.count(savePost.getId()));
            List<AdminNewsReply> newsReplies = adminNewsReplyMapper.selectPostReplyByNewsIdOrderByAddTime(savePost.getId());
            if (CollectionUtils.isNotEmpty(newsReplies)) {
                savePost.setReplyInfo(JSONObject.toJSONString(newsReplies));
            }
        }
    }

    @Override
    public int getOrder() {
        return OrderConstant.getOrder(this.getClass());
    }
}
