package com.bxm.localnews.news.create.filter;

import com.bxm.localnews.news.constant.OrderConstant;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.create.context.AdminPostContext;
import com.bxm.localnews.news.hotpost.ShareCashPostService;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashInfoEntity;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.newidea.component.annotations.FilterBean;
import com.bxm.newidea.component.filter.IFilter;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.RandomUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.news.constant.LogicGroupConstant.ADMIN_POST_CREATE_FILTER;

/**
 * 运营发布热文（现金奖励贴），额外记录相关信息
 *
 * @author liujia
 * @date 1/18/21 9:08 PM
 **/
@FilterBean(group = ADMIN_POST_CREATE_FILTER)
@AllArgsConstructor
@Slf4j
public class AdminPostShareCashFilter implements IFilter<AdminPostContext> {

    private ShareCashPostService shareCashPostService;

    private RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public void doFilter(AdminPostContext context) {
        AdminForumPost savePost = context.getSavePost();

        ForumPostShareCashInfoEntity existsShareCashInfo = shareCashPostService.getDetail(savePost.getId());
        ForumPostShareCashInfoEntity entity = new ForumPostShareCashInfoEntity();

        // 未开启热文
        if (null == savePost.getIsShareCash() || Objects.equals(savePost.getIsShareCash(), 0)) {
            // 帖子不是热文，将历史热文数据设置为下架
            if (existsShareCashInfo != null) {
                shareCashPostService.changeStatus(savePost.getId(), 0);
            }
            return;
        }

        //如果是热文帖子,且缓存中没有该帖子的随机已发金额
        boolean existsRandomAward = redisHashMapAdapter.exists(buildHotPostRandomAwardKey(), String.valueOf(savePost.getId()));
        if (Objects.equals(savePost.getIsShareCash(), 1) && !existsRandomAward) {
            redisHashMapAdapter.put(buildHotPostRandomAwardKey(),
                    String.valueOf(savePost.getId()),
                    RandomUtils.randomDouble(2, 8, 20));
        }

        entity.setAmount(savePost.getAmount());
        if (null == savePost.getAwardStartTime()) {
            entity.setAwardStartTime(savePost.getPublishTime());
        } else {
            entity.setAwardStartTime(savePost.getAwardStartTime());
        }
        entity.setAwardEndTime(savePost.getAwardEndTime());


        if (null == existsShareCashInfo) {
            entity.setPostId(savePost.getId());
            entity.setCreator(context.getOperator());
        } else {
            entity.setId(existsShareCashInfo.getId());
            entity.setPostId(existsShareCashInfo.getPostId());
            // 是否热文， 同时也是热文的状态
            entity.setStatus(savePost.getIsShareCash());


            // 如果设置的奖励金额小于已发放金额，则直接设置为下架
            if (existsShareCashInfo.getUsedAmount().compareTo(entity.getAmount()) >= 0) {
                entity.setStatus(0);

                log.warn("帖子[{}]的奖励金额[{}]比已发放金额[{}]少，设置为下线",
                        savePost.getId(),
                        entity.getAmount(),
                        existsShareCashInfo.getUsedAmount());
            }
        }

        log.info("热文状态为：{}", entity.getStatus());
        // 如果活动状态设置为下架，则热文也自动为过期状态
        if (Objects.equals(savePost.getActivityStatus(), 0)) {
            log.info("活动下架设置热文状态为过期:{}", entity.getStatus());
            entity.setStatus(0);
        }
        //如果修改的过期时间早于现在，则直接设置为过期状态
        if (Objects.nonNull(savePost.getAwardEndTime()) && DateUtils.before(savePost.getAwardEndTime(), new Date())) {
            entity.setStatus(0);
        }
        //如果设置的开始时间早于现在，则先设置过期
        if (DateUtils.after(savePost.getAwardStartTime(), new Date())) {
            entity.setStatus(0);
        }

        shareCashPostService.save(entity);
    }

    /**
     * 构建热文帖子随机初始已发奖励金额
     *
     * @return redis key
     */
    private KeyGenerator buildHotPostRandomAwardKey() {
        return RedisConfig.HOT_POST_INIT_RANDOM_AWARD_KEY.copy();
    }

    @Override
    public int getOrder() {
        return OrderConstant.getOrder(this.getClass());
    }
}
