package com.bxm.localnews.news.create.filter;

import com.bxm.localnews.news.constant.OrderConstant;
import com.bxm.localnews.news.create.context.UserPostContext;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.model.vo.ForumBasicVo;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.TopicVo;
import com.bxm.localnews.news.util.AreaCodeUtils;
import com.bxm.localnews.news.util.UploadUtils;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.newidea.component.annotations.FilterBean;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.filter.IFilter;
import com.bxm.newidea.component.tools.InputFilter;
import com.google.common.base.Joiner;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.LogicGroupConstant.USER_POST_CREATE_FILTER;

/**
 * 用户发帖内容的前置处理，对内容进行统一调整
 * 用户发帖的内容为文本部分和图片、视频部分，这里将其组装在一起作为content，并且进行一部分前置解析
 *
 * @author liujia
 * @date 1/15/21 2:30 PM
 **/
@FilterBean(group = USER_POST_CREATE_FILTER)
@Slf4j
@AllArgsConstructor
public class UserPostContentAssembleFilter implements IFilter<UserPostContext> {

    private ForumTopicMapper forumTopicMapper;

    private UploadUtils uploadUtils;

    @Override
    public void doFilter(UserPostContext context) {
        ForumPostVo savePost = context.getSavePost();
        ForumBasicVo requestPost = context.getRequestPost();

        if (null == savePost) {
            savePost = new ForumPostVo();
            BeanUtils.copyProperties(requestPost, savePost);
            context.setSavePost(savePost);
        }

        if (StringUtils.isBlank(savePost.getLocation())) {
            savePost.setLocation(null);
        }

        assembleContent(savePost);

        //新人一键发布的帖子无需审核
        if (Objects.nonNull(requestPost.getIsNewReport()) && requestPost.getIsNewReport() == 1) {
            savePost.setStatus(PostStatusEnum.BLOCKED.getCode());
        } else {
            savePost.setStatus(PostStatusEnum.APPROVING.getCode());
        }
        savePost.setIsFirstUserPost((byte) (context.isFirstUserPost() ? 1 : 0));

        // 如果是直接通过的帖子，并且不是新人贴，则设置为审核通过
        if (context.isDirectlyPassed() && !Objects.equals(requestPost.getIsNewReport(), (byte) 1)) {
            savePost.setStatus(PostStatusEnum.NORMAL.getCode());
        }

        // 将话题组装为字符串，方便后续话题查询组装
        if (org.springframework.util.CollectionUtils.isEmpty(savePost.getTopicIdList())) {
            savePost.setTopicIds(StringUtils.EMPTY);
        } else {
            savePost.setTopicIds(Joiner.on(",").join(savePost.getTopicIdList()));
        }

        // 设置投放地区,如果请求参数中不包含地区信息，则使用用户的定位信息，防止发放到全国
        if (StringUtils.isBlank(savePost.getAreaCode())) {
            log.info("发帖信息[{}]信息为空，使用用户的当前定位信息", savePost.getId());

            if (null != context.getPostUserInfo()) {
                savePost.setAreaCode(context.getPostUserInfo().getLocationCode());
            }
        }
        savePost.setDeliveryType(0);
        if (StringUtils.isNotBlank(savePost.getAreaCode())) {
            savePost.setDeliveryType(1);
            savePost.setAreaCode(AreaCodeUtils.completeAreaCode(savePost.getAreaCode()));
        }
    }

    /**
     * 组装帖子最终显示的富文本内容
     * 替换内容中出现的特殊字符、不支持的emoji表情
     * 将图片和视频与内容组装在一起
     *
     * @param savePost 帖子信息
     */
    private void assembleContent(ForumPostVo savePost) {
        // emoji表情编码替换
        savePost.setTextField(EmojiCodeParser.replaceSoftbankEmoji(savePost.getTextField()));
        // 过滤内容中的特殊字符
        savePost.setTextField(InputFilter.builder(savePost.getTextField()).filte());
        // 填充话题信息
        if (CollectionUtils.isNotEmpty(savePost.getTopicIdList())) {
            TopicVo topic = forumTopicMapper.selectTopicById(savePost.getTopicIdList().get(0), null);
            if (topic != null) {
                savePost.setForumContent(topic.getContent());
            }
        }

        // 如果内容的最最后一个字符是换行符，进行替换
        int index = savePost.getTextField().lastIndexOf("\n");
        if (index != -1 && index == savePost.getTextField().length() - 1) {
            savePost.setTextField(savePost.getTextField().substring(0, savePost.getTextField().length() - 1));
        }

        // 组装content textField可能是富文本
        StringBuilder content = new StringBuilder("<p>");
        if (StringUtils.isNotBlank(savePost.getTextField())) {
            content.append(savePost.getTextField().replaceAll("\\n", "</p><p>"));
        }
        content.append("</p>");
        if (StringUtils.isNotBlank(savePost.getExtraContent())) {
            content.append("<p>");
            content.append(savePost.getExtraContent());
            content.append("</p>");
        }
        if (CollectionUtils.isNotEmpty(savePost.getPostImgList())) {
            dealPostImgList(savePost, content);
        }
        savePost.setContent(content.toString());
    }

    /**
     * 处理图片和视频，将图片和视频组装为富文本形式
     *
     * @param forumPostVo 帖子信息
     * @param content     帖子内容
     */
    private void dealPostImgList(ForumPostVo forumPostVo, StringBuilder content) {
        forumPostVo.getPostImgList().forEach(e -> {
            if (StringUtils.isBlank(e.getType())) {
                e.setType("IMG");
            }
        });
        //保证视频在图片的前面
        List<PostImgVo> postVideoList = forumPostVo.getPostImgList().stream().filter(e -> "VIDEO".equals(e.getType())).collect(Collectors.toList());
        List<PostImgVo> postImgList = forumPostVo.getPostImgList().stream().filter(e -> "IMG".equals(e.getType())).collect(Collectors.toList());

        for (PostImgVo postImgVo : postVideoList) {
            if (StringUtils.isNotBlank(postImgVo.getVideoUrl())) {
                content.append("<p>&nbsp;<video controls=\"controls\" src=\"");
                content.append(postImgVo.getVideoUrl());
                content.append("\" poster=\"");
                content.append(postImgVo.getImgUrl());
                content.append("\" size=\"");
                content.append(postImgVo.getSize() == null ? 0 : postImgVo.getSize());
                content.append("\" duration=\"");
                content.append(postImgVo.getDuration() == null ? 0 : postImgVo.getDuration());
                content.append("\"></video>&nbsp;</p>");
            }
        }

        for (PostImgVo postImgVo : postImgList) {
            String imgUrl = postImgVo.getImgUrl();
            //这里主要是为了客户端图片没有上传oss而是传了第三方图片，会过期，所以要在这里判断[TMD]
            if (!StringUtils.contains(imgUrl, "wstong.com")) {
                imgUrl = uploadUtils.uploadImgToOssBySrc(imgUrl);
            }
            if (StringUtils.isNotBlank(postImgVo.getImgUrl())) {
                content.append("<p><img src=\"");
                content.append(imgUrl);
                content.append("\"></img></p>");
            }
        }

        postVideoList.addAll(postImgList);
        forumPostVo.setPostImgList(postVideoList);
    }

    @Override
    public int getOrder() {
        return OrderConstant.getOrder(this.getClass());
    }
}
