package com.bxm.localnews.news.create.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.integration.NewsSeqComponent;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.constant.LogicGroupConstant;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.convert.ForumPostCreateConvert;
import com.bxm.localnews.news.create.PostCreateService;
import com.bxm.localnews.news.create.context.AdminPostContext;
import com.bxm.localnews.news.create.context.UserNoteContext;
import com.bxm.localnews.news.create.context.UserPostContext;
import com.bxm.localnews.news.domain.AdminForumPostMapper;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.event.AdminPostCreateEvent;
import com.bxm.localnews.news.event.UserNoteCreateEvent;
import com.bxm.localnews.news.event.UserPostCreateEvent;
import com.bxm.localnews.news.model.dto.ForumPostCreateDTO;
import com.bxm.localnews.news.model.param.NoteParam;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.ForumBasicVo;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.post.ForumPostService;
import com.bxm.newidea.component.filter.FilterChainExecutor;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.rule.RuleGroupExecutor;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;

/**
 * @author liujia
 * @date 1/15/21 2:16 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class PostCreateServiceImpl implements PostCreateService {

    private final FilterChainExecutor filterChainExecutor;

    private final RuleGroupExecutor ruleGroupExecutor;

    private final SequenceCreater sequenceCreater;

    private final DistributedLock distributedLock;

    private final ForumPostMapper forumPostMapper;

    private final NewsSeqComponent newsSeqComponent;

    private final UserIntegrationService userIntegrationService;

    private final ForumPostCreateConvert forumPostCreateConvert;

    private final AdminForumPostMapper adminForumPostMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final ForumPostService forumPostService;

    @Override
    public Message saveUserPost(ForumBasicVo forumBasicVo, BasicParam basicParam, boolean isDirectlyPassed) {
        log.info("用户发帖 forumBasicVo:[{}]，basicParam：{},isDirectlyPassed:{}",
                toJSONString(forumBasicVo),
                toJSONString(basicParam),
                isDirectlyPassed);

        if (Objects.isNull(forumBasicVo.getAreaCode())) {
            return Message.build(false, "用户所属区域不能为空");
        }

        // 避免频繁发帖或者重复点击
        KeyGenerator lockKey = RedisConfig.NEW_REPORT_LOCK.copy().appendKey(forumBasicVo.getUserId());
        String stringId = sequenceCreater.nextStringId();
        if (!distributedLock.lock(lockKey.gen(), stringId)) {
            return Message.build(false, "请勿频繁发帖");
        }

        UserPostContext context = new UserPostContext();
        context.setBasicParam(basicParam);
        context.setRequestPost(forumBasicVo);
        context.setDirectlyPassed(isDirectlyPassed);
        context.setSaveOrUpdate(null == forumBasicVo.getId());
        context.setPostUserInfo(userIntegrationService.selectUserFromCache(forumBasicVo.getUserId()));
        if (context.isUpdatePost()) {
            context.setBeforePost(forumPostMapper.selectWithoutContent(forumBasicVo.getId()));
        } else {
            forumBasicVo.setId(newsSeqComponent.getPostId());
        }

        // 判断是否可以发帖
        if (!ruleGroupExecutor.apply(LogicGroupConstant.USER_POST_CREATE_RULE, context)) {
            return Message.build(false, context.getErrorMsg());
        }

        // 处理帖子信息
        filterChainExecutor.doFilter(LogicGroupConstant.USER_POST_CREATE_FILTER, context);

        Message message = Message.build();
        if (context.isUpdatePost()) {
            forumPostMapper.updateByPrimaryKeySelective(context.getSavePost());
        } else {
            if (this.forumPostMapper.insertSelective(context.getSavePost()) > 0) {
                // 完成发帖的相关任务，返回信息给客户端，显示任务的完成情况
                ForumPostVo forumPostVo = context.getSavePost();
                NewsCompleTaskDTO newsCompleTaskDTO = forumPostService.completeTask(forumPostVo);

                ForumPostCreateDTO forumPostCreateDTO = forumPostCreateConvert.generateForumPost(forumPostVo,
                        newsCompleTaskDTO,
                        forumBasicVo.getUserId());

                message.addParam("completeTaskAndPush", forumPostCreateDTO);
            } else {
                return Message.build(false, "发帖失败，请稍后重试");
            }
        }

        // 帖子发布完成事件，异步处理后续逻辑
        SpringContextHolder.getApplicationContext().publishEvent(new UserPostCreateEvent(context));

        return message;
    }

    @Override
    public Message saveUserNote(NoteParam param) {
        log.info("用户发布小纸条：{}", param);

        UserNoteContext context = new UserNoteContext();
        context.setParam(param);
        if (null != param.getUserId()) {
            context.setPostUserInfo(userIntegrationService.selectUserFromCache(param.getUserId()));
        }

        if (!ruleGroupExecutor.apply(LogicGroupConstant.USER_NOTE_CREATE_RULE, context)) {
            return Message.build(false, context.getErrorMsg());
        }

        filterChainExecutor.doFilter(LogicGroupConstant.USER_NOTE_CREATE_FILTER, context);

        if (this.forumPostMapper.insertSelective(context.getSaveNote()) > 0) {
            // 发布事件，触发后续的相关逻辑
            SpringContextHolder.getApplicationContext().publishEvent(new UserNoteCreateEvent(context));
        }

        return Message.build().addParam("note", context.getResponseNote());
    }

    @Override
    public Message saveAdminPost(AdminForumPost adminForumPost, Long operator) {

        String stringId = sequenceCreater.nextStringId();
        if (!distributedLock.lock(buildSaveOrUpdatePostKey(adminForumPost.getId()), stringId)) {
            return Message.build(false, "请勿频繁发帖");
        }

        AdminForumPost existedAdminForumPost = this.adminForumPostMapper.selectByPrimaryKey(adminForumPost.getId());

        log.info("运营操作发帖，参数:[{}],原有数据:[{}]",
                JSONObject.toJSONString(adminForumPost),
                JSONObject.toJSONString(existedAdminForumPost));

        AdminPostContext context = new AdminPostContext();
        context.setBeforePost(existedAdminForumPost);
        context.setRequestPost(adminForumPost);
        context.setOperator(operator);
        context.setSaveOrUpdate(null == existedAdminForumPost);

        // 验证前置逻辑
        if (!ruleGroupExecutor.apply(LogicGroupConstant.ADMIN_POST_CREATE_RULE, context)) {
            return Message.build(false, context.getErrorMsg());
        }

        // 填充帖子信息，处理帖子内容等
        filterChainExecutor.doFilter(LogicGroupConstant.ADMIN_POST_CREATE_FILTER, context);

        // 数据最终入库
        AdminForumPost savePost = context.getSavePost();
        if (context.isUpdatePost()) {
            this.adminForumPostMapper.updateByIdPartSelective(savePost);
        } else {
            this.adminForumPostMapper.insertSelective(savePost);
        }

        // 发布事件，触发后续的相关逻辑
        SpringContextHolder.getApplicationContext().publishEvent(new AdminPostCreateEvent(context));

        //分布式锁解锁
        distributedLock.unlock(buildSaveOrUpdatePostKey(adminForumPost.getId()), stringId);
        return Message.build();
    }

    /**
     * 构建更新或创建
     *
     * @param postId 帖子id
     * @return 分布式锁key
     */
    private String buildSaveOrUpdatePostKey(Long postId) {
        return RedisConfig.SAVE_OR_UPDATE_POST_KEY.copy().appendKey(postId).gen();
    }
}
