package com.bxm.localnews.news.create.listener;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.integration.PushMsgIntegService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.news.action.PostClickService;
import com.bxm.localnews.news.action.PostLikeService;
import com.bxm.localnews.news.activity.PostAwardService;
import com.bxm.localnews.news.approve.PostApproveService;
import com.bxm.localnews.news.create.context.AdminPostContext;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.enums.PostTypeEnum;
import com.bxm.localnews.news.event.AdminPostCreateEvent;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.note.NoteService;
import com.bxm.localnews.news.post.ForumPostContentService;
import com.bxm.localnews.news.post.ForumPostService;
import com.bxm.localnews.news.post.PostTagService;
import com.bxm.localnews.news.recommend.RecommendPostService;
import com.bxm.localnews.news.statistics.ForumPostStatisticService;
import com.bxm.localnews.news.util.FormPostContentUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * 运营发帖成功后的相关处理逻辑
 *
 * @author liujia
 * @date 1/15/21 4:31 PM
 **/
@Component
@AllArgsConstructor
@Slf4j
public class AdminPostCreateListener {

    private final PostAwardService postAwardService;

    private final PostTagService postTagService;

    private final ForumPostContentService forumPostContentService;

    private final PostClickService postClickService;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final PushMsgIntegService pushMsgIntegService;

    private final DomainIntegrationService domainIntegrationService;

    private final PostApproveService postApproveService;

    private final RecommendPostService recommendPostService;

    private final PostLikeService postLikeService;

    private final ForumPostStatisticService forumPostStatisticService;

    private final NoteService noteService;

    private final ForumPostService forumPostService;

    /**
     * 保存帖子时，判断是否给与现金奖励,只有非运营发布的帖子才给奖励
     * 编辑和新增时均会触发
     *
     * @param event 运营发帖完成事件
     */
    @EventListener
    public void addPostAward(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();

        if (context.isUpdatePost()) {
            AdminForumPost post = context.getSavePost();

            postAwardService.execGrantAward(post, FormPostContentUtil.getForumPostTitle(post.getTitle(), post.getTextField()));
        }
    }

    /**
     * 如果帖子变更了状态，触发审核流程
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void triggerApprove(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost savePost = context.getSavePost();
        AdminForumPost beforePost = context.getBeforePost();

        //状态变更，且最终状态为1，5，6时走审批流程
        if (context.isUpdatePost()
                && !Objects.equals(savePost.getStatus(), beforePost.getStatus())
                && isPostShow(savePost.getStatus())) {
            log.info("运营管理员[{}]对此帖子进行保存:[{}]", context.getOperator(), JSONObject.toJSONString(savePost));

            postApproveService.approve(savePost.getId(),
                    PostStatusEnum.getByCode(savePost.getStatus()),
                    null,
                    context.getOperator());
        }
    }

    /**
     * 帖子是否通过审核（正常显示，仅楼主可见，本地圈屏蔽）
     *
     * @param status 状态
     * @return 是否通过
     */
    private boolean isPostShow(Integer status) {
        return Objects.equals(status, PostStatusEnum.NORMAL.getCode())
                || Objects.equals(status, PostStatusEnum.SELF.getCode())
                || Objects.equals(status, PostStatusEnum.BLOCKED.getCode());
    }

    /**
     * 重置帖子的标签
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void resetPostTag(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost post = context.getSavePost();
        postTagService.resetForumPostTag(post);
    }

    /**
     * 对帖子中的视频进行异步转码操作
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void transCodePostVideo(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost post = context.getSavePost();

        forumPostService.execTransCode(post.getId(), post.getImgList(), null);
    }

    /**
     * 给帖子增加模拟数据
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void addMockData(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost adminForumPost = context.getSavePost();

        // 重新计算点击次数
        postClickService.setClickCountInfo(context.getSavePost(), context.getBeforePost());

        if (!context.isUpdatePost()) {
            postLikeService.simulateData(adminForumPost);
        }
    }

    /**
     * 给帖子增加模拟数据
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void addStatisticsData(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost adminForumPost = context.getSavePost();

        // 新增的时候，增加面包块的统计数据
        if (!context.isUpdatePost()) {
            forumPostStatisticService.addCrumbsTotal(adminForumPost.getDeliveryType(),
                    adminForumPost.getAreaCode(),
                    adminForumPost.getPostType());
        }
    }

    /**
     * 帖子编辑时，变更推荐库相关的逻辑
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void updateRecommend(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost adminForumPost = context.getSavePost();

        if (context.isUpdatePost()) {
            recommendPostService.updateRecommendWhenChanged(adminForumPost);
        }
    }

    /**
     * 给用户发送相关的消息推送
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void sendPush(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost post = context.getSavePost();
        AdminForumPost existsPost = context.getBeforePost();

        // 判断内容中是否存在@用户
        forumPostContentService.afterCreate(post);

        if (context.isUpdatePost()) {
            if (Objects.equals(post.getStatus(), PostStatusEnum.NORMAL.getCode())) {
                //得到推送的文案
                String forumPostTitle = FormPostContentUtil.getForumPostTitle(post.getTitle(), post.getTextField());
                //增加用户小红花
                userAccountIntegrationService.addUserGold(post, existsPost, forumPostTitle);
                //精华帖、现金帖推送、爆料贴推送
                pushMsgIntegService.pushMsg(post, existsPost, forumPostTitle, domainIntegrationService.getInnerH5BaseUrl());
            }
        }
    }

    /**
     * 如果审核的用户小纸条缓存，则删除缓存
     *
     * @param event 运营发帖事件
     */
    @EventListener
    public void removeNoteCache(AdminPostCreateEvent event) {
        AdminPostContext context = event.getContext();
        AdminForumPost adminForumPost = context.getSavePost();

        if (Objects.equals(PostTypeEnum.NOTE.getCode(), adminForumPost.getPostType())) {
            noteService.removeCache(adminForumPost.getUserId());
        }
    }
}
