package com.bxm.localnews.news.create.listener;

import com.bxm.localnews.integration.BizLogIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.event.NoteActionEvent;
import com.bxm.localnews.news.event.UserActionEvent;
import com.bxm.localnews.news.event.UserNoteCreateEvent;
import com.bxm.localnews.news.model.param.NoteParam;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.note.NoteService;
import com.bxm.localnews.news.post.ForumPostService;
import com.bxm.localnews.news.statistics.ForumPostStatisticService;
import com.bxm.localnews.news.topic.ForumTopicService;
import com.google.common.eventbus.EventBus;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import java.util.Date;

/**
 * 用户发布小纸条后的相关监听事件
 *
 * @author liujia
 * @date 1/18/21 5:29 PM
 **/
@Component
@AllArgsConstructor
@Slf4j
public class UserNoteCreateListener {

    private final ForumTopicService forumTopicService;

    private final ForumPostStatisticService forumPostStatisticService;

    private final EventBus userActionEventBus;

    private final BizLogIntegrationService bizLogIntegrationService;

    private final NoteService noteService;

    private final UserIntegrationService userIntegrationService;

    private final ForumPostService forumPostService;

    /**
     * 设置小纸条的话题
     *
     * @param event 小纸条发布事件
     */
    @EventListener
    public void modifyTopic(UserNoteCreateEvent event) {
        NoteParam noteParam = event.getContext().getParam();
        ForumPostVo forumPostVo = event.getContext().getSaveNote();

        forumPostService.updateTopic(forumPostVo.getId(),
                forumPostVo.getTopicIdList(),
                forumPostVo.getSecondTopicId());

        // 清理用户发布小纸条缓存
        noteService.removeCache(noteParam.getUserId());
    }

    /**
     * 增加小纸条相关的统计数据
     *
     * @param event 小纸条发布事件
     */
    @EventListener
    public void addStatisticsData(UserNoteCreateEvent event) {
        NoteParam noteParam = event.getContext().getParam();
        ForumPostVo forumPostVo = event.getContext().getSaveNote();

        if (StringUtils.isBlank(noteParam.getAreaCode())) {
            forumPostStatisticService.addCrumbsTotal(0, noteParam.getAreaCode(), (byte) 2);
        } else {
            forumPostStatisticService.addCrumbsTotal(1, noteParam.getAreaCode(), (byte) 2);
        }

        // 增加用户的发布数量
        userIntegrationService.addPostReplyNum(forumPostVo.getUserId(), 1);
    }

    /**
     * 小纸条的发布事件
     *
     * @param event 小纸条发布事件
     */
    @EventListener
    public void sendEvent(UserNoteCreateEvent event) {
        NoteParam noteParam = event.getContext().getParam();
        ForumPostVo forumPostVo = event.getContext().getSaveNote();

        //上报发小纸条信息到用户推荐数据源服务(异步)
        UserActionEvent userActionEvent = NoteActionEvent.of().setLastNoteId(forumPostVo.getId())
                .setLastNoteTime(new Date())
                .setUserId(noteParam.getUserId());

        userActionEventBus.post(userActionEvent);
    }

    /**
     * 数据中心的相关埋点
     *
     * @param event 小纸条发布事件
     */
    @EventListener
    public void addLog(UserNoteCreateEvent event) {
        NoteParam noteParam = event.getContext().getParam();
        ForumPostVo forumPostVo = event.getContext().getSaveNote();

        bizLogIntegrationService.noteSucceed(noteParam, forumPostVo.getId(), forumPostVo.getUserId(), forumPostVo.getAreaCode());
    }
}
