package com.bxm.localnews.news.create.listener;

import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.integration.BizLogIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.action.PostLikeService;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.create.context.UserPostContext;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.event.ForumPostActionEvent;
import com.bxm.localnews.news.event.UserActionEvent;
import com.bxm.localnews.news.event.UserPostCreateEvent;
import com.bxm.localnews.news.model.vo.ForumBasicVo;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.post.ForumPostService;
import com.bxm.localnews.news.post.PostCountService;
import com.bxm.localnews.news.statistics.ForumPostStatisticService;
import com.bxm.localnews.news.task.FirstPostLikeCallback;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.eventbus.EventBus;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.Calendar;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 用户发帖成功后的监听器
 *
 * @author liujia
 * @date 1/15/21 4:31 PM
 **/
@Slf4j
@Component
@AllArgsConstructor
public class UserPostCreateListener {

    private final PostLikeService postLikeService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final MixedRecommendPoolMapper mixedRecommendPoolMapper;

    private final EventBus userActionEventBus;

    private final BizLogIntegrationService bizLogIntegrationService;

    private final ForumPostStatisticService forumPostStatisticService;

    private final PostCountService postCountService;

    private final ForumPostService forumPostService;

    private UserIntegrationService userIntegrationService;

    private final ScheduleService scheduleService;

    private final FirstPostLikeCallback firstPostLikeCallback;

    /**
     * 删除帖子的话题，重新插入新的选择的话题，一个帖子可以选择多个话题
     *
     * @param event 用户发帖完成事件
     */
    @EventListener
    public void setPostTopic(UserPostCreateEvent event) {
        ForumPostVo forumPostVo = event.getContext().getSavePost();

        forumPostService.updateTopic(forumPostVo.getId(),
                forumPostVo.getTopicIdList(),
                forumPostVo.getSecondTopicId());
    }

    /**
     * 帖子发布完成后，增加话题广场的显示数量
     *
     * @param event 用户发帖完成事件
     */
    @EventListener
    public void addCrumbsAreaTotal(UserPostCreateEvent event) {
        if (event.getContext().isUpdatePost()) {
            return;
        }

        ForumBasicVo forumBasicVo = event.getContext().getRequestPost();
        // 发布帖子时，增加话题广场面包块消息数
        KeyGenerator crumbsAreaTotalKey = RedisConfig.CRUMBS_AREA_TOTAL.copy().appendKey(1);
        // 获取key是否存在
        boolean keyExists = redisHashMapAdapter.exists(crumbsAreaTotalKey, forumBasicVo.getAreaCode());
        // 如果key存在
        if (keyExists) {
            // 发布帖子数量在原来基础上+1
            redisHashMapAdapter.increment(crumbsAreaTotalKey, forumBasicVo.getAreaCode(), 1);
        } else {
            //初始消息数
            final int initNum = 1;
            // 如果不是存在则初始化帖子数为1
            redisHashMapAdapter.put(crumbsAreaTotalKey, forumBasicVo.getAreaCode(), initNum);
        }
    }

    /**
     * 用户编辑帖子后，取消推荐
     *
     * @param event 用户发帖完成事件
     */
    @EventListener
    public void cancelRecommend(UserPostCreateEvent event) {
        UserPostContext context = event.getContext();
        if (context.isUpdatePost()) {
            Long postId = context.getSavePost().getId();
            this.mixedRecommendPoolMapper.deleteMixRecommandPoolById(postId);
        }
    }

    /**
     * 用户首次发帖后，发送事件
     *
     * @param event 用户发帖完成事件
     */
    @EventListener
    public void sendUserPostEvent(UserPostCreateEvent event) {
        if (event.getContext().isUpdatePost()) {
            return;
        }

        ForumPostVo forumPostVo = event.getContext().getSavePost();

        //上报发帖信息到用户推荐数据源服务
        UserActionEvent userActionEvent = ForumPostActionEvent.of()
                .setLastPostId(forumPostVo.getId())
                .setLastPostTime(new Date())
                .setUserId(forumPostVo.getUserId());

        userActionEventBus.post(userActionEvent);
    }

    /**
     * 发帖后上报对应的数据埋点
     *
     * @param event 用户发帖完成事件
     */
    @EventListener
    public void reportBizLog(UserPostCreateEvent event) {
        if (event.getContext().isUpdatePost()) {
            return;
        }

        ForumBasicVo forumBasicVo = event.getContext().getRequestPost();
        ForumPostVo forumPostVo = event.getContext().getSavePost();
        BasicParam basicParam = event.getContext().getBasicParam();

        // 新人发帖事件
        if (Objects.nonNull(forumBasicVo.getIsNewReport()) && forumBasicVo.getIsNewReport() == 1) {
            PointReportParam reportParam = PointReportParam.build(basicParam)
                    .e("3034")
                    .ev("122." + forumPostVo.getId())
                    .put("uid", String.valueOf(forumPostVo.getUserId()))
                    .put("a", forumPostVo.getAreaCode());

            bizLogIntegrationService.point(reportParam);
        }

        // 用户发帖事件
        if (basicParam != null) {
            PointReportParam param = PointReportParam.build(basicParam);

            param.e("2011")
                    .put("a", forumPostVo.getAreaCode())
                    .put("nid", String.valueOf(forumPostVo.getId()))
                    .put("uid", String.valueOf(forumPostVo.getUserId()));

            bizLogIntegrationService.point(param);
        }
    }

    /**
     * 增加用户的发帖数量统计
     */
    @EventListener
    public void addVirtualLikeAndComment(UserPostCreateEvent event) {
        if (event.getContext().isUpdatePost()) {
            return;
        }

        ForumPostVo forumPostVo = event.getContext().getSavePost();

        //增加点赞评论数，如果是新人点赞直接加点赞数（异步）
        if (Objects.equals(PostStatusEnum.BLOCKED.getCode(), forumPostVo.getStatus())
                || Objects.equals(forumPostVo.getIsNewReport(), (byte) 1)) {
            ThreadLocalRandom random = ThreadLocalRandom.current();
            int num = random.nextInt(5, 20);

            //增加马甲号点赞
            postLikeService.addPostLike(forumPostVo, num);
            //增加马甲号评论
            forumPostStatisticService.addPostReply(forumPostVo);

            Date fireDate = DateUtils.addField(new Date(), Calendar.SECOND, 30);
            scheduleService.push(OnceTaskBuilder.builder(fireDate, firstPostLikeCallback)
                    .taskName("firstRefreshPostLike_" + forumPostVo.getId())
                    .build());
        }
    }

    /**
     * 设置帖子的预计点击数
     */
    @EventListener
    public void addClickCount(UserPostCreateEvent event) {
        if (event.getContext().isUpdatePost()) {
            return;
        }

        ForumPostVo forumPostVo = event.getContext().getSavePost();
        // 初始基数
        long initialBasicNum = postCountService.getInitialBasicNum(forumPostVo.getAreaCode(), false);

        forumPostVo.setInitialBasicNum(initialBasicNum);
        // 该帖子最终需要达到的用户阅读数量
        forumPostVo.setFinalClickCount(postCountService.getFinalClickCount(0, initialBasicNum));
    }

    /**
     * 增加用户的发帖数量统计
     */
    @EventListener
    public void addUserPublishNum(UserPostCreateEvent event) {
        if (event.getContext().isUpdatePost()) {
            return;
        }

        ForumPostVo forumPostVo = event.getContext().getSavePost();
        // 增加用户的发布数量
        userIntegrationService.addPostReplyNum(forumPostVo.getUserId(), 1);
    }

    /**
     * 如果包含视频，则进行对应的转码
     */
    @EventListener
    @Async
    public void transCodePostVideo(UserPostCreateEvent event) {
        ForumPostVo forumPostVo = event.getContext().getSavePost();

        if (log.isDebugEnabled()) {
            log.debug("触发视频转码,帖子ID：{}", forumPostVo.getId());
        }

        forumPostService.execTransCode(forumPostVo.getId(), null, forumPostVo.getPostImgList());
    }
}
