package com.bxm.localnews.news.create.process.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.ClientConfigProperties;
import com.bxm.localnews.news.create.PostCreateService;
import com.bxm.localnews.news.model.enums.SpecialForumEnum;
import com.bxm.localnews.news.model.param.PostProcessContext;
import com.bxm.localnews.news.model.vo.ForumBasicVo;
import com.bxm.localnews.news.post.ForumPostService;
import com.bxm.localnews.news.util.PlaceHolderUtil;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;

import static com.alibaba.fastjson.JSON.toJSONString;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.springframework.util.CollectionUtils.isEmpty;

/**
 * 发送新人报道帖子
 *
 * @author gonzo
 * @date 2020-11-25 16:03
 **/
@Slf4j
@AllArgsConstructor
@Service
public class NovicePostProcess extends AbstractPostProcess {

    private final ClientConfigProperties clientConfigProperties;

    private final ForumPostService forumPostService;

    private final LocationIntegrationService locationIntegrationService;

    private final UserIntegrationService userIntegrationService;

    /**
     * 可以抽取一个工具类...
     */
    private static final String USER_NAME = "userName";

    private static final String AREA_NAME = "areaName";

    private static final AtomicReference<Long> NOVICE_TOPIC_ID = new AtomicReference();

    private final PostCreateService postCreateService;

    @Override
    protected Message doPost(PostProcessContext context) {
        // 地址信息 用选择的城市信息
        LocationDTO locationDetailByCode = locationIntegrationService.getLocationByGeocode(context.getParam().getAreaCode());
        String locationName = isNull(locationDetailByCode) ? "" : locationDetailByCode.getName();

        UserBean userInfo = userIntegrationService.selectUserFromCache(context.getParam().getUserId());

        // 用户头像
        List<PostImgVo> postImg = Collections.emptyList();

        if (nonNull(userInfo)) {
            PostImgVo postImgVo = new PostImgVo();
            postImgVo.setType("IMG");
            postImgVo.setImgUrl(userInfo.getHeadImg());
            postImg = Collections.singletonList(postImgVo);
        }

        ForumBasicVo forumBasicVo = new ForumBasicVo();
        forumBasicVo.setAreaCode(context.getParam().getAreaCode());
        forumBasicVo.setTopicIdList(Collections.singletonList(getNoviceTopicId()));
        forumBasicVo.setIsNewReport((byte) 1);
        forumBasicVo.setTitle("");
        forumBasicVo.setUserId(context.getParam().getUserId());
        forumBasicVo.setLocation(locationName);
        forumBasicVo.setTextField(getPublishContent(userInfo, locationName));
        forumBasicVo.setPostImgList(postImg);
        log.info("用户: {} 发送新人帖子信息: {}", context.getParam().getUserId(), toJSONString(forumBasicVo));

        // 发帖
        return postCreateService.saveUserPost(forumBasicVo, context.getParam(), false);
    }

    private Long getNoviceTopicId() {
        if (Objects.isNull(NOVICE_TOPIC_ID.get())) {
            String newReport = clientConfigProperties.getNewReport();
            JSONObject jsonObject = JSON.parseObject(newReport);
            NOVICE_TOPIC_ID.set(jsonObject.getLong("id"));
        }

        return NOVICE_TOPIC_ID.get();
    }

    private String getPublishContent(UserBean userInfo, String locationName) {
        String publishContent = "新人报到，来混个眼熟，以后我会经常出现~";

        if (nonNull(clientConfigProperties.getPublishContent()) && nonNull(userInfo) && isNotBlank(locationName)) {
            List<String> publishContents = JSON.parseArray(clientConfigProperties.getPublishContent(), String.class);
            if (!isEmpty(publishContents)) {
                // 随机获取一个
                publishContent = publishContents.get(RandomUtils.nextInt(0, publishContents.size()));
                // 替换关键字
                publishContent = replaceKeyWord(publishContent, userInfo.getNickname(), locationName);
            }
        }

        return publishContent;
    }

    private String replaceKeyWord(String str, String userName, String areaName) {
        str = PlaceHolderUtil.replace(str, AREA_NAME, areaName, areaName);
        return PlaceHolderUtil.replace(str, USER_NAME, userName, userName);
    }

    @Override
    public SpecialForumEnum support() {
        return SpecialForumEnum.NOVICE;
    }
}
