package com.bxm.localnews.news.create.rule;

import com.bxm.localnews.news.constant.OrderConstant;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.create.context.AdminPostContext;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashInfoMapper;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashInfoEntity;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.newidea.component.annotations.RuleBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.rule.IRule;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.news.constant.LogicGroupConstant.ADMIN_POST_CREATE_RULE;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/1/25 2:13 下午
 */
@RuleBean(group = ADMIN_POST_CREATE_RULE)
@Slf4j
@AllArgsConstructor
public class AdminPostAwardRule implements IRule<AdminPostContext> {

    private final RedisStringAdapter redisStringAdapter;

    private final ForumPostShareCashInfoMapper postShareCashInfoMapper;

    @Override
    public boolean apply(AdminPostContext adminPostContext) {
        AdminForumPost requestPost = adminPostContext.getRequestPost();
        AdminForumPost beforePost = adminPostContext.getBeforePost();

        //如果要保存的不是热文帖子则直接返回
        if (Objects.equals(requestPost.getIsShareCash(), 0)) {
            return true;
        }

        Date awardStartTime = requestPost.getAwardStartTime();
        Date awardEndTime = requestPost.getAwardEndTime();

        if (Objects.nonNull(awardStartTime) && Objects.nonNull(awardEndTime)) {
            if (DateUtils.after(awardStartTime, awardEndTime)) {
                adminPostContext.setErrorMsg("热文有效开始时间不能晚于结束时间");
                return false;
            }
        }

        //此处仅做验证
        // 如果是修改,比较是增加还是减少
        if (adminPostContext.isUpdatePost()) {

            ForumPostShareCashInfoEntity shareCashInfoEntity = postShareCashInfoMapper.getByPostId(beforePost.getId());
            // 如果是空值说明第一次设置为热文帖子
            if (Objects.isNull(shareCashInfoEntity)) {
                return true;
            } else {
                beforePost.setAmount(shareCashInfoEntity.getAmount());
            }

            BigDecimal changeAmount = requestPost.getAmount().subtract(beforePost.getAmount());
            //如果小于零，尝试比较金额,判断金额是否充足
            if (changeAmount.compareTo(BigDecimal.ZERO) < 0) {
                //剩余金额
                BigDecimal remainingAmount = redisStringAdapter.get(buildPostAwardKey(requestPost.getId()), BigDecimal.class);
                //判断是否够减
                if (remainingAmount.doubleValue() < (Math.abs(changeAmount.doubleValue()))) {
                    adminPostContext.setErrorMsg("热文分享奖金不足");
                    return false;
                }
            }
        }
        return true;
    }

    @Override
    public int getOrder() {
        return OrderConstant.getOrder(this.getClass());
    }

    /**
     * 构建热文奖励金额key
     *
     * @param postId 帖子id
     * @return redis key
     */
    private KeyGenerator buildPostAwardKey(Long postId) {
        return RedisConfig.SHARE_POST_AWARD_KEY.copy().appendKey(postId);
    }

}