package com.bxm.localnews.news.create.rule;

import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.OrderConstant;
import com.bxm.localnews.news.create.context.AdminPostContext;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.newidea.component.annotations.RuleBean;
import com.bxm.newidea.component.rule.IRule;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.math.BigDecimal;

import static com.bxm.localnews.news.constant.LogicGroupConstant.ADMIN_POST_CREATE_RULE;

/**
 * 后台管理发帖的前置检查
 *
 * @author liujia
 * @date 1/18/21 3:13 PM
 **/
@RuleBean(group = ADMIN_POST_CREATE_RULE)
@Slf4j
@AllArgsConstructor
public class AdminPostCheckRule implements IRule<AdminPostContext> {

    private final ForumProperties forumProperties;

    @Override
    public boolean apply(AdminPostContext context) {
        AdminForumPost adminForumPost = context.getRequestPost();

        //id必传
        if (null == adminForumPost.getId()) {
            context.setErrorMsg("ID未传");
            return false;
        }

        // 判断运营发放奖励是否超额
        BigDecimal cashReward = adminForumPost.getCashReward();
        if (log.isDebugEnabled()) {
            log.debug("现金发放的金额：{}", cashReward);
        }

        if (cashReward != null) {
            int cash = cashReward.intValue();
            if (log.isDebugEnabled()) {
                log.debug("进入判断转化为int类型：{}", cash);
            }
            if (cash > forumProperties.getPostRewardCashMax()) {
                context.setErrorMsg("现金奖励最大" + forumProperties.getPostRewardCashMax() + "元");
                return false;
            }
            if (cash < forumProperties.getPostRewardCashMin()) {
                context.setErrorMsg("现金奖励最小" + forumProperties.getPostRewardCashMin() + "元");
                return false;
            }
        }

        // 热文有效期不得早于帖子展示发布时间
        if (DateUtils.before(adminForumPost.getAwardStartTime(), adminForumPost.getPublishTime())) {
            context.setErrorMsg("热文有效期不得在发布时间之前");
            return false;
        }

        if (DateUtils.after(adminForumPost.getAwardStartTime(), adminForumPost.getAwardEndTime())) {
            context.setErrorMsg("热文有效期截止时间在起始时间之前");
            return false;
        }

        return true;
    }

    @Override
    public int getOrder() {
        return OrderConstant.getOrder(this.getClass());
    }
}
