package com.bxm.localnews.news.detail.impl;

import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.news.action.PostClickService;
import com.bxm.localnews.news.constant.LogicGroupConstant;
import com.bxm.localnews.news.detail.ForumPostDetailService;
import com.bxm.localnews.news.detail.context.ForumPostDetailContext;
import com.bxm.localnews.news.detail.context.PostDetailOriginalParam;
import com.bxm.localnews.news.detail.filter.*;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.newidea.component.filter.FilterChainExecutor;
import com.bxm.newidea.component.rule.RuleGroupExecutor;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

import static com.bxm.localnews.news.constant.LogicGroupConstant.POST_DETAIL_FILTER;

/**
 * @author liujia
 * @date 1/13/21 9:13 PM
 **/
@Service
@Slf4j
public class ForumPostDetailServiceImpl implements ForumPostDetailService {

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private RuleGroupExecutor ruleGroupExecutor;

    @Resource
    private FilterChainExecutor filterChainExecutor;

    @Resource
    private PostClickService postClickService;

    @Override
    public ForumPostVo get(PostDetailOriginalParam param) {
        long start = System.currentTimeMillis();

        if (log.isDebugEnabled()) {
            log.debug("获取帖子详情，请求参数：{}", param);
        }

        Preconditions.checkArgument(null != param && null != param.getId());

        ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(param.getId());

        if (null != forumPostVo) {
            ForumPostDetailContext context = new ForumPostDetailContext(param);
            context.setPostInfo(forumPostVo);
            context.skipFilter(HotReplayFillFilter.class);

            // 判断帖子是否可以展示
            if (!ruleGroupExecutor.apply(LogicGroupConstant.POST_DETAIL_RULE, context)) {
                return context.getPostInfo();
            }

            // 填充帖子内容
            filterChainExecutor.parallelDoFilter(POST_DETAIL_FILTER, context);

            // 请求结束后的事件，用于增加阅读数、增加站外阅读限制等
            if (null != param.getBasicParam()) {
                postClickService.doAsyncReadPost(context.getUserId(),
                        context.getPostId(),
                        param.getShareUserId(),
                        param.getBasicParam().getPlatform(),
                        forumPostVo,
                        param.getIp());
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("帖子[{}]获取耗时:{}", param.getId(), System.currentTimeMillis() - start);
        }

        // 返回帖子信息，在过滤器中已经进行了内容填充处理
        return forumPostVo;
    }

    @Override
    public ForumPostVo get(ForumPostDetailContext context) {
        ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(context.getPostId());

        if (context.getOriginalParam().getBasicParam() == null) {
            context.getOriginalParam().setBasicParam(new BasicParam());
        }

        context.setPostInfo(forumPostVo);
        context.skipFilter(HotReplayFillFilter.class);

        filterChainExecutor.doFilter(POST_DETAIL_FILTER, context);

        return forumPostVo;
    }

    @Override
    public ForumPostVo getBriefPost(Long postId, Long userId, String areaCode) {
        ForumPostVo forumPostVo = forumPostMapper.selectWithoutContent(postId);

        PostDetailOriginalParam param = new PostDetailOriginalParam();
        param.setId(postId);
        param.setUserId(userId);
        param.setAreaCode(areaCode);

        ForumPostDetailContext context = new ForumPostDetailContext(param);
        context.setPostInfo(forumPostVo);

        // 去掉一些耗时的，正常帖子信息使用不到的过滤器
        context.skipFilter(LikeFillFilter.class)
                .skipFilter(CollectFillFilter.class)
                .skipFilter(RelationFillFilter.class)
                .skipFilter(TopicFillFilter.class)
                .skipFilter(PluginLoadFilter.class)
                .skipFilter(ContentAssemblyFilter.class)
                .skipFilter(HotReplayFillFilter.class)
                .skipFilter(VestTypeFillFilter.class)
                .skipFilter(ActivityPostFillFilter.class)
                .skipFilter(ShareCashPostFillFilter.class)
                .skipFilter(CommentCalFilter.class);

        filterChainExecutor.doFilter(POST_DETAIL_FILTER, context);

        return forumPostVo;
    }
}
