package com.bxm.localnews.news.hotpost.impl;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.activity.ForumPostActivityMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashInfoMapper;
import com.bxm.localnews.news.hotpost.HotPostBulletService;
import com.bxm.localnews.news.model.dto.PostBarrageContentDTO;
import com.bxm.localnews.news.model.dto.PostBarrageDTO;
import com.bxm.localnews.news.model.entity.activity.ForumPostActivityEntity;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashInfoEntity;
import com.bxm.localnews.news.model.param.activity.PostDetailBarrageParam;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.hotpost.HotPostBulletVO;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.localnews.vo.VirtualUserInfo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.RandomUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/1/28 9:46 上午
 */
@Service
@Slf4j
@AllArgsConstructor
public class HotPostBulletServiceImpl implements HotPostBulletService {

    private final UserIntegrationService userIntegrationService;

    private final ForumPostShareCashInfoMapper forumPostShareCashInfoMapper;

    private final ForumPostActivityMapper forumPostActivityMapper;

    private final ForumPostMapper forumPostMapper;

    private final RedisSetAdapter redisSetAdapter;

    /**
     * 弹幕用户人数
     */
    private static final int BULLET_USER_NUM = 10;

    @Override
    public PostBarrageDTO getPostBarrage(PostDetailBarrageParam param) {

        PostBarrageDTO resultDTO = new PostBarrageDTO();

        ForumPostVo forumPostVo = forumPostMapper.selectWithoutContent(param.getPostId());

        ForumPostActivityEntity forumPostActivityEntity = forumPostActivityMapper.selectByPostId(param.getPostId());

        // 如果不是活动帖子，则直接返回空集合
        if (Objects.equals(forumPostVo.getActivityPost(), 0) ||
                Objects.isNull(forumPostActivityEntity) ||
                Objects.equals(forumPostActivityEntity.getStatus(), 0)) {
            resultDTO.setContentList(Collections.emptyList());
            resultDTO.setBulletType(1);

            return resultDTO;
        }

        //判断站内站外，如果是站外则直接返回活动帖子弹幕
        if (param.getPlatform() == PlatformEnum.WEB.getCode()) {
            buildActivityBarrage(param.getPostId(), resultDTO);
            return resultDTO;
        }

        ForumPostShareCashInfoEntity postShareCashInfoEntity = forumPostShareCashInfoMapper.getByPostId(param.getPostId());

        //热文有效则显示热文奖励弹幕,1、是一个热文 2、热文信息不问空 3、热文状态为有效状态
        if (Objects.equals(forumPostVo.getShareCash(), 1) &&
                Objects.nonNull(postShareCashInfoEntity) &&
                Objects.equals(postShareCashInfoEntity.getStatus(), 1)) {
            buildSharePostBarrage(resultDTO, forumPostVo.getShareCount());
        } else {
            //参与活动弹幕
            buildActivityBarrage(param.getPostId(), resultDTO);
        }

        return resultDTO;
    }

    /**
     * 构建活动帖子弹幕列表
     *
     * @param postId    帖子id
     * @param resultDTO 返回弹幕结果对象
     */
    private void buildActivityBarrage(Long postId, PostBarrageDTO resultDTO) {
        resultDTO.setBulletType(1);

        Set<Long> joinActivityUsers = redisSetAdapter.getAllMembers(buildJoinActivityUserKey(postId), Long.class);
        //如果点击参与人数小于三则返回空的弹幕, 或者帖子状态为下架，返回空弹幕
        if (joinActivityUsers.size() < 3) {
            resultDTO.setContentList(Collections.emptyList());
            return;
        }

        List<PostBarrageContentDTO> contentList = joinActivityUsers.stream().limit(10).map(userId -> {
            UserBean userCache = userIntegrationService.selectUserFromCache(userId);
            PostBarrageContentDTO contentDTO = new PostBarrageContentDTO();
            contentDTO.setUserId(userId);
            contentDTO.setHeadImg(userCache.getHeadImg());
            contentDTO.setNickName(retentionFiveLength(userCache.getNickname()));

            return contentDTO;
        }).collect(Collectors.toList());

        resultDTO.setContentList(contentList);
    }

    @Override
    public List<HotPostBulletVO> shareHotPostBullet(String areaCode) {
        List<VirtualUserInfo> virtualUserList = userIntegrationService.getVirtualUserList(BULLET_USER_NUM);
        List<HotPostBulletVO> resultList = new ArrayList<>();

        virtualUserList.forEach(u -> {
            HotPostBulletVO bulletItemVO = new HotPostBulletVO();
            BeanUtils.copyProperties(u, bulletItemVO);
            bulletItemVO.setUserId(u.getId());
            resultList.add(bulletItemVO);
        });
        return resultList;
    }

    /**
     * 构建热文分享奖金弹幕列表
     *
     * @param resultDTO  弹幕最终返回结果对象
     * @param shareCount 分享数量
     */
    private void buildSharePostBarrage(PostBarrageDTO resultDTO, Integer shareCount) {
        resultDTO.setBulletType(0);

        //如果分享数量小于10则返回一个空的弹幕列表
        if (shareCount < 10) {
            resultDTO.setContentList(Collections.emptyList());
            return;
        }

        List<VirtualUserInfo> virtualUserList = userIntegrationService.getVirtualUserList(BULLET_USER_NUM);

        List<PostBarrageContentDTO> contentList = virtualUserList.stream().map(virtualUserInfo -> {
            PostBarrageContentDTO contentDTO = new PostBarrageContentDTO();

            contentDTO.setNickName(retentionFiveLength(virtualUserInfo.getNickname()));
            contentDTO.setHeadImg(virtualUserInfo.getHeadImg());
            contentDTO.setUserId(virtualUserInfo.getId());
            contentDTO.setAwardAmount(BigDecimal.valueOf(RandomUtils.randomDouble(2, 0.3, 3)));
            return contentDTO;
        }).collect(Collectors.toList());

        resultDTO.setContentList(contentList);

    }

    /**
     * 构建参加活动帖子缓存key
     *
     * @param postId 帖子id
     * @return 缓存key
     */
    private KeyGenerator buildJoinActivityUserKey(Long postId) {
        return RedisConfig.ACTIVITY_POST_JOIN_USER_KEY.copy().appendKey(postId);
    }

    /**
     * 保留用户五位的用户名
     *
     * @return 处理后的用户名
     */
    private String retentionFiveLength(String sourceNickname) {

        if (StringUtils.isNotBlank(sourceNickname)) {
            if (sourceNickname.length() > 5) {
                return sourceNickname.substring(0, 5) + "...";
            } else {
                return sourceNickname;
            }
        }
        return "";
    }
}