package com.bxm.localnews.news.hotpost.impl;

import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.PushSoundEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.localnews.news.config.BizConfigProperties;
import com.bxm.localnews.news.config.H5JumpAddressProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashCounterMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashInfoMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashRecordMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashSingleCountMapper;
import com.bxm.localnews.news.enums.HotPostCashSettlementStatusEnum;
import com.bxm.localnews.news.enums.HotPostShareRecordStatusEnum;
import com.bxm.localnews.news.hotpost.HotPostSettlementService;
import com.bxm.localnews.news.hotpost.filter.chain.HotPostCashFilterChain;
import com.bxm.localnews.news.hotpost.filter.context.HotPostShareContext;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashCounterEntity;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashInfoEntity;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashRecordEntity;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashSingleCountEntity;
import com.bxm.localnews.news.model.param.CashAccountParam;
import com.bxm.localnews.news.model.param.EarningsSaveOrUpdateParam;
import com.bxm.localnews.news.model.vo.hotpost.HotPostReadDataDTO;
import com.bxm.localnews.url.ProtocolFactory;
import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.UserEarningsTypeEnum;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 热文帖子结算service实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/1/27 2:16 下午
 */
@Service
@Slf4j
@AllArgsConstructor
public class HotPostSettlementServiceImpl implements HotPostSettlementService {

    private final HotPostCashFilterChain hotPostCashFilterChain;

    private final RedisStringAdapter redisStringAdapter;

    private final ForumPostShareCashRecordMapper forumPostShareCashRecordMapper;

    private final ForumPostShareCashInfoMapper forumPostShareCashInfoMapper;

    private final ForumPostShareCashCounterMapper forumPostShareCashCounterMapper;

    private final ForumPostShareCashSingleCountMapper forumPostShareCashSingleCountMapper;

    private final SequenceCreater sequenceCreater;

    private final RedisSetAdapter redisSetAdapter;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final BizConfigProperties bizConfigProperties;

    private final MessageSender messageSender;

    private final DomainIntegrationService domainIntegrationService;

    private final H5JumpAddressProperties h5JumpAddressProperties;

    @Override
    public void handleReadData(HotPostReadDataDTO hotPostReadDataDTO) {
        //获取所有热文帖子id
        ForumPostShareCashInfoEntity postShareCashInfoEntity = forumPostShareCashInfoMapper.getByPostId(hotPostReadDataDTO.getPostId());
        BigDecimal postAward = postShareCashInfoEntity.getAward();

        handleHotPostReadData(hotPostReadDataDTO, postAward);
    }

    @Override
    public void settlementHotPostCash() {

        String yesterdayStr = getYesterdaySimpleDateStr();
        String yesterdayNoRepeatStr = getYesterdayDateNoSeparatorStr();

        List<ForumPostShareCashCounterEntity> forumPostShareCashCounterEntityList = forumPostShareCashCounterMapper.selectByAwardDate(yesterdayStr);

        //结算昨天的用户奖励
        for (ForumPostShareCashCounterEntity counterEntity : forumPostShareCashCounterEntityList) {
            //必须为待结算
            if (counterEntity.getStatus() != 0) {
                continue;
            }
            //增加收益到用户账户，收益流水
            EarningsSaveOrUpdateParam userEarningsParam = new EarningsSaveOrUpdateParam();
            userEarningsParam.setEarningsType(UserEarningsTypeEnum.SHARE_HOT_POST.getEarningsType());
            userEarningsParam.setEarningsTitle("分享热文奖励金");
            userEarningsParam.setOrderTime(new Date());
            userEarningsParam.setImg(bizConfigProperties.getPlatFromImgUrl());

            // 调用用户服务，给用户增加可用余额K
            CashAccountParam param = new CashAccountParam();
            param.setExtData(userEarningsParam);
            param.setAccountAction(AccountActionEnum.ADD_HOT_POST_SHARE_CASH);
            param.setCashFlowType(CashFlowTypeEnum.OTHER);
            param.setAmount(counterEntity.getAwardAmount());
            param.setUserId(counterEntity.getUserId());
            param.setRelationId(counterEntity.getId());
            param.setRemark("您的分享的帖子昨日获得奖励" + counterEntity.getAwardAmount().stripTrailingZeros().toPlainString() + "元");

            //如果增加用户可提现金额成功，则设置当日的统计数据为已结算
            boolean addDrawableCashResult = userAccountIntegrationService.cashAccountOperation(param);

            //更新分享热文收益余额，将当日的统计记录标记为已结算
            if (addDrawableCashResult) {
                ForumPostShareCashCounterEntity updateCashCounterEntity = new ForumPostShareCashCounterEntity();

                //设置为已结算
                updateCashCounterEntity.setId(counterEntity.getId());
                updateCashCounterEntity.setStatus(HotPostCashSettlementStatusEnum.SETTLEMENT.getCode());

                forumPostShareCashCounterMapper.updateByPrimaryKeySelective(updateCashCounterEntity);
            }
        }
        //移除前一天的用户阅读缓存（记录只能阅读一次的缓存）
        redisSetAdapter.remove(buildJoinSharePostUserKey(yesterdayNoRepeatStr));
    }


    @Override
    public void pushSettlementMsg() {
        String yesterdayStr = getYesterdaySimpleDateStr();

        List<ForumPostShareCashCounterEntity> settlementList = forumPostShareCashCounterMapper.selectByAwardDateAndSettlement(yesterdayStr);

        for (ForumPostShareCashCounterEntity counterEntity : settlementList) {
            executePushSettlement(counterEntity);
        }
    }

    /**
     * 执行收益到账推送
     *
     * @param counterEntity 推送的收益信息
     */
    private void executePushSettlement(ForumPostShareCashCounterEntity counterEntity) {
        String title = "分享热文奖励金已到账";
        String content = String.format("您昨日分享的本地热文，有%s人阅读，平台奖励%s元已到账，继续分享领更多～",
                counterEntity.getReadCount(), counterEntity.getAwardAmount().setScale(2, RoundingMode.HALF_DOWN));

        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.JUMP_TYPE);

        //TODO
        String url = domainIntegrationService.getInnerH5BaseUrl() +
                String.format(h5JumpAddressProperties.getIncomePageUrl(), counterEntity.getUserId());

        String protocol = ProtocolFactory.appH5().url(url).build();

        info.setProtocol(protocol);

        PushMessage message = PushMessage.build();
        message.setTitle(title);
        message.setContent(content);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPayloadInfo(info);
        message.setSound(PushSoundEnum.DEFAULT_SOUND);

        message.assign(counterEntity.getUserId());

        messageSender.sendPushMessage(message);
    }

    /**
     * 处理热门帖子阅读数据
     */
    private void handleHotPostReadData(HotPostReadDataDTO readDataDTO, BigDecimal award) {

        HotPostShareContext shareContext = new HotPostShareContext();

        BeanUtils.copyProperties(readDataDTO, shareContext);

        Message checkResult = hotPostCashFilterChain.handle(shareContext);

        if (Boolean.FALSE.equals(checkResult.isSuccess())) {
            log.warn("用户奖励校验失败，原因：" + checkResult.getLastMessage() + "，帖子id：{}, 用户id:{}", readDataDTO.getPostId(), readDataDTO.getShareUserId());
            return;
        }

        //如果不存在，则去初始化当前奖金金额信息
        if (Boolean.FALSE.equals(redisStringAdapter.hasKey(buildPostAwardKey(readDataDTO.getPostId())))) {
            loadHotPostShareDb(readDataDTO.getPostId());
        }

        //尝试去减金额,当缓存存，并且金额减完大于等于零
        if (redisStringAdapter.decrement(buildPostAwardKey(readDataDTO.getPostId()), award.doubleValue()) >= 0) {
            //1、已发金额增加
            int increaseResult = forumPostShareCashInfoMapper.increaseSendAward(readDataDTO.getPostId(), award);
            //如果影响行数大于0
            if (increaseResult > 0) {
                //获取奖励成功后续操作,更新奖励金额，和已发奖励金额
                ForumPostShareCashRecordEntity newRecordEntity = createForumPostShareCashRecordEntity(shareContext, award);

                //2、插入获取奖励记录
                forumPostShareCashRecordMapper.insertSelective(newRecordEntity);

                //3、增加单用户统计信息
                if (Objects.isNull(forumPostShareCashSingleCountMapper.selectByUserIdAndPostId(readDataDTO.getShareUserId(), readDataDTO.getPostId()))) {
                    //创建用户单用户统计信息
                    ForumPostShareCashSingleCountEntity newSingleCountEntity = new ForumPostShareCashSingleCountEntity();
                    newSingleCountEntity.setUserId(readDataDTO.getShareUserId());
                    newSingleCountEntity.setPostId(readDataDTO.getPostId());
                    newSingleCountEntity.setId(sequenceCreater.nextLongId());

                    forumPostShareCashSingleCountMapper.insertSelective(newSingleCountEntity);
                }
                //增加有效阅读人数
                forumPostShareCashSingleCountMapper.increaseReadCount(readDataDTO.getShareUserId(), readDataDTO.getPostId());
                //增加奖励数量
                forumPostShareCashSingleCountMapper.increaseAmount(readDataDTO.getShareUserId(), readDataDTO.getPostId(), award);

                //4、增加用户帖子统计信息
                String currentDayStr = getSimpleDateStr(readDataDTO.getReportDate());

                if (Objects.isNull(forumPostShareCashCounterMapper.selectByDateAndUserId(readDataDTO.getShareUserId(), currentDayStr))) {
                    //创建用户今日统计信息
                    ForumPostShareCashCounterEntity newCounterEntity = new ForumPostShareCashCounterEntity();
                    newCounterEntity.setUserId(readDataDTO.getShareUserId());
                    newCounterEntity.setAwardDate(currentDayStr);
                    newCounterEntity.setId(sequenceCreater.nextLongId());

                    forumPostShareCashCounterMapper.insertSelective(newCounterEntity);
                }

                //增加当前日期收益和阅读数
                forumPostShareCashCounterMapper.increaseTodayAwardAmount(readDataDTO.getShareUserId(), currentDayStr, award);
                forumPostShareCashCounterMapper.increaseTodayReadCount(readDataDTO.getShareUserId(), currentDayStr);

                BigDecimal remainAmount = redisStringAdapter.get(buildPostAwardKey(readDataDTO.getPostId()), BigDecimal.class);
                //如果剩余金额小于等于零，则直接下架
                if (BigDecimal.ZERO.compareTo(remainAmount) >= 0) {
                    //设置热文分享活动为失效
                    forumPostShareCashInfoMapper.offlineHotPost(readDataDTO.getPostId());
                }
            } else {
                log.error("热文帖子奖金已发金额超过活动总金额，出现错误情况，帖子id：{}", readDataDTO.getPostId());
                //下架热文帖子活动
                forumPostShareCashInfoMapper.offlineHotPost(readDataDTO.getPostId());
            }
        } else {
            log.warn("奖金不足，扣除热文奖金失败，帖子id：{}", readDataDTO.getPostId());
            //如果减完奖金之后小于0，则将奖金数量加回来
            redisStringAdapter.increment(buildPostAwardKey(readDataDTO.getPostId()), award.doubleValue());
            //设置热文分享活动为失效
            forumPostShareCashInfoMapper.offlineHotPost(readDataDTO.getPostId());
        }
    }

    /**
     * 重新加载从DB加载奖金金额
     *
     * @param postId 帖子id
     */
    private void loadHotPostShareDb(Long postId) {
        ForumPostShareCashInfoEntity shareCashInfoEntity = forumPostShareCashInfoMapper.getByPostId(postId);
        redisStringAdapter.set(buildPostAwardKey(postId), shareCashInfoEntity.getAmount().subtract(shareCashInfoEntity.getUsedAmount()));
    }


    /**
     * 构建每日用户参与阅读热文帖子key
     *
     * @param simpleDateStr 当前日期字符串
     * @return redis key
     */
    private KeyGenerator buildJoinSharePostUserKey(String simpleDateStr) {
        return RedisConfig.TODAY_JOIN_SHARE_POST_USER_KEY.copy().appendKey(simpleDateStr);
    }

    /**
     * 获取当前日期字符串，格式：2021-01-02
     *
     * @return 当前日期字符串
     */
    private String getSimpleDateStr(Date now) {
        return DateUtils.DATE_FORMAT_THREAD_LOCAL.get().format(now);
    }

    /**
     * 获取当前日期字符串，格式：2021-01-02
     *
     * @return 当前日期字符串
     */
    private String getTodaySimpleDateStr() {
        return DateUtils.DATE_FORMAT_THREAD_LOCAL.get().format(new Date());
    }

    /**
     * 获取当前日期字符串，格式：2021-01-02
     *
     * @return 当前日期字符串
     */
    private String getYesterdaySimpleDateStr() {
        Date yesterday = DateUtils.addField(new Date(), Calendar.DATE, -1);
        return DateUtils.DATE_FORMAT_THREAD_LOCAL.get().format(yesterday);
    }


    /**
     * 获取今日不带连字符的日期字符串
     *
     * @return 日期字符串
     */
    private String getTodayDateNoSeparatorStr() {
        return DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date());
    }

    /**
     * 获取昨日不带连字符的日期字符串
     *
     * @return 日期字符串
     */
    private String getYesterdayDateNoSeparatorStr() {
        Date yesterday = DateUtils.addField(new Date(), Calendar.DATE, -1);
        return DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(yesterday);
    }

    /**
     * 构建热文现金浏览记录
     *
     * @param shareContext 热文分享上下文
     * @return 创建的对象
     */
    private ForumPostShareCashRecordEntity createForumPostShareCashRecordEntity(HotPostShareContext shareContext, BigDecimal award) {
        ForumPostShareCashRecordEntity newRecordEntity = new ForumPostShareCashRecordEntity();
        newRecordEntity.setUserId(shareContext.getShareUserId());
        newRecordEntity.setId(sequenceCreater.nextLongId());
        newRecordEntity.setCreateTime(new Date());
        newRecordEntity.setAward(award);
        newRecordEntity.setIp(shareContext.getIp());
        newRecordEntity.setId(sequenceCreater.nextLongId());
        newRecordEntity.setUserAgent(shareContext.getUserAgent());
        newRecordEntity.setPostId(shareContext.getPostId());
        newRecordEntity.setToken(shareContext.getOpenid());
        newRecordEntity.setStatus(HotPostShareRecordStatusEnum.NOT_SEND.getCode());
        return newRecordEntity;
    }

    /**
     * 构建热文奖励金额key
     *
     * @param postId 帖子id
     * @return redis key
     */
    private KeyGenerator buildPostAwardKey(Long postId) {
        return RedisConfig.SHARE_POST_AWARD_KEY.copy().appendKey(postId);
    }
}