package com.bxm.localnews.news.hotpost.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.WebTokenIntegrationService;
import com.bxm.localnews.news.config.NewsProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashCounterMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashInfoMapper;
import com.bxm.localnews.news.hotpost.HotPostShareService;
import com.bxm.localnews.news.hotpost.msg.sender.HotPostJoinMsgSenderService;
import com.bxm.localnews.news.model.dto.hotpost.HotPostShareCashDTO;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashCounterEntity;
import com.bxm.localnews.news.model.param.activity.ActivityPostReportParam;
import com.bxm.localnews.news.model.param.hotpost.HotPostListPageParam;
import com.bxm.localnews.news.model.param.hotpost.HotPostReportParam;
import com.bxm.localnews.news.model.vo.PostImg;
import com.bxm.localnews.news.model.vo.hotpost.HotPostCounterInfoVO;
import com.bxm.localnews.news.model.vo.hotpost.HotPostItemVO;
import com.bxm.localnews.news.model.vo.hotpost.HotPostReadDataDTO;
import com.bxm.localnews.news.model.vo.hotpost.ReadReportVO;
import com.bxm.localnews.news.util.PlaceHolderUtil;
import com.bxm.localnews.url.ProtocolFactory;
import com.bxm.localnews.vo.WebTokenVerifyVO;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.util.WebUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.parseArray;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 热文分享帖子service实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/1/18 9:38 上午
 */
@Service
@Slf4j
@AllArgsConstructor
public class HotPostShareServiceImpl implements HotPostShareService {

    private static final String USER_AGENT_HEADER_KEY = "user-agent";

    private final ForumPostShareCashInfoMapper forumPostShareCashInfoMapper;

    private final ForumPostShareCashCounterMapper forumPostShareCashCounterMapper;

    private final WebTokenIntegrationService webTokenIntegrationService;

    private final RedisSetAdapter redisSetAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final LocationIntegrationService locationIntegrationService;

    private final NewsProperties newsProperties;

    private final HotPostJoinMsgSenderService hotPostJoinMsgSenderService;

    /**
     * web token header key
     */
    private static final String WEB_TOKEN_HEADER_KEY = "web-token";

    @Override
    public HotPostCounterInfoVO counterInfo(Long userId) {

        String currentDay = getSimpleDateStr(new Date());

        ForumPostShareCashCounterEntity postShareCashCounterEntity =
                forumPostShareCashCounterMapper.selectByDateAndUserId(userId, currentDay);

        HotPostCounterInfoVO resultVO = new HotPostCounterInfoVO();

        //设置总收益
        resultVO.setTotalAward(forumPostShareCashCounterMapper.countAwardAmount(userId, currentDay).setScale(2, RoundingMode.HALF_DOWN));
        //如果为空说明今天没有统计数据，初始化为零
        if (Objects.isNull(postShareCashCounterEntity)) {
            resultVO.setTodayAward(BigDecimal.ZERO);
            resultVO.setTodayReadCount(0);
            return resultVO;
        }
        //不为空则正常取值
        resultVO.setTodayAward(postShareCashCounterEntity.getAwardAmount().setScale(2, RoundingMode.HALF_DOWN));
        resultVO.setTodayReadCount(postShareCashCounterEntity.getReadCount());

        return resultVO;
    }

    @Override
    public List<HotPostItemVO> hotPostList(HotPostListPageParam param) {
        List<HotPostItemVO> hotPostList = forumPostShareCashInfoMapper.getHotPostList(param);

        for (HotPostItemVO hotPostItemVO : hotPostList) {
            //拼接预览图
            String previewImg = hotPostItemVO.getPreviewImg();
            if (isNotBlank(previewImg)) {
                List<PostImg> postImgList = parseArray(previewImg, PostImg.class);
                if (CollectionUtils.isNotEmpty(postImgList)) {
                    hotPostItemVO.setPreviewImg(postImgList.get(0).getImgUrl());
                } else {
                    hotPostItemVO.setPreviewImg(newsProperties.getDefaultShareCover());
                }
            }

            LocationDTO locationInfo = locationIntegrationService.getLocationByGeocode(param.getAreaCode());

            //设置分享地址url
            hotPostItemVO.setShareUrl(ProtocolFactory.forumPost().outer()
                    .postId(hotPostItemVO.getPostId())
                    .userId(param.getUserId())
                    .areaCode(param.getAreaCode())
                    .cityName(locationInfo.getName())
                    .build());

            //处理已发金额，如果分享数为0，则设置已发金额为0
            if (hotPostItemVO.getShareCount() == 0) {
                hotPostItemVO.setSentAward(BigDecimal.ZERO);
            }

            //如果分享数大于0，则最终奖励为真实已发金额 * 2 + 固化的随机数
            if (hotPostItemVO.getShareCount() > 0) {
                BigDecimal randomAward = redisHashMapAdapter.get(buildHotPostRandomAwardKey(), String.valueOf(hotPostItemVO.getPostId()), BigDecimal.class);
                hotPostItemVO.setSentAward(hotPostItemVO.getSentAward().multiply(BigDecimal.valueOf(2))
                        .add(Objects.isNull(randomAward) ? BigDecimal.ZERO : randomAward).setScale(2, RoundingMode.HALF_DOWN));
            }
            //设置精度
            hotPostItemVO.setMyAward(hotPostItemVO.getMyAward().setScale(2, RoundingMode.HALF_DOWN));

            //设置展示标题
            if (StringUtils.isBlank(hotPostItemVO.getPostTitle())) {
                hotPostItemVO.setPostDesc(hotPostItemVO.getPostTextField());
                hotPostItemVO.setPostTitle(hotPostItemVO.getPostTextField());
            } else {
                hotPostItemVO.setPostDesc(hotPostItemVO.getPostTitle());
                hotPostItemVO.setPostTitle(hotPostItemVO.getPostTitle());
            }

            //格式化地区占位符
            hotPostItemVO.setPostTitle(PlaceHolderUtil.replace(hotPostItemVO.getPostTitle(), "areaname", locationInfo.getName(), "本地"));

            //格式化地区占位符
            hotPostItemVO.setPostDesc(PlaceHolderUtil.replace(hotPostItemVO.getPostDesc(), "areaname", locationInfo.getName(), "本地"));
        }

        return hotPostList;
    }

    @Override
    public ReadReportVO reportRead(HotPostReportParam param, HttpServletRequest request) {

        WebTokenVerifyVO webTokenVerifyVO = checkExpiredToken(request);

        if (webTokenVerifyVO.getExpired()) {
            return ReadReportVO.builder().webTokenExpired(true).build();
        }

        if (Objects.isNull(param.getShareUserId())) {
            log.warn("热文帖子分享用户id为空，帖子id：{}", param.getPostId());
            return ReadReportVO.builder().webTokenExpired(false).build();
        }

        log.info("用户参与热文分享：{}", JSON.toJSONString(webTokenVerifyVO));
        //校验用户今天是否参加过热文阅读
        if (checkUserTodayJoin(webTokenVerifyVO.getOpenid())) {
            log.info("用户首次参与热文分享：{}", webTokenVerifyVO.getOpenid());
            HotPostReadDataDTO hotPostReadDataDTO = HotPostReadDataDTO.builder()
                    .openid(webTokenVerifyVO.getOpenid())
                    .postId(param.getPostId())
                    .reportDate(new Date())
                    .shareUserId(param.getShareUserId())
                    .ip(WebUtils.getIpAddr(request))
                    .userAgent(request.getHeader(USER_AGENT_HEADER_KEY))
                    .areaCode(param.getAreaCode())
                    .build();
            hotPostJoinMsgSenderService.send(hotPostReadDataDTO);
        }

        return ReadReportVO.builder().webTokenExpired(false).build();
    }

    /**
     * 获取不带连字符的日期字符串
     *
     * @param now 当前日期
     * @return 日期字符串
     */
    private String getSimpleDateNoSeparatorStr(Date now) {
        return DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(now);
    }

    /**
     * 检查用户今天是否参与过活动
     *
     * @param openid openid
     * @return 是否参与过活动
     */
    private boolean checkUserTodayJoin(String openid) {
        return redisSetAdapter.add(buildJoinSharePostUserKey(getSimpleDateNoSeparatorStr(new Date())), openid) > 0;
    }


    /**
     * 检查token是否是过期的
     *
     * @param request 热文分享上下文
     * @return true过期，false未过期
     */
    public WebTokenVerifyVO checkExpiredToken(HttpServletRequest request) {
        //校验token是否合法，如果不合法则不记录
        String webToken = request.getHeader(WEB_TOKEN_HEADER_KEY);
        if (isBlank(webToken)) {
            return WebTokenVerifyVO.builder().expired(true).build();
        }

        WebTokenVerifyVO verifyResultVO = webTokenIntegrationService.verifyWebToken(webToken);
        if (Objects.isNull(verifyResultVO)) {
            return WebTokenVerifyVO.builder().expired(true).build();
        }
        return verifyResultVO;
    }


    @Override
    public List<HotPostShareCashDTO> getCashInfo() {
        return forumPostShareCashCounterMapper.getCashInfo();
    }


    @Override
    public HotPostShareCashDTO getUserCashInfo(Long userId) {

        HotPostShareCashDTO userCashInfo = forumPostShareCashCounterMapper.getUserCashInfo(userId);
        if (Objects.isNull(userCashInfo)) {
            HotPostShareCashDTO emptyShareCashDTO = new HotPostShareCashDTO();
            emptyShareCashDTO.setUserId(userId);
            emptyShareCashDTO.setPostShareCash(BigDecimal.ZERO);

            return emptyShareCashDTO;
        }

        return userCashInfo;
    }

    @Override
    public void reportJoinActivity(ActivityPostReportParam param) {
        //保存到set缓存中
        redisSetAdapter.add(buildJoinActivityUserKey(param.getPostId()), param.getUserId());
    }

    /**
     * 构建参加活动帖子缓存key
     *
     * @param postId 帖子id
     * @return 缓存key
     */
    private KeyGenerator buildJoinActivityUserKey(Long postId) {
        return RedisConfig.ACTIVITY_POST_JOIN_USER_KEY.copy().appendKey(postId);
    }

    /**
     * 构建每日用户参与阅读热文帖子key
     *
     * @param simpleDateStr 当前日期字符串
     * @return redis key
     */
    private KeyGenerator buildJoinSharePostUserKey(String simpleDateStr) {
        return RedisConfig.TODAY_JOIN_SHARE_POST_USER_KEY.copy().appendKey(simpleDateStr);
    }

    /**
     * 构建热文帖子随机初始已发奖励金额
     *
     * @return redis key
     */
    private KeyGenerator buildHotPostRandomAwardKey() {
        return RedisConfig.HOT_POST_INIT_RANDOM_AWARD_KEY.copy();
    }

    /**
     * 获取当前日期字符串，格式：2021-01-02
     *
     * @return 当前日期字符串
     */
    private String getSimpleDateStr(Date now) {
        return DateUtils.DATE_FORMAT_THREAD_LOCAL.get().format(now);
    }
}