package com.bxm.localnews.news.hotpost.impl;

import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.activity.ForumPostActivityMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashInfoMapper;
import com.bxm.localnews.news.hotpost.ShareCashPostService;
import com.bxm.localnews.news.model.dto.hotpost.ShareCashPostItemDTO;
import com.bxm.localnews.news.model.entity.activity.ForumPostActivityEntity;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashInfoEntity;
import com.bxm.localnews.news.model.param.hotpost.ShareCashPostAwardParam;
import com.bxm.localnews.news.model.param.hotpost.ShareCashPostOrderParam;
import com.bxm.localnews.news.model.param.hotpost.ShareCashPostPageParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @author liujia
 * @date 1/19/21 2:08 PM
 **/
@Service
@AllArgsConstructor
@Slf4j
public class ShareCashPostServiceImpl implements ShareCashPostService {

    private ForumPostShareCashInfoMapper forumPostShareCashInfoMapper;

    private ForumPostActivityMapper postActivityMapper;

    private SequenceCreater sequenceCreater;

    private ForumProperties forumProperties;

    private LocationIntegrationService locationIntegrationService;

    private RedisStringAdapter redisStringAdapter;

    @Override
    public Message changeOrder(ShareCashPostOrderParam param) {
        Preconditions.checkArgument(param != null);

        ForumPostShareCashInfoEntity updateEntity = new ForumPostShareCashInfoEntity();
        updateEntity.setPostId(param.getId());
        updateEntity.setModifyTime(new Date());
        updateEntity.setOrder(param.getOrder());
        return Message.build(forumPostShareCashInfoMapper.updateByPrimaryKeySelective(updateEntity));
    }

    @Override
    public Message changeStatus(Long postId, Integer status) {
        ForumPostShareCashInfoEntity existsPost = getDetail(postId);
        ForumPostActivityEntity existActivityPost = postActivityMapper.selectByPostId(postId);

        if (Objects.equals(status, 1)) {
            if (existsPost.getAmount().compareTo(existsPost.getUsedAmount()) <= 0) {
                return Message.build(false, "剩余发放金额不足，无法重启活动");
            }

            //如果是一个活动帖子并且处于下架状态，则热文无法恢复为有效状态
            if (Objects.nonNull(existActivityPost) && Objects.equals(existActivityPost.getStatus(), 0)) {
                return Message.build(false, "活动帖子为下架状态，即使在有效期也无法将热文上架");
            }

            if (null != existsPost.getAwardEndTime() && DateUtils.before(existsPost.getAwardEndTime(), new Date())) {
                return Message.build(false, "活动已过了截止有效期，无法开启");
            }
        }

        ForumPostShareCashInfoEntity updateEntity = new ForumPostShareCashInfoEntity();
        updateEntity.setPostId(postId);
        updateEntity.setModifyTime(new Date());
        updateEntity.setStatus(status);
        return Message.build(forumPostShareCashInfoMapper.updateByPrimaryKeySelective(updateEntity));
    }

    @Override
    public Message changeAwardTime(ShareCashPostAwardParam param) {
        Preconditions.checkArgument(param != null);

        Date awardStartTime = param.getAwardStartTime();
        Date awardEndTime = param.getAwardEndTime();
        if (Objects.nonNull(awardStartTime) && Objects.nonNull(awardEndTime)) {
            if (DateUtils.after(awardStartTime, awardEndTime)) {
                return Message.build(false, "热文有效开始时间不能晚于结束时间");
            }
        }

        ForumPostShareCashInfoEntity updateEntity = new ForumPostShareCashInfoEntity();

        updateEntity.setPostId(param.getId());
        updateEntity.setModifyTime(new Date());
        updateEntity.setAwardStartTime(param.getAwardStartTime());
        updateEntity.setAwardEndTime(param.getAwardEndTime());
        //如果修改的过期时间早于现在，则直接设置为过期状态
        if (Objects.nonNull(awardEndTime) && DateUtils.before(awardEndTime, new Date())) {
            updateEntity.setStatus(0);
        }
        //如果设置的开始时间早于现在，则先设置过期
        if (DateUtils.after(awardStartTime, new Date())) {
            updateEntity.setStatus(0);
        }

        return Message.build(forumPostShareCashInfoMapper.updateAwardTime(updateEntity));
    }

    @Override
    public ForumPostShareCashInfoEntity getDetail(Long postId) {
        return forumPostShareCashInfoMapper.getByPostId(postId);
    }

    @Override
    public PageWarper<ShareCashPostItemDTO> queryByPage(ShareCashPostPageParam param) {
        List<ShareCashPostItemDTO> shareCashPosts = forumPostShareCashInfoMapper.queryByPage(param);

        for (ShareCashPostItemDTO post : shareCashPosts) {
            //设置地区
            if (StringUtils.isNotEmpty(post.getAreaDetail())) {
                String[] areaCodeArray = post.getAreaDetail().split(",");
                post.setAreaDetail(locationIntegrationService.batchGetDetailJson(areaCodeArray));
            }

            if (Objects.equals(post.getPostType(), "1")) {
                post.setPostType("活动帖子");
            } else {
                post.setPostType("社区帖子");
            }
        }

        return new PageWarper<>(shareCashPosts);
    }

    @Override
    public Message save(ForumPostShareCashInfoEntity entity) {
        int result = 0;

        if (null == entity.getId()) {
            entity.setId(sequenceCreater.nextLongId());
            entity.setCreateTime(new Date());
            entity.setOrder(0);
            entity.setStatus(Objects.isNull(entity.getStatus()) ? 1 : entity.getStatus());
            entity.setUsedAmount(BigDecimal.ZERO);
            entity.setAward(forumProperties.getShareCashAward());

            result = forumPostShareCashInfoMapper.insert(entity);

            redisStringAdapter.set(buildPostAwardKey(entity.getPostId()), entity.getAmount());
        } else {
            ForumPostShareCashInfoEntity beforeInfo = forumPostShareCashInfoMapper.selectByPrimaryKey(entity.getId());

            log.info("热文奖励金额：更新前的值: {}, 更新后的值：{}", beforeInfo.getAmount(), entity.getAmount());
            log.info("更新热文帖子状态：更新前的值: {}, 更新后的值：{}", beforeInfo.getStatus(), entity.getStatus());
            // 如果是修改,比较是增加还是减少
            BigDecimal changeAmount = entity.getAmount().subtract(beforeInfo.getAmount());

            //如果不存在，则去初始化当前奖金金额信息
            if (Boolean.FALSE.equals(redisStringAdapter.hasKey(buildPostAwardKey(entity.getPostId())))) {
                loadHotPostShareDb(entity.getPostId());
            }

            if (changeAmount.compareTo(BigDecimal.ZERO) >= 0) {
                //如果变化金额大于零，直接进行加操作
                redisStringAdapter.increment(buildPostAwardKey(entity.getPostId()), Math.abs(changeAmount.doubleValue()));

                entity.setModifyTime(new Date());
                result = forumPostShareCashInfoMapper.updateByPrimaryKeySelective(entity);
                log.info("热文奖励金额更新成功，增加或保持金额不变：更新前的值: {}, 更新后的值：{}", beforeInfo.getAmount(), entity.getAmount());
            }

            if (changeAmount.compareTo(BigDecimal.ZERO) < 0) {
                log.info("热文奖励金额更新成功，减少奖金金额：{}", Math.abs(changeAmount.doubleValue()));
                //如果修改的总金额小于已发金额则不进行金额的扣减
                if (entity.getAmount().compareTo(beforeInfo.getUsedAmount()) < 0) {
                    entity.setModifyTime(new Date());
                    entity.setAmount(null);
                    result = forumPostShareCashInfoMapper.updateByPrimaryKeySelective(entity);
                    log.warn("帖子[{}]的奖励金额[{}]比已发放金额[{}]少，不更新金额",
                            entity.getPostId(),
                            entity.getAmount(),
                            entity.getUsedAmount());
                    //直接返回
                    return Message.build(result);
                }
                //如果小于零，尝试去减金额
                if (redisStringAdapter.decrement(buildPostAwardKey(entity.getPostId()), Math.abs(changeAmount.doubleValue())) >= 0) {
                    //减成功，则去更新奖金金额
                    entity.setModifyTime(new Date());
                    result = forumPostShareCashInfoMapper.updateByPrimaryKeySelective(entity);
                    log.info("热文奖励金额更新成功，减少奖金金额：更新前的值: {}, 更新后的值：{}", beforeInfo.getAmount(), entity.getAmount());
                } else {
                    //如果减完奖金之后小于0，则将奖金数量加回来
                    redisStringAdapter.increment(buildPostAwardKey(entity.getPostId()), Math.abs(changeAmount.doubleValue()));

                    log.error("热文分享奖金金额不足，帖子id：{}", entity.getPostId());
                    return Message.build(false, "热文分享奖金金额不足");
                }
            }

        }

        return Message.build(result);
    }

    @Override
    public void setExpired() {
        Date phaseTime = DateUtils.addField(new Date(), Calendar.HOUR, -1);
        List<ForumPostShareCashInfoEntity> expiredPostList = forumPostShareCashInfoMapper.getActiveAndExpiredPost(phaseTime);

        for (ForumPostShareCashInfoEntity entity : expiredPostList) {
            changeStatus(entity.getPostId(), 0);
        }
    }

    @Override
    public void setStart() {
        List<ForumPostShareCashInfoEntity> startPostList = forumPostShareCashInfoMapper.getOfflineAndExpiredPost();

        for (ForumPostShareCashInfoEntity entity : startPostList) {
            changeStatus(entity.getPostId(), 1);
        }
    }

    /**
     * 构建热文奖励金额key
     *
     * @param postId 帖子id
     * @return redis key
     */
    private KeyGenerator buildPostAwardKey(Long postId) {
        return RedisConfig.SHARE_POST_AWARD_KEY.copy().appendKey(postId);
    }

    /**
     * 重新加载从DB加载奖金金额
     *
     * @param postId 帖子id
     */
    private void loadHotPostShareDb(Long postId) {
        ForumPostShareCashInfoEntity shareCashInfoEntity = forumPostShareCashInfoMapper.getByPostId(postId);
        redisStringAdapter.set(buildPostAwardKey(postId), shareCashInfoEntity.getAmount().subtract(shareCashInfoEntity.getUsedAmount()));
    }
}
