package com.bxm.localnews.news.list.filter;

import com.bxm.localnews.news.constant.OrderConstant;
import com.bxm.localnews.news.detail.context.ForumPostDetailContext;
import com.bxm.localnews.news.detail.context.PostDetailOriginalParam;
import com.bxm.localnews.news.detail.filter.*;
import com.bxm.localnews.news.detail.helper.ForumPostImageHelper;
import com.bxm.localnews.news.enums.DisplayAreaEnum;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.model.param.ForumPostFillContext;
import com.bxm.newidea.component.annotations.FilterBean;
import com.bxm.newidea.component.filter.FilterChainExecutor;
import com.bxm.newidea.component.filter.IFilter;
import org.apache.commons.collections.CollectionUtils;

import javax.annotation.Resource;

import static com.bxm.localnews.news.constant.LogicGroupConstant.POST_DETAIL_FILTER;
import static com.bxm.localnews.news.constant.LogicGroupConstant.POST_LIST_FILTER;

/**
 * 单个帖子的额外信息填充，复用帖子详情的部分处理逻辑
 *
 * @author liujia
 * @date 1/14/21 5:14 PM
 **/
@FilterBean(group = POST_LIST_FILTER)
public class PostInfoFillFilter implements IFilter<ForumPostFillContext> {

    @Resource
    private FilterChainExecutor filterChainExecutor;

    @Resource
    private ForumPostImageHelper forumPostImageHelper;

    @Override
    public void doFilter(ForumPostFillContext context) {
        context.getData().forEach(forumPostVo -> {
            forumPostVo.setPostContentImgList(forumPostVo.getPostImgList());

            PostDetailOriginalParam param = new PostDetailOriginalParam(forumPostVo.getId());
            param.setUserId(context.getUserId());
            param.setAreaCode(context.getAreaCode());

            ForumPostDetailContext detailContext = new ForumPostDetailContext(param);
            detailContext.setPostInfo(forumPostVo);
            detailContext.setLocation(context.getLocation());

            // 指定执行的逻辑
            detailContext.choiceFilter(BaseInfoFillFilter.class)
                    .choiceFilter(ImageFillFilter.class)
                    .choiceFilter(TopicFillFilter.class)
                    .choiceFilter(RelationFillFilter.class)
                    .choiceFilter(CommentCalFilter.class)
                    .choiceFilter(TagSwitchFilter.class)
                    .choiceFilter(PlaceHolderFilter.class);

            // 如果需要显示热评，则添加热评，默认不添加
            if (context.isLoadHotReplay()) {
                detailContext.choiceFilter(HotReplayFillFilter.class);
            }

            filterChainExecutor.doFilter(POST_DETAIL_FILTER, detailContext);

            //是否在没有标题的情况下用内容补充标题
            if (context.isFillTitle()) {
                forumPostVo.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
            }

            if (DisplayAreaEnum.OTHER.equals(context.getDisplayArea())) {
                //本地圈的封面图设置
                forumPostImageHelper.exchangeDetailPost(forumPostVo);
            } else if (DisplayAreaEnum.INDEX.equals(context.getDisplayArea())) {
                //首页的封面图设置
                forumPostImageHelper.exchangeCoverPost(forumPostVo);
            } else if (DisplayAreaEnum.IM.equals(context.getDisplayArea())) {
                //IM 内分享图设置
                forumPostImageHelper.exchangeCoverPost(forumPostVo);
            }

            //填充剩余评论数
            forumPostVo.setLeftRelyNum(CollectionUtils.isNotEmpty(forumPostVo.getReplyInfo())
                    ? forumPostVo.getCommentCount() - forumPostVo.getReplyInfo().size()
                    : forumPostVo.getCommentCount());
        });
    }

    @Override
    public int getOrder() {
        return OrderConstant.getOrder(this.getClass());
    }
}
