package com.bxm.localnews.news.list.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.enums.DisplayAreaEnum;
import com.bxm.localnews.news.list.PostListService;
import com.bxm.localnews.news.model.param.FollowUserForumsParam;
import com.bxm.localnews.news.model.param.ForumPostFillContext;
import com.bxm.localnews.news.model.param.ForumPostListUserQueryParam;
import com.bxm.localnews.news.model.param.HomePagePostParam;
import com.bxm.localnews.news.model.vo.FollowUserForumPostVo;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.util.FormPostContentUtil;
import com.bxm.localnews.news.util.ListUtil;
import com.bxm.newidea.component.filter.FilterChainExecutor;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.LogicGroupConstant.POST_LIST_FILTER;
import static org.apache.commons.lang3.StringUtils.isBlank;


/**
 * @author liujia
 * @date 1/14/21 4:58 PM
 **/
@Slf4j
@Service
@AllArgsConstructor
public class PostListServiceImpl implements PostListService {

    private LocationIntegrationService locationIntegrationService;

    private FilterChainExecutor filterChainExecutor;

    private ForumPostMapper forumPostMapper;

    private RedisStringAdapter redisStringAdapter;

    private UserIntegrationService userIntegrationService;

    private ForumProperties forumProperties;

    @Override
    public void fillExtInfo(ForumPostFillContext context) {
        if (CollectionUtils.isEmpty(context.getData())) {
            return;
        }

        //填充地区
        if (StringUtils.isNotBlank(context.getAreaCode()) && null == context.getLocation()) {
            context.setLocation(locationIntegrationService.getLocationByGeocode(context.getAreaCode()));
        }

        filterChainExecutor.parallelDoFilter(POST_LIST_FILTER, context);
    }


    @Override
    public PageWarper<ForumPostVo> getHomePagePostList(HomePagePostParam param) {
        PageWarper<ForumPostVo> forumPostWarper = new PageWarper<>(forumPostMapper.listHomePagePostList(param));
        List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
        if (CollectionUtils.isNotEmpty(forumPostVoList)) {
            ForumPostFillContext context = ForumPostFillContext.builder()
                    .data(forumPostVoList)
                    .userId(param.getUserId())
                    .areaCode(param.getAreaCode())
                    .fillTitle(false)
                    .displayArea(DisplayAreaEnum.OTHER)
                    .build();
            fillExtInfo(context);
        }
        return forumPostWarper;
    }

    @Override
    public List<ForumPostVo> getRewardPostList() {
        String rewardList = redisStringAdapter.getString(RedisConfig.FORUM_POST_REWARD_LIST.copy());

        List<ForumPostVo> forumPostVoList;
        if (isBlank(rewardList)) {
            forumPostVoList = forumPostMapper.getRewardPostList(50);
        } else {
            forumPostVoList = JSON.parseArray(rewardList, ForumPostVo.class);
        }

        if (CollectionUtils.isNotEmpty(forumPostVoList)) {
            forumPostVoList.forEach(e -> FormPostContentUtil.replace(e, null));
        }

        return forumPostVoList;
    }

    @Override
    public List<ForumPostVo> getPostListByIds(String postIds) {
        List<Long> postIdList = ListUtil.convertStringToList(postIds);
        if (CollectionUtils.isEmpty(postIdList)) {
            return null;
        }

        List<ForumPostVo> forumPostVoList = forumPostMapper.listPostByIds(postIdList);
        if (CollectionUtils.isNotEmpty(forumPostVoList)) {
            ForumPostFillContext context = ForumPostFillContext.builder()
                    .data(forumPostVoList)
                    .userId(null)
                    .areaCode(null)
                    .fillTitle(false)
                    .displayArea(DisplayAreaEnum.OTHER)
                    .build();
            fillExtInfo(context);
        }

        return forumPostVoList;
    }

    @Override
    public FollowUserForumPostVo getFollowedUserForumList(FollowUserForumsParam param) {
        if (Objects.isNull(param.getUserId())) {
            return new FollowUserForumPostVo();
        }

        FollowUserForumPostVo result = new FollowUserForumPostVo();

        //获取用户的关注列表
        List<Long> followedUserIds = userIntegrationService.getUserFollowedUserIds(param.getUserId());

        if (CollectionUtils.isEmpty(followedUserIds)) {
            return new FollowUserForumPostVo();
        }
        //缓存中存有负数，需要剔除
        followedUserIds = followedUserIds.stream().filter(s -> s > 0).collect(Collectors.toList());
        param.setFollowedUserIds(CollectionUtils.isNotEmpty(followedUserIds) ? followedUserIds : Lists.newArrayList(0L));

        PageWarper<ForumPostVo> pageWarper = new PageWarper<>(forumPostMapper.getForumListByFollowedUserIds(param));

        //组装帖子内容详情
        ForumPostFillContext context = ForumPostFillContext.builder()
                .data(pageWarper.getList())
                .userId(param.getUserId())
                .fillTitle(false)
                .loadHotReplay(true)
                .displayArea(DisplayAreaEnum.OTHER)
                .build();
        fillExtInfo(context);


        result.setShowFollowGuide(followedUserIds.size() < forumProperties.getGuideShowFollowedNum() ? 1 : 0);
        result.setFollowGuideDesc(forumProperties.getGuideToFollowedText());
        result.setList(pageWarper);

        return result;
    }


    @Override
    public PageWarper<ForumPostVo> listForumPostByUser(ForumPostListUserQueryParam param) {
        PageWarper<ForumPostVo> forumPostWarper = new PageWarper<>();

        if (null == param || null == param.getType()) {
            return null;
        }
        //我的发布
        if (1 == param.getType()) {
            forumPostWarper = new PageWarper<>(forumPostMapper.listPostByIdsInUser(param));
            List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
            if (CollectionUtils.isNotEmpty(forumPostVoList)) {
                ForumPostFillContext context = ForumPostFillContext.builder()
                        .data(forumPostVoList)
                        .userId(param.getUserId())
                        .areaCode(null)
                        .fillTitle(false)
                        .displayArea(DisplayAreaEnum.OTHER)
                        .build();
                fillExtInfo(context);
            }
            //我的收藏（旧版本）
        } else if (2 == param.getType()) {
            forumPostWarper = new PageWarper<>(forumPostMapper.getCollectPostList(param));
            List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
            if (CollectionUtils.isNotEmpty(forumPostVoList)) {
                ForumPostFillContext context = ForumPostFillContext.builder()
                        .data(forumPostVoList)
                        .userId(param.getUserId())
                        .areaCode(null)
                        .fillTitle(true)
                        .displayArea(DisplayAreaEnum.OTHER)
                        .build();
                fillExtInfo(context);
            }
        } else if (3 == param.getType()) {
            //IM内人分享调用
            HomePagePostParam postParam = new HomePagePostParam();
            postParam.setTargetUserId(param.getUserId());
            postParam.setUserId(param.getUserId());
            BeanUtils.copyProperties(param, postParam);
            forumPostWarper = new PageWarper<>(forumPostMapper.listHomePagePostList(postParam));
            List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
            if (CollectionUtils.isNotEmpty(forumPostVoList)) {
                ForumPostFillContext context = ForumPostFillContext.builder()
                        .data(forumPostVoList)
                        .userId(param.getUserId())
                        .areaCode(null)
                        .fillTitle(false)
                        .displayArea(DisplayAreaEnum.IM)
                        .build();
                fillExtInfo(context);
            }
        }
        return forumPostWarper;
    }
}
