package com.bxm.localnews.news.note.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.enums.UrlCategoryEnum;
import com.bxm.localnews.news.enums.UrlPositionEnum;
import com.bxm.localnews.news.enums.UrlTypeEnum;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.NotePublishVo;
import com.bxm.localnews.news.model.vo.NoteVo;
import com.bxm.localnews.news.note.NoteService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class NoteServiceImpl extends BaseService implements NoteService {

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private MixedRecommendPoolMapper mixedRecommendPoolMapper;

    @Resource
    private ForumProperties forumProperties;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private IUrlFactory iUrlFactory;

    @Override
    public Message delete(Long noteId) {
        Long userId = forumPostMapper.getUserIdByPostId(noteId);
        ForumPostVo forumPostVo = new ForumPostVo();
        forumPostVo.setId(noteId);
        forumPostVo.setStatus((int) PostStatusEnum.USER_DELETE.getCode());
        //更新帖子信息表
        forumPostMapper.updateByPrimaryKeySelective(forumPostVo);
        //删除内容推荐库的内容
        mixedRecommendPoolMapper.deleteMixRecommandPoolById(noteId);
        //更新用户发布数
        userIntegrationService.addPostReplyNum(userId, 2);
        //清除用户小纸条缓存
        removeCache(userId);
        return Message.build(true);
    }

    @Override
    public NotePublishVo getNotePublish(Long userId) {
        NotePublishVo notePublishVo = new NotePublishVo();
        notePublishVo.setPublishNum(getPublishNum(userId));
        // 引导文案
        notePublishVo.setGuidance(forumProperties.getNoteGuidance());
        //获取小纸条轮播列表
        notePublishVo.setCarouselList(forumPostMapper.getCarouselNoteList());

        return notePublishVo;
    }

    @Override
    public List<NoteVo> batchGetUserNote(List<Long> userIdList, Long currentUserId) {
        List<NoteVo> userNoteVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userNoteVoList;
        }

        boolean hasCurrent = false;
        if (currentUserId != null && userIdList.contains(currentUserId)) {
            userIdList.remove(currentUserId);
            hasCurrent = true;
        }

        if (CollectionUtils.isNotEmpty(userIdList)) {
            //先取缓存
            Set<String> userIds = userIdList.stream().map(Object::toString).collect(Collectors.toSet());
            List<String> userNoteCache = redisHashMapAdapter.multiGet(RedisConfig.USER_FORUM_NOTE, userIds, String.class);
            log.debug("从缓存中获取用户小纸条数据：" + JSON.toJSONString(userNoteCache));

            //处理缓存数据
            if (CollectionUtils.isNotEmpty(userNoteCache)) {
                userNoteCache.forEach(userCache -> {
                    List<NoteVo> noteVoList = JSON.parseArray(userCache, NoteVo.class);
                    if (CollectionUtils.isNotEmpty(noteVoList)) {
                        userIdList.remove(noteVoList.get(0).getUserId());
                        userNoteVoList.add(noteVoList.get(0));
                    }
                });
            }

            //若缓存中无法取到所有用户小纸条信息
            if (CollectionUtils.isNotEmpty(userIdList)) {
                userNoteVoList.addAll(batchLoadUserNoteVoToRedis(userIdList, true));
            }
        }

        //当前用户需返回仅自己可见内容
        if (hasCurrent) {
            List<NoteVo> noteVoList = forumPostMapper.batchGetUserNotes(Collections.singletonList(currentUserId), 1);
            if (CollectionUtils.isNotEmpty(noteVoList)) {
                userNoteVoList.add(noteVoList.get(0));
            }
        }

        userNoteVoList.forEach(userNote -> this.completeNoteInfo(userNote, currentUserId));

        log.debug("返回用户小纸条：" + JSON.toJSONString(userNoteVoList));

        return userNoteVoList;
    }

    @Override
    public void cacheUserNote(Date startTime) {
        //获取有更新的帖子用户
        List<Long> userIdList = forumPostMapper.getUpdatedUser(startTime);
        //缓存数据
        batchLoadUserNoteVoToRedis(userIdList, true);
    }

    @Override
    public void removeCache(Long userId) {
        if (null != userId) {
            redisHashMapAdapter.remove(RedisConfig.USER_FORUM_NOTE, userId.toString());
        }
    }

    @Override
    public List<NoteVo> getUserNoteList(Long userId) {
        if (null == userId) {
            return new ArrayList<>();
        }

        //先取缓存
        String userNoteCache = redisHashMapAdapter.get(RedisConfig.USER_FORUM_NOTE, userId.toString(), String.class);

        //处理缓存数据
        if (StringUtils.isNotBlank(userNoteCache)) {
            List<NoteVo> noteVoList = JSON.parseArray(userNoteCache, NoteVo.class);
            if (CollectionUtils.isNotEmpty(noteVoList)) {
                return noteVoList;
            }
        }

        //若缓存中无法取到用户图片信息
        return batchLoadUserNoteVoToRedis(Collections.singletonList(userId), false);
    }

    private List<NoteVo> batchLoadUserNoteVoToRedis(List<Long> userIdList, boolean getFirst) {
        List<NoteVo> resultList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return resultList;
        }

        //获取用户小纸条
        List<NoteVo> noteList = forumPostMapper.batchGetUserNotes(userIdList, 0);
        if (CollectionUtils.isEmpty(noteList)) {
            return resultList;
        }

        //映射用户小纸条
        Map<Long, List<NoteVo>> userNoteMap = noteList.stream().collect(Collectors.groupingBy(NoteVo::getUserId));
        Map<String, String> userNoteStrMap = new HashMap<>();
        userIdList.forEach(userId -> {
            List<NoteVo> noteVoList = userNoteMap.get(userId);
            //即使为空，也要覆盖缓存
            userNoteStrMap.put(userId.toString(), JSON.toJSONString(noteVoList));

            //根据是否每个用户一条获取结果
            if (CollectionUtils.isNotEmpty(noteVoList)) {
                if (getFirst) {
                    resultList.add(noteVoList.get(0));
                } else {
                    resultList.addAll(noteVoList);
                }
            }
        });

        //存入缓存
        if (userNoteMap.size() > 0) {
            redisHashMapAdapter.putAll(RedisConfig.USER_FORUM_NOTE, userNoteStrMap);
        }

        return resultList;
    }

    @Override
    public void completeNoteInfo(NoteVo noteVo, Long userId) {
        //填充分享地址
        noteVo.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.FORUM, UrlTypeEnum.SHARE, noteVo.getId(), userId));
        //填充小程序分享地址
        noteVo.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.POST, noteVo.getId(), userId));
        //填充分享图片
        noteVo.setShareImg(noteVo.getUserImg());
    }

    private Integer getPublishNum(Long userId) {
        KeyGenerator key = getUserNotePublishKey(userId);
        if (redisStringAdapter.hasKey(key)) {
            return redisStringAdapter.getInt(key);
        } else {
            //初始化
            Integer publishNum = forumProperties.getUserPublishNum();
            redisStringAdapter.set(key, publishNum);
            redisStringAdapter.expire(key, DateUtils.getCurSeconds());
            return publishNum;
        }
    }

    private KeyGenerator getUserNotePublishKey(Long userId) {
        return RedisConfig.USER_NOTE_PUBLISH.copy().appendKey(userId);
    }
}
