package com.bxm.localnews.news.post.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.dto.MerchantMemberUserDTO;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.domain.AdminEditorMessageMapper;
import com.bxm.localnews.news.domain.AdminForumPostMapper;
import com.bxm.localnews.news.domain.AdminMixedRecommendPoolMapper;
import com.bxm.localnews.news.domain.AdminTopicMapper;
import com.bxm.localnews.news.domain.activity.ForumPostActivityMapper;
import com.bxm.localnews.news.domain.activity.ForumPostShareCashInfoMapper;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.enums.PostTypeEnum;
import com.bxm.localnews.news.model.dto.AdminGetPostUrlDTO;
import com.bxm.localnews.news.model.dto.VirtualUserOverviewDTO;
import com.bxm.localnews.news.model.entity.activity.ForumPostActivityEntity;
import com.bxm.localnews.news.model.entity.activity.ForumPostShareCashInfoEntity;
import com.bxm.localnews.news.model.param.AdminForumPostParam;
import com.bxm.localnews.news.model.param.VirtualUserQueryParam;
import com.bxm.localnews.news.model.vo.*;
import com.bxm.localnews.news.note.NoteService;
import com.bxm.localnews.news.post.AdminForumPostService;
import com.bxm.localnews.news.post.PostTagService;
import com.bxm.localnews.news.topic.AdminTopicService;
import com.bxm.localnews.news.util.DealContentUtil;
import com.bxm.localnews.news.util.ImportUtils;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.localnews.url.ProtocolFactory;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.*;
import java.util.stream.Collectors;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
@Slf4j
@AllArgsConstructor
public class AdminForumPostServiceImpl extends BaseService implements AdminForumPostService {

    private AdminTopicMapper adminTopicMapper;

    private AdminForumPostMapper adminForumPostMapper;

    private AdminEditorMessageMapper adminEditorMessageMapper;

    private AdminMixedRecommendPoolMapper adminMixedRecommendPoolMapper;

    private LocationIntegrationService locationService;

    private AdminTopicService adminTopicService;

    private NewsSeqComponent newsSeqComponent;

    private ForumProperties forumProperties;

    private UserAccountIntegrationService userAccountIntegrationService;

    private DealContentUtil dealContentUtil;

    private UserIntegrationService userIntegrationService;

    private VirtualUserIntegrationService virtualUserIntegrationService;

    private DomainIntegrationService domainIntegrationService;

    private ShortLinkIntegrationService shortLinkIntegrationService;

    private ForumPostShareCashInfoMapper forumPostShareCashInfoMapper;

    private ForumPostActivityMapper forumPostActivityMapper;

    private MerchantMemberIntegrationService merchantMemberIntegrationService;

    private PostTagService postTagService;

    private NoteService noteService;

    @Override
    public PageWarper<AdminBaseForumPost> getApproveList(AdminForumPostParam param) {
        PageWarper<AdminBaseForumPost> forumList = new PageWarper<>(adminForumPostMapper.getList(param));

        if (CollectionUtils.isNotEmpty(forumList.getList())) {
            forumList.getList().parallelStream().forEach(this::completePostInfo);
        }

        return forumList;
    }

    @Override
    public List<AdminBaseForumPostExcelVO> getApproveListExcelVO(AdminForumPostParam param) {
        PageWarper<AdminBaseForumPost> approveList = getApproveList(param);

        return approveList.getList().stream().map(this::convert).collect(Collectors.toList());

    }

    private AdminBaseForumPostExcelVO convert(AdminBaseForumPost post) {
        byte yes = 1;
        AdminBaseForumPostExcelVO excelVO = new AdminBaseForumPostExcelVO();
        BeanUtils.copyProperties(post, excelVO);

        excelVO.setId(Objects.toString(post.getId()));
        excelVO.setTitle(isBlank(post.getTitle()) ? post.getTextField() : post.getTitle());
        excelVO.setUserId(Objects.toString(post.getUserId()));
        excelVO.setIsBrilliant(Objects.equals(post.getIsBrilliant(), yes) ? "是" : "否");
        excelVO.setIsRecommend(Objects.equals(post.getIsRecommend(), yes) ? "是" : "否");
        excelVO.setIsRed(Objects.equals(post.getIsRed(), yes) ? "是" : "否");
        excelVO.setIsCash(Objects.equals(post.getIsCash(), yes) ? "是" : "否");
        excelVO.setCashReward(isNull(post.getCashReward()) ? "0" : Objects.toString(post.getCashReward()));
        excelVO.setIsBroke(Objects.equals(post.getIsBroke(), yes) ? "是" : "否");
        excelVO.setCreateTime(DateUtils.formatDateTime(post.getCreateTime()));
        excelVO.setPublishTime(DateUtils.formatDateTime(post.getPublishTime()));
        excelVO.setDisplayTime(DateUtils.formatDateTime(post.getDisplayTime()));
        excelVO.setDeliveryType(Objects.equals(post.getDeliveryType(), 0) ? "全国" : "地方");
        excelVO.setPostType(Objects.equals(post.getPostType(), yes) ? "普通帖子" : "小纸条");
        excelVO.setValidDate(DateUtils.formatDateTime(post.getValidDate()));
        excelVO.setTopicIdList(Objects.toString(post.getTopicList().stream().map(AdminTopic::getTitle).collect(Collectors.toList())));
        excelVO.setModifyTime(DateUtils.formatDateTime(post.getModifyTime()));
        excelVO.setIdentity(Objects.equals(post.getIdentity(), yes) ? "真实用户" : "马甲号");
        excelVO.setIsValid(Objects.equals(post.getIsValid(), Boolean.TRUE) ? "未过期" : "过期");

        return excelVO;
    }

    private String status(Integer status) {
        PostStatusEnum statusEnum = PostStatusEnum.getByCode(status);
        if (null == statusEnum) {
            return "无";
        }

        return statusEnum.getName();
    }

    @Override
    public AdminForumPost selectByPrimaryKey(Long id) {
        AdminForumPost adminForumPost = adminForumPostMapper.selectByPrimaryKey(id);

        if (null == adminForumPost) {
            return null;
        }

        //填充通用属性
        this.completePostInfo(adminForumPost);

        //设置编辑寄语
        if (adminForumPost.getEditorMessageId() != null) {
            AdminEditorMessage adminEditorMessage = adminEditorMessageMapper.selectByPrimaryKey(adminForumPost.getEditorMessageId());
            if (adminEditorMessage != null) {
                adminForumPost.setEditorMessageName(adminEditorMessage.getTemplateName());
            }
        }

        //设置马甲号
        VirtualUserQueryParam virtualUserQueryParam = new VirtualUserQueryParam();
        virtualUserQueryParam.setId(adminForumPost.getUserId());
        List<VirtualUserOverviewDTO> virtualUserList = virtualUserIntegrationService.list(virtualUserQueryParam);
        if (CollectionUtils.isNotEmpty(virtualUserList)) {
            adminForumPost.setUserInfo(virtualUserList.get(0).getUserInfo());
        }

        //设置图片
        List<PostImg> postImgList = new ArrayList<>();
        if (isNotBlank(adminForumPost.getImgList())) {
            postImgList = JSONArray.parseArray(adminForumPost.getImgList(), PostImg.class);
        }

        //如果没有封面，设置封面为内容中的图片
        if (StringUtils.isBlank(adminForumPost.getCoverList())) {
            if (CollectionUtils.isNotEmpty(postImgList)) {
                List<String> imgList = postImgList.stream()
                        .filter(e -> StringUtils.equals(e.getType(), "IMG"))
                        .map(PostImg::getImgUrl).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(imgList)) {
                    adminForumPost.setCoverList(JSONObject.toJSONString(imgList));
                }
            }
        }

        //设置视频封面
        if (CollectionUtils.isNotEmpty(postImgList)) {
            List<String> videoList = postImgList.stream()
                    .filter(e -> StringUtils.equals(e.getType(), "VIDEO"))
                    .map(PostImg::getImgUrl).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(videoList)) {
                adminForumPost.setVideoPosterList(JSONObject.toJSONString(videoList));
            }
        }

        //填充内容
        if (isNotBlank(adminForumPost.getContent())) {
            adminForumPost.setContent(dealContentUtil.fixContent(adminForumPost.getContent()));
        }

        //填充活动帖子信息
        if (Objects.equals(adminForumPost.getActivityPost(), 1)) {
            ForumPostActivityEntity forumPostActivityEntity = forumPostActivityMapper.selectByPostId(adminForumPost.getId());
            if (Objects.nonNull(forumPostActivityEntity)) {
                BeanUtils.copyProperties(forumPostActivityEntity, adminForumPost, "id");
                adminForumPost.setActivityStatus(forumPostActivityEntity.getStatus());

                //填充关联商家信息
                MerchantMemberUserDTO relationMerchantInfo = merchantMemberIntegrationService.getRelationMerchantInfo(adminForumPost.getUserId());
                if (Objects.nonNull(relationMerchantInfo)) {
                    adminForumPost.setMerchantId(relationMerchantInfo.getMerchantId());
                    adminForumPost.setMerchantName(relationMerchantInfo.getMerchantName());
                }
            }
        }

        //填充热文信息
        if (Objects.equals(adminForumPost.getIsShareCash(), 1)) {
            ForumPostShareCashInfoEntity shareCashInfoEntity = forumPostShareCashInfoMapper.getByPostId(adminForumPost.getId());
            if (Objects.nonNull(shareCashInfoEntity)) {
                BeanUtils.copyProperties(shareCashInfoEntity, adminForumPost, "id");
            }
        }

        return adminForumPost;
    }

    @Override
    public Boolean hasCashPost(String ids) {
        List<Long> idList = parseIds(ids);
        if (CollectionUtils.isEmpty(idList)) {
            return false;
        }

        for (Long id : idList) {
            AdminForumPost post = selectByPrimaryKey(id);
            if (Objects.equals(post.getIsCash(), (byte) 1)) {
                return true;
            }
        }

        return false;
    }

    private List<Long> parseIds(String ids) {
        if (StringUtils.isBlank(ids)) {
            return Lists.newArrayList();
        }

        return Arrays.stream(ids.split(",")).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
    }

    @Override
    public int deleteOrRecover(String ids, Integer status, Byte deductGold) {
        List<Long> idList = parseIds(ids);
        if (CollectionUtils.isEmpty(idList)) {
            return 0;
        }

        for (Long id : idList) {
            AdminForumPost adminForumPost = this.adminForumPostMapper.selectByPrimaryKey(id);
            if (adminForumPost == null) {
                return 0;
            }
            adminForumPost.setStatus(status);
            adminForumPost.setIsRecommend((byte) 0);
            this.adminForumPostMapper.updateByIdPartSelective(adminForumPost);

            if ((byte) 1 == status) {
                postTagService.savePostTag(adminForumPost);
            } else {
                // 用户的帖子数量 - 1 ，这里的方法名有歧义
                userIntegrationService.addPostReplyNum(adminForumPost.getUserId(), 2);

                if (null != deductGold && (byte) 1 == deductGold) {
                    userAccountIntegrationService.deductUserGold(adminForumPost.getId(), adminForumPost.getUserId());
                }
                //删除推荐库中的内容
                this.adminMixedRecommendPoolMapper.deleteByPrimaryKey(id);
                //删除用户小纸条缓存
                if (PostTypeEnum.NOTE.getCode() == adminForumPost.getPostType()) {
                    noteService.removeCache(adminForumPost.getUserId());
                }
            }
        }

        return 1;
    }

    @Override
    public Long getCreateId(Integer postType) {
        if (null == postType) {
            return newsSeqComponent.getPostId();
        }

        // 活动帖子
        if (PostTypeEnum.ACTIVITY.getCode() == postType) {
            return newsSeqComponent.getActivityPostId();
        }

        return newsSeqComponent.getPostId();
    }

    @Override
    public String getPostUrl(AdminGetPostUrlDTO adminGetPostUrlDTO) {
        //0-短链 1-长链接 2-站内链接
        if (Objects.equals(adminGetPostUrlDTO.getType(), 2)) {
            return ProtocolFactory.forumPost().app().postId(adminGetPostUrlDTO.getId()).build();
        }

        String areaCode = null;
        if (isNotBlank(adminGetPostUrlDTO.getAreaCode())) {
            areaCode = Splitter.on(",").split(adminGetPostUrlDTO.getAreaCode()).iterator().next();
        }

        String postUrl = ProtocolFactory.forumPost().outer()
                .userId(adminGetPostUrlDTO.getUserId())
                .postId(adminGetPostUrlDTO.getId())
                .channel(adminGetPostUrlDTO.getChannel())
                .domainViewScene(DomainScene.DomainViewScene.WX_JS_VIEW)
                .cityName(adminGetPostUrlDTO.getCityName())
                .areaCode(areaCode)
                .noExtend()
                .build();

        return shortLinkIntegrationService.generateShortUrl(postUrl);
    }

    @Override
    public void importNotes(Long userId, MultipartFile... multipartFiles) {
        if (null == multipartFiles) {
            return;
        }

        List<AdminForumPost> adminForumPostList = new ArrayList<>();
        List<Long> postIdList = new ArrayList<>();
        for (MultipartFile multipartFile : multipartFiles) {
            List<List<String>> resultList = ImportUtils.readExcelFile(multipartFile);
            if (CollectionUtils.isEmpty(resultList)) {
                continue;
            }
            logger.debug("解析结果：" + JSON.toJSONString(resultList));

            for (List<String> result : resultList) {
                // 构造forumPost
                AdminForumPost adminForumPost = getForumPost(userId, postIdList, result);
                if (adminForumPost == null) {
                    continue;
                }

                adminForumPostList.add(adminForumPost);
            }
        }

        logger.debug("小纸条列表" + JSON.toJSONString(adminForumPostList));

        //插入数据
        if (CollectionUtils.isNotEmpty(adminForumPostList)) {
            MybatisBatchBuilder.create(AdminForumPostMapper.class, adminForumPostList).run(AdminForumPostMapper::insertSelective);
            this.adminTopicService.batchUpdatePostTopic(forumProperties.getNoteTopicId(), postIdList);
            adminForumPostList.forEach(postTagService::resetForumPostTag);
        }
    }

    private AdminForumPost getForumPost(Long userId, List<Long> postIdList, List<String> result) {
        AdminForumPost adminForumPost = new AdminForumPost();
        Long id = getCreateId(null);
        postIdList.add(id);
        adminForumPost.setId(id);
        adminForumPost.setCreateTime(new Date());
        adminForumPost.setModifyTime(new Date());
        adminForumPost.setCreator(userId);
        adminForumPost.setStatus(PostStatusEnum.NORMAL.getCode());
        adminForumPost.setTopicIdList(Collections.singletonList(forumProperties.getNoteTopicId()));
        adminForumPost.setPostType(PostTypeEnum.NOTE.getCode());

        //获取用户
        String postUserId = result.get(0);
        if (StringUtils.isBlank(postUserId)) {
            return null;
        }
        adminForumPost.setUserId(Long.valueOf(postUserId));
        UserBean userBean = userIntegrationService.selectUserFromCache(adminForumPost.getUserId());
        if (null != userBean) {
            adminForumPost.setUserImg(userBean.getHeadImg());
            adminForumPost.setUserName(userBean.getNickname());
        }
        //获取文本
        String content = result.get(1);
        if (StringUtils.isBlank(content)) {
            return null;
        }
        adminForumPost.setTextField(content);
        adminForumPost.setContent(content);
        //获取定时发布时间
        String time = result.get(2);
        if (StringUtils.isBlank(time)) {
            return null;
        }
        adminForumPost.setPublishTime(DateUtils.parseDateTime(time));
        adminForumPost.setDisplayTime(adminForumPost.getPublishTime());
        //获取有效期
        String validDay = result.get(3);
        if (StringUtils.isBlank(validDay)) {
            return null;
        }
        adminForumPost.setValidDay(Integer.valueOf(validDay));
        adminForumPost.setValidDate(DateUtils.addField(adminForumPost.getDisplayTime(), Calendar.DAY_OF_MONTH, adminForumPost.getValidDay()));
        //获取地区
        adminForumPost.setDeliveryType(0);
        if (result.size() > 4) {
            adminForumPost.setAreaCode(result.get(4));
            if (isNotBlank(adminForumPost.getAreaCode())) {
                adminForumPost.setDeliveryType(1);
            }
        }
        return adminForumPost;
    }

    /**
     * 填充一些通用参数
     *
     * @param forumPost 帖子
     */
    private <T extends AdminBaseForumPost> void completePostInfo(T forumPost) {
        //设置地区
        if (StringUtils.isNotEmpty(forumPost.getAreaCode())) {
            String[] areaCodeArray = forumPost.getAreaCode().split(",");
            forumPost.setAreaDetail(locationService.batchGetDetailJson(areaCodeArray));
        }
        //设置话题
        forumPost.setTopicList(adminTopicMapper.getListByPostId(forumPost.getId()));

        UserBean userBean = userIntegrationService.selectUserFromCache(forumPost.getUserId());
        //设置用户状态
        forumPost.setUserState(userBean.getState() == null ? null : Integer.valueOf(userBean.getState()));
        forumPost.setNickName(userBean.getNickname());

        //设置帖子链接
        forumPost.setLinkUrl(ProtocolFactory.forumPost().outer()
                .noExtend()
                .userId(forumPost.getUserId())
                .postId(forumPost.getId())
                .build());

        //设置发帖人是否马甲号
        forumPost.setIdentity((byte) (Objects.equals(userBean.getState(), (byte) 3) ? 0 : 1));

        //设置投票组件
        restorePluginId(forumPost);

        //设置小纸条是否过期
        forumPost.setIsValid((byte) 2 == forumPost.getPostType() && new Date().before(forumPost.getValidDate()));

        // 设置是否是未到发布时间的预发布帖子
        forumPost.setIsPrePublish(0);
        if (nonNull(forumPost.getPublishTime())) {
            forumPost.setIsPrePublish(forumPost.getPublishTime().after(new Date()) ? 1 : 0);
        }

    }


    private <T extends AdminBaseForumPost> void restorePluginId(T post) {
        if (isNotBlank(post.getPlugins())) {
            JSONArray jsonArray = JSONArray.parseArray(post.getPlugins());
            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject json = jsonArray.getJSONObject(i);
                if ("VOTE".equals(json.get("type"))) {
                    post.setVotePluginId(json.getString("id"));
                }
            }
        }
    }


}
