package com.bxm.localnews.news.post.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.MessageService;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.news.content.ContentReplacer;
import com.bxm.localnews.news.domain.AdminForumPostMapper;
import com.bxm.localnews.news.model.constant.RedisCacheKey;
import com.bxm.localnews.news.model.dto.ContentContext;
import com.bxm.localnews.news.model.param.ForumPostUserAtParam;
import com.bxm.localnews.news.model.param.ReplaceTopicNeedParam;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.post.ForumPostContentService;
import com.bxm.localnews.news.statistics.ForumPostStatisticService;
import com.bxm.localnews.news.task.ForumPostUserAtTask;
import com.bxm.localnews.news.util.FormPostContentUtil;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.select.Elements;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.model.enums.ForumContentPlaceHolderEnum.*;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.*;

/**
 * 帖子内容处理
 *
 * @author gonzo
 * @date 2020-10-27 18:51
 **/
@Slf4j
@Service
@AllArgsConstructor
public class ForumPostContentServiceImpl implements ForumPostContentService {

    private static final String WST_USER_HREF = "a[" + USER_JUMP_URL.getPlaceHolder() + "]";

    private static final String POST_PROTOCOL = "wst://community/postDetail?postId=%s&tp=post";

    private final MessageService messageService;

    private final ContentReplacer contentReplacer;

    private final RedisStringAdapter redisStringAdapter;

    private final AdminForumPostMapper adminForumPostMapper;

    private final ScheduleService scheduleService;

    private final ForumPostUserAtTask forumPostUserAtTask;

    private final ForumPostStatisticService forumPostStatisticService;


    /**
     * 10天失效
     */
    private static final long EXPIRE = 60 * 60 * 24 * 10L;

    @Override
    public void afterCreate(AdminForumPost forumPostVo) {

        try {
            // 提取HTTP
            Document document = Jsoup.parse(forumPostVo.getContent());
            document.outputSettings().prettyPrint(true).indentAmount(1);

            // 判断是否是立即推送的帖子
            Date now = new Date();
            if (isNull(forumPostVo.getPublishTime())
                    || forumPostVo.getPublishTime().before(now)
                    || forumPostVo.getPublishTime().getTime() == now.getTime()) {

                // 是的话，立即推送
                processAtInfoAndPush(document, forumPostVo.getId(), Collections.emptyList());
            } else {
                // 否则不进行推送，等定时任务执行帖子发布的时候再推送
                ForumPostUserAtParam atParam = new ForumPostUserAtParam();
                atParam.setPostId(forumPostVo.getId());
                scheduleService.push(OnceTaskBuilder.builder(
                        ForumPostUserAtTask.generateTaskName(forumPostVo.getId()), forumPostVo.getPublishTime(), forumPostUserAtTask).callbackParam(atParam).build());
            }
        } catch (Exception e) {
            log.error("处理帖子内容失败, forumPostVo: {}", JSON.toJSONString(forumPostVo), e);
        }
    }

    @Override
    public void afterUp(ForumPostUserAtParam param) {

        AdminForumPost adminForumPost = adminForumPostMapper.selectByPrimaryKey(param.getPostId());
        if (Objects.nonNull(adminForumPost)) {
            // 提取HTTP
            Document document = Jsoup.parse(adminForumPost.getContent());
            document.outputSettings().prettyPrint(true).indentAmount(1);

            // 直接推送
            processAtInfoAndPush(document, adminForumPost.getId(), Collections.emptyList());
        }
    }

    @Override
    public void pushAtInfo(AdminForumPost forumPostVo) {

        try {
            //提取HTTP
            Document document = Jsoup.parse(forumPostVo.getContent());
            document.outputSettings().prettyPrint(true).indentAmount(1);

            // 判断是否是立即推送的帖子
            Date now = new Date();
            if (isNull(forumPostVo.getPublishTime())
                    || forumPostVo.getPublishTime().before(now)
                    || forumPostVo.getPublishTime().getTime() == now.getTime()) {

                // 是的话，立即推送 带上上一次的用户id做过滤
                List<Long> lastUserIds = Lists.newArrayList();
                String s = redisStringAdapter.get(getKey(forumPostVo.getId()), String.class);
                if (StringUtils.isNotBlank(s)) {
                    lastUserIds = JSON.parseArray(s, Long.class);
                }

                // 解析并推送
                processAtInfoAndPush(document, forumPostVo.getId(), lastUserIds);
            }
        } catch (Exception e) {
            log.error("处理帖子内容失败, forumPostVo: {}", JSON.toJSONString(forumPostVo), e);
        }
    }

    /**
     * 给用户发送推送
     *
     * @param userId 用户id
     * @param postId 帖子id
     */
    private void push(Long userId, Long postId) {
        // 给用户发送推送
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.JUMP_TYPE);
        // 跳转协议
        info.setProtocol(String.format(POST_PROTOCOL, postId));

        PushMessage pushMessage = PushMessage.build();
        pushMessage.setPayloadInfo(info);
        pushMessage.setTitle("你的好友@了你");
        pushMessage.setContent("你的好友@了你，快来看看");
        pushMessage.setPushReceiveScope(PushReceiveScope.pushSignle(userId));

        if (log.isDebugEnabled()) {
            log.debug("给用户: {} 发送@推送: {}", userId, JSON.toJSON(pushMessage));
        }

        // 发送推送
        messageService.pushMessage(pushMessage);
    }

    private KeyGenerator getKey(Long postId) {
        return RedisCacheKey.FORUM_AT_USER_IDS.copy().appendKey(Objects.toString(postId));
    }

    /**
     * 解析帖子内容里@到的用户，并进行推送
     *
     * @param document 帖子的html内容
     * @param postId   帖子id
     */
    private void processAtInfoAndPush(Document document, Long postId, List<Long> lastUserIds) {
        List<Long> userIds = processAtInfo(document);

        if (!CollectionUtils.isEmpty(userIds)) {
            // 缓存中记录最新的帖子的@的用户信息
            redisStringAdapter.set(getKey(postId), JSON.toJSONString(userIds), EXPIRE);

            // 如果是编辑，则过滤掉之前已经存在过的用户id再次进行推送
            if (!CollectionUtils.isEmpty(lastUserIds)) {
                userIds = userIds.stream().filter(p -> !lastUserIds.contains(p)).collect(Collectors.toList());
            }

            // 给用户推送
            userIds.forEach(p -> {
                push(p, postId);
            });
        }
    }

    /**
     * 给帖子内容里@到的用户进行推送
     *
     * @param document 帖子的html内容
     */
    private List<Long> processAtInfo(Document document) {

        // <a wst-user-href=>@用户</a>
        Elements select = document.select(WST_USER_HREF);
        if (Objects.nonNull(select) && !select.isEmpty()) {
            return select.stream().map(e -> {
                // 查询标签
                String wstUserHref = e.attr(USER_JUMP_URL.getPlaceHolder());
                if (isNotBlank(wstUserHref) && isNumeric(wstUserHref)) {
                    return Long.valueOf(wstUserHref);
                }

                return 0L;
            }).collect(Collectors.toList());
        }

        return Collections.emptyList();
    }

    @Override
    public void replacePlaceHolder(ForumPostVo forumPostVo, LocationDTO locationDTO, BasicParam basicParam, Long userId) {
        replacePostTitleOrField(forumPostVo, locationDTO);
        // 替换内容
        forumPostVo.setContent(replacePlaceHolder(forumPostVo.getContent(), locationDTO, basicParam, userId));

        forumPostVo.setEditorMessage(replacePlaceHolder(forumPostVo.getEditorMessage(), locationDTO, basicParam, userId));
    }

    @Override
    public void replacePostTitleOrField(ForumPostVo forumPostVo, LocationDTO locationDTO) {
        if (locationDTO == null) {
            locationDTO = new LocationDTO();
        }
        // 替换内容
        forumPostVo.setEnablePlaceholder((byte) 1);
        FormPostContentUtil.replace(forumPostVo, locationDTO);
    }


    @Override
    public String replacePlaceHolder(String content, LocationDTO locationDTO, BasicParam basicParam, Long userId) {

        if (isNull(locationDTO) || isBlank(content)) {
            return content;
        }

        // 构造替换时候需要的基础信息
        ReplaceTopicNeedParam param = new ReplaceTopicNeedParam();
        param.merge(basicParam);

        param.setUserId(userId);
        param.setAreaName(locationDTO.getName());
        param.setAreaCode(locationDTO.getCode());

        // 构造上下文对象
        ContentContext context = new ContentContext(content);
        // 替换城市名称
        context.addReplaceType(AREA_NAME, locationDTO::getName)
                .addReplaceType(AREA_NAME_2, locationDTO::getName)
                // 帖子跳转替换
                .addReplaceType(TOPIC_JUMP_URL, () -> param)
                // 用户跳转替换
                .addReplaceType(USER_JUMP_URL, () -> param)
                // 炫耀帖子替换
                .addReplaceType(LOTTERY_FLAUNT_JUMP_URL, () -> param);

        // 执行替换
        contentReplacer.replace(context);

        // 重新赋值
        return context.getContent();
    }
}
