package com.bxm.localnews.news.post.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.detail.ForumPostDetailService;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.dto.BaseForumPostDTO;
import com.bxm.localnews.news.dto.ImgDTO;
import com.bxm.localnews.news.dto.RelationDTO;
import com.bxm.localnews.news.enums.PostTypeEnum;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.image.ImageHelper;
import com.bxm.localnews.news.list.PostListService;
import com.bxm.localnews.news.model.dto.ForumPostBriefInfoDto;
import com.bxm.localnews.news.model.dto.InteractRankInfo;
import com.bxm.localnews.news.model.param.ForumPostFillContext;
import com.bxm.localnews.news.model.vo.*;
import com.bxm.localnews.news.post.ForumPostFacadeService;
import com.bxm.localnews.news.topic.ForumTopicService;
import com.bxm.localnews.news.util.FormPostContentUtil;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.localnews.news.vo.UserImgVo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.PageParam;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static com.github.pagehelper.page.PageMethod.startPage;

@Service
@Log4j2
public class ForumPostFacadeServiceImpl implements ForumPostFacadeService {

    private final ForumPostMapper forumPostMapper;

    private final RedisSetAdapter redisSetAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserProperties userProperties;

    private final ForumTopicMapper forumTopicMapper;

    private final ForumTopicService forumTopicService;

    private final NewsReplyMapper newsReplyMapper;

    private final LocationIntegrationService locationIntegrationService;

    private final ForumPostDetailService forumPostDetailService;

    private final ImageHelper imageHelper;

    private final PostListService postListService;

    public ForumPostFacadeServiceImpl(ForumPostMapper forumPostMapper, RedisSetAdapter redisSetAdapter,
                                      RedisHashMapAdapter redisHashMapAdapter, UserProperties userProperties,
                                      ForumTopicMapper forumTopicMapper,
                                      ForumTopicService forumTopicService, NewsReplyMapper newsReplyMapper,
                                      LocationIntegrationService locationIntegrationService,
                                      ForumPostDetailService forumPostDetailService,
                                      ImageHelper imageHelper,
                                      PostListService postListService) {
        this.forumPostMapper = forumPostMapper;
        this.redisSetAdapter = redisSetAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userProperties = userProperties;
        this.forumTopicMapper = forumTopicMapper;
        this.forumTopicService = forumTopicService;
        this.newsReplyMapper = newsReplyMapper;
        this.locationIntegrationService = locationIntegrationService;
        this.forumPostDetailService = forumPostDetailService;
        this.imageHelper = imageHelper;
        this.postListService = postListService;
    }

    @Override
    public ForumPostBriefInfoDto getBriefInfo(Long postId, Long userId, String areaCode, String ip) {

        ForumPostVo forumPostVo = forumPostDetailService.getBriefPost(postId, userId, areaCode);


        ForumPostBriefInfoDto briefInfoDto = new ForumPostBriefInfoDto();
        if (null != forumPostVo) {
            PostImgVo imgInfo = forumPostVo.getShareImg();
            if (null != imgInfo) {
                briefInfoDto.setConvertImgUrl(imgInfo.getImgUrl());
            }

            if (!CollectionUtils.isEmpty(forumPostVo.getPlugins())) {
                for (ForumPlugin plugin : forumPostVo.getPlugins()) {
                    if ("VOTE".equals(plugin.getType())) {
                        briefInfoDto.setHasVote(true);
                        break;
                    }
                }
            }

            briefInfoDto.setId(postId);
            briefInfoDto.setTitle(getTitle(forumPostVo));
        }

        return briefInfoDto;
    }

    @Override
    public ForumPostBriefInfoDto getBriefInfo(Long postId) {
        return getBriefInfo(postId, null, null, null);
    }

    private String getTitle(ForumPostVo forumPostVo) {
        String title = forumPostVo.getTitle();
        if (StringUtils.isBlank(title) || PostTypeEnum.NOTE.getCode() == forumPostVo.getPostType()) {
            title = StringUtils.substring(forumPostVo.getTextField(), 0, 50);
        }
        return title;
    }

    @Override
    public Integer getPublishPostNum(Long userId) {
        return forumPostMapper.selectPublishPostNumByUserId(userId);
    }

    @Override
    public List<ForumPostClickCountVo> getRecentPosts(Integer clickCountLimit) {
        return forumPostMapper.getRecentPosts(clickCountLimit, DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -1));
    }

    @Override
    public List<PostClickCountVo> getUnFullClickPosts(PageParam param) {
        return startPage(param).doSelectPage(forumPostMapper::getUnFullClickPosts);
    }

    @Override
    public List<ForumPostClickCountVo> getRecentRecommendPosts(Integer clickCountLimit) {
        return forumPostMapper.getRecentRecommendPosts(clickCountLimit, DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -120));
    }

    @Override
    public void batchAddClick(List<ForumPostClickCountVo> forumPostClickList) {
        if (CollectionUtils.isNotEmpty(forumPostClickList)) {
            MybatisBatchBuilder.create(ForumPostMapper.class, forumPostClickList).run(ForumPostMapper::batchAddClick);
        }
    }

    @Override
    public BaseForumPostDTO getForumPostByUserId(Long userId, Long targetUserId, String areaCode) {
        KeyGenerator recentKey = RedisConfig.RECENT_DEVELOPMENTS.copy().appendKey(String.valueOf(userId));
        //每次取目标用户的第一条帖子
        ForumPostVo forumPostVo = forumPostMapper.selectPostByUserIdAndWeek(targetUserId);
        BaseForumPostDTO result = null;
        //若用户的最新的第一条帖子已被获取,则返回空
        boolean flag = Objects.nonNull(forumPostVo) && !redisSetAdapter.exists(recentKey, forumPostVo.getId());
        if (flag) {
            List<PostImgVo> coverList = imageHelper.converSelect(forumPostVo.getCoverList(), forumPostVo.getPostImgList(), forumPostVo.getCoverSelect());
            //替换占位符
            //设置为开启替换占位符
            forumPostVo.setEnablePlaceholder((byte) 1);
            FormPostContentUtil.replace(forumPostVo, locationIntegrationService.getLocationByGeocode(areaCode));

            result = BaseForumPostDTO.builder()
                    .id(forumPostVo.getId())
                    //分享图片为空时取默认分享图
                    .imgUrl(CollectionUtils.isEmpty(coverList) ? null : coverList.get(0).getImgUrl())
                    .textField(forumPostVo.getTextField())
                    .title(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()))
                    .publishTime(forumPostVo.getPublishTime())
                    .build();
            redisSetAdapter.add(recentKey, forumPostVo.getId());
            //缓存7天后过期
            redisSetAdapter.expire(recentKey, DateUtils.WEEK_MILLISECOND / 1000);
        }
        return result;
    }

    @Override
    public List<UserImgVo> batchGetUserImg(List<Long> userIdList, Long currentUserId) {
        List<UserImgVo> userImgVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userImgVoList;
        }

        boolean hasCurrent = false;
        if (currentUserId != null && userIdList.contains(currentUserId)) {
            userIdList.remove(currentUserId);
            hasCurrent = true;
        }

        if (CollectionUtils.isNotEmpty(userIdList)) {
            //先取缓存
            Set<String> userIds = userIdList.stream().map(Object::toString).collect(Collectors.toSet());
            List<String> userForumImgCache = redisHashMapAdapter.multiGet(RedisConfig.USER_FORUM_IMG, userIds, String.class);

            //处理缓存数据
            if (CollectionUtils.isNotEmpty(userForumImgCache)) {
                userForumImgCache.forEach(userCache -> {
                    UserImgVo userImgVo = JSON.parseObject(userCache, UserImgVo.class);
                    if (null != userImgVo) {
                        userIdList.remove(userImgVo.getUserId());
                        userImgVoList.add(userImgVo);
                    }
                });
            }

            //若缓存中无法取到所有用户图片信息
            if (CollectionUtils.isNotEmpty(userIdList)) {
                userImgVoList.addAll(batchLoadUserImgVoToRedis(userIdList));
            }
        }

        //当前用户需返回仅自己可见内容
        if (hasCurrent) {
            List<ForumPostVo> forumPostVoList = forumPostMapper.batchGetUserPosts(
                    Collections.singletonList(currentUserId), userProperties.getAdvertTopicId(), 1);
            UserImgVo userImgVo = convertPostToUserImg(forumPostVoList, currentUserId);
            if (userImgVo != null) {
                userImgVoList.add(userImgVo);
            }
        }

        return userImgVoList;
    }

    @Override
    public void cacheUserPostImg(Date startTime) {
        //获取有更新的帖子用户
        List<Long> userIdList = forumPostMapper.getUpdatedUser(startTime);
        //缓存数据
        batchLoadUserImgVoToRedis(userIdList);
    }

    @Override
    public TopicFacadeVO getTopicByForumId(Long forumId) {
        List<RelationDTO> relationList = forumTopicMapper.getPostTopicRelation(Collections.singletonList(forumId));
        if (CollectionUtils.isEmpty(relationList)) {
            return null;
        }

        List<TopicVo> topicVoList = forumTopicService.getTopicList(null);

        //缓存取
        if (CollectionUtils.isNotEmpty(topicVoList)) {
            for (TopicVo topicVo : topicVoList) {
                if (topicVo.getId().equals(relationList.get(0).getBid())) {
                    return convert(topicVo);
                }
            }
        }

        //没有的话再走db
        TopicVo topicVo = forumTopicMapper.selectTopicById(relationList.get(0).getBid(), null);

        if (Objects.nonNull(topicVo)) {
            return convert(topicVo);
        }

        return null;
    }

    @Override
    public TopicFacadeVO getTopicById(Long id) {
        TopicVo topicVo = forumTopicService.getTopicDetail(id, null, null);

        if (Objects.isNull(topicVo)) {
            return null;
        }
        return convert(topicVo);
    }

    private TopicFacadeVO convert(TopicVo topicVo) {
        TopicFacadeVO facadeVO = new TopicFacadeVO();
        facadeVO.setId(topicVo.getId());
        facadeVO.setTitle(topicVo.getTitle());
        facadeVO.setDescp(topicVo.getDescp());
        facadeVO.setImg(topicVo.getImg());
        facadeVO.setParticipantsNum(topicVo.getParticipantsNum());
        facadeVO.setAppletShareUrl(topicVo.getAppletShareUrl());
        facadeVO.setShareUrl(topicVo.getShareUrl());
        facadeVO.setLeadPostContent(topicVo.getLeadPostContent());
        facadeVO.setRecommendContent(topicVo.getRecommendContent());
        facadeVO.setJoinCode(topicVo.getJoinCode());
        facadeVO.setJoinHeadImgList(topicVo.getJoinHeadImgList());
        facadeVO.setContent(topicVo.getContent());
        return facadeVO;
    }

    private List<UserImgVo> batchLoadUserImgVoToRedis(List<Long> userIdList) {
        log.info("============需要从数据库获取图片的idList : {}", JSON.toJSONString(userIdList));
        List<UserImgVo> userImgVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userImgVoList;
        }

        //获取用户帖子
        List<ForumPostVo> forumPostVoList = forumPostMapper.batchGetUserPosts(userIdList, userProperties.getAdvertTopicId(), 0);

        log.info("==============数据库获取值结果集 forumPostVoList ：{}  ", JSON.toJSONString(forumPostVoList));
        if (CollectionUtils.isEmpty(forumPostVoList)) {
            return userImgVoList;
        }

        //映射用户帖子图片
        Map<Long, List<ForumPostVo>> userForumPostMap = forumPostVoList.stream().collect(Collectors.groupingBy(ForumPostVo::getUserId));
        Map<String, String> userImgVoMap = Maps.newHashMap();
        userIdList.forEach(userId -> {
            log.info("== userId: {}", userId);
            UserImgVo userImgVo = convertPostToUserImg(userForumPostMap.get(userId), userId);
            log.info("== userImgVo: {}", JSON.toJSONString(userImgVo));
            //即使为空，也要覆盖缓存
            userImgVoMap.put(userId.toString(), JSON.toJSONString(userImgVo));
            if (null != userImgVo) {
                userImgVoList.add(userImgVo);
            }
        });
        log.info("============= result : {}", JSON.toJSONString(userImgVoMap));
        //存入缓存
        if (userImgVoMap.size() > 0) {
            redisHashMapAdapter.putAll(RedisConfig.USER_FORUM_IMG, userImgVoMap);
        }

        return userImgVoList;
    }

    private UserImgVo convertPostToUserImg(List<ForumPostVo> forumPostVos, Long userId) {
        if (CollectionUtils.isEmpty(forumPostVos)) {
            return null;
        }

        // 只获取4个
        int size = 0;
        List<ImgDTO> imgDtoList = new ArrayList<>();
        for (ForumPostVo forumPostVo : forumPostVos) {
            List<PostImgVo> postImgVoList = forumPostVo.getPostImgList();
            if (CollectionUtils.isNotEmpty(postImgVoList)) {
                for (PostImgVo postImgVo : postImgVoList) {
                    if ("IMG".equals(postImgVo.getType())) {
                        ImgDTO imgDto = new ImgDTO();
                        imgDto.setId(forumPostVo.getId());
                        imgDto.setCategory("POST");
                        imgDto.setType(postImgVo.getType());
                        imgDto.setImgUrl(postImgVo.getImgUrl());
                        imgDtoList.add(imgDto);
                        size++;
                        if (size >= 4) {
                            break;
                        }
                    }
                }
            }
            if (size >= 4) {
                break;
            }
        }

        UserImgVo userImgVo = new UserImgVo();
        userImgVo.setUserId(userId);
        userImgVo.setImgDtoList(imgDtoList);
        return userImgVo;
    }

    @Override
    public ForumPostVo getForumPostById(Long id) {
        return forumPostMapper.selectBasicPostById(id);
    }

    @Override
    public List<ForumRankVo> forumRank(String areaCode, List<Long> userIds) {
        //总榜单
        List<ForumRankVo> list = forumPostMapper.getForumRankInfo(areaCode, userIds);

        //填充榜单内用户评论
        list.forEach(item -> item.setPostTitleInfo(this.fill(item.getUserId())));

        return list.size() > 0 ? list : new ArrayList<>();
    }

    private List<ForumPostTitleInfoVo> fill(Long userId) {
        List<ForumPostVo> list = forumPostMapper.selectPostTitletByUserId(userId);
        ForumPostFillContext context = ForumPostFillContext.builder()
                .data(list)
                .fillTitle(true)
                .loadCollectStatus(false)
                .loadComment(false)
                .loadHotReplay(false)
                .loadLikeStatus(false)
                .build();

        postListService.fillExtInfo(context);
        return list.stream().map(item -> {
            ForumPostTitleInfoVo forumPostTitleInfoVo = new ForumPostTitleInfoVo();
            forumPostTitleInfoVo.setId(item.getId());
            forumPostTitleInfoVo.setTitle(item.getTitle());
            return forumPostTitleInfoVo;
        }).collect(Collectors.toList());
    }

    @Override
    public ForumPostRankForUserVo forumRankForUser(Long userId) {
        return forumPostMapper.getForumRankInfoForUser(userId);
    }

    @Override
    public List<InteractRankInfo> getInteractRankInfo(String areaCode, int limitNum, List<Long> userIds) {
        return newsReplyMapper.selectInteractRankInfo(areaCode, limitNum, userIds);
    }

    @Override
    public Integer replyNumForUser(Long userId) {
        return newsReplyMapper.getReplyNumById(userId);
    }

    @Override
    public List<RecommendUserFacadeVo> getRecommendUserInfo(String areaCode, List<Long> userIds) {
        return forumPostMapper.getRecommendUserInfo(areaCode, CollectionUtils.isEmpty(userIds)
                ? Lists.newArrayList(0L)
                : userIds);
    }


}
