package com.bxm.localnews.news.post.impl;

import com.alibaba.fastjson.JSONArray;
import com.bxm.localnews.integration.MissionIntegrationService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.enums.TaskEnum;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.model.dto.PostTopicDTO;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.post.ForumPostService;
import com.bxm.localnews.news.util.OssTransCodeUtils;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.localnews.news.vo.TranscodeMap;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.sync.core.CacheHolder;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.experimental.Tolerate;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static com.bxm.localnews.news.constant.MemoryCacheKey.LAST_POST_TIME_CACHE;
import static java.util.Objects.isNull;

/**
 * @author liujia
 */
@Service
@Log4j2
@AllArgsConstructor
public class ForumPostServiceImpl extends BaseService implements ForumPostService {

    private ForumPostMapper forumPostMapper;

    private MixedRecommendPoolMapper mixedRecommendPoolMapper;

    private UserIntegrationService userIntegrationService;

    private UserAccountIntegrationService userAccountIntegrationService;

    private OssTransCodeUtils ossTransCodeUtils;

    private ForumTopicMapper forumTopicMapper;

    private MissionIntegrationService missionIntegrationService;

    private CacheHolder cacheHolder;

    @Override
    public Message deleteForumPost(Long id) {
        ForumPostVo forumPostVo = this.forumPostMapper.selectWithoutContent(id);
        String result = checkPostInfo(forumPostVo);
        if (StringUtils.isNotBlank(result)) {
            return Message.build(Boolean.FALSE, result);
        }
        //状态置为删除
        ForumPostVo updatePost = new ForumPostVo();
        updatePost.setId(id);
        updatePost.setModifyTime(new Date());
        updatePost.setStatus(PostStatusEnum.USER_DELETE.getCode());

        //更新帖子信息表
        this.forumPostMapper.updateByPrimaryKeySelective(updatePost);
        //删除内容推荐库的内容
        this.mixedRecommendPoolMapper.deleteMixRecommandPoolById(id);

        Integer goldNum = userAccountIntegrationService.countGoldByPostId(id, forumPostVo.getUserId());
        if (null != goldNum && goldNum > 0) {
            String content = "删除【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】";
            userAccountIntegrationService.addGold(
                    AccountGoldParam.buildPostDeleteParam(forumPostVo.getUserId(), -goldNum, id, content));
        }

        //更新用户发布数
        userIntegrationService.addPostReplyNum(forumPostVo.getUserId(), 2);
        return Message.build(Boolean.TRUE);
    }

    @Override
    public Boolean getContentUpdate(Date lastRequestTime, String areaCode) {
        if (isNull(lastRequestTime)) {
            return true;
        }

        Date maxPublishTime = cacheHolder.get(LAST_POST_TIME_CACHE, areaCode);

        if (null != maxPublishTime) {
            return maxPublishTime.compareTo(lastRequestTime) > 0;
        }
        return false;
    }

    @Tolerate
    public void init() {
        cacheHolder.set(LAST_POST_TIME_CACHE,
                (areaCode) -> forumPostMapper.selectMaxPublishTime(areaCode),
                200,
                TimeUnit.SECONDS,
                10);
    }

    @Override
    public void execTransCode(Long postId, String imageList, List<PostImgVo> postImgList) {
        if (postImgList == null) {
            postImgList = Lists.newArrayList();
        }

        if (null != imageList) {
            postImgList.addAll(JSONArray.parseArray(imageList, PostImgVo.class));
        }

        postImgList.forEach(postImgVo -> {
            if ("VIDEO".equalsIgnoreCase(postImgVo.getType())) {
                ossTransCodeUtils.transcodeByOriginUrl(postImgVo.getVideoUrl(), postId, (destUrl) -> {
                    TranscodeMap transcodeMap = new TranscodeMap();
                    transcodeMap.setOriginUrl("");
                    transcodeMap.setDestUrl(destUrl);
                    transcodeMap.setPostId(postId);

                    forumPostMapper.batchUpdatePostVideo(transcodeMap);
                });
            }
        });
    }

    @Async
    @Override
    public void doTriggerUpdateInfo(Long userId, String nickname, String headImg) {
        List<ForumPostVo> forumPostVoList = forumPostMapper.selectForumPostByUser(userId);
        for (ForumPostVo forumPostVo : forumPostVoList) {
            forumPostMapper.updateUserInfo(forumPostVo.getId(), userId, headImg, nickname);
        }
    }

    @Override
    public void updateTopic(Long postId, List<Long> topicIdList, Long secondTopicId) {
        if (null == postId) {
            return;
        }

        this.forumTopicMapper.batchDeleteTopic(postId);

        if (CollectionUtils.isNotEmpty(topicIdList)) {
            List<PostTopicDTO> list = new ArrayList<>();
            topicIdList.forEach(topicId -> list.add(getPostTopic(postId, topicId, secondTopicId)));
            this.forumTopicMapper.batchInsertTopic(list);

        }
    }

    private PostTopicDTO getPostTopic(Long postId, Long topicId, Long secondTopicId) {
        PostTopicDTO postTopicDTO = new PostTopicDTO();
        postTopicDTO.setId(nextSequence());
        postTopicDTO.setPostId(postId);
        postTopicDTO.setTopicId(topicId);
        postTopicDTO.setSecondTopicId(secondTopicId == null ? 0L : secondTopicId);
        return postTopicDTO;
    }

    /**
     * 校验帖子是否能被删除
     *
     * @param forumPostVo 帖子
     * @return 删除提示
     */
    private String checkPostInfo(ForumPostVo forumPostVo) {
        String message = StringUtils.EMPTY;
        if (forumPostVo == null) {
            message = "帖子不存在";
        } else if (forumPostVo.getIsCash() == 1) {
            message = "现金奖励内容删除需要联系客服哦";
        } else if (forumPostVo.getIsBroke() == 1) {
            message = "爆料内容删除需要联系客服哦";
        } else if (forumPostVo.getIsRecommend() == 1) {
            message = "优质头条删除需要联系客服哦";
        } else if (forumPostVo.getIsBrilliant() == 1) {
            message = "精华内容删除需要联系客服哦";
        }
        return message;
    }

    @Override
    public NewsCompleTaskDTO completeTask(ForumPostVo forumPostVo) {
        Long userId = forumPostVo.getUserId();
        Long postId = forumPostVo.getId();
        //首次发帖
        String content = "首次发布【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】";
        NewsCompleTaskDTO newsCompleTaskDTO = missionIntegrationService.compleTask(userId, TaskEnum.TASK_FIRST_POST_INTIVATION.name(), postId.toString(), content);
        //日常发帖
        if (null == newsCompleTaskDTO || null == newsCompleTaskDTO.getGoldNum() || 0L == newsCompleTaskDTO.getGoldNum()) {
            content = "发布【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】";
            newsCompleTaskDTO = missionIntegrationService.compleTask(userId, TaskEnum.TASK_POST_INTIVATION.name(), postId.toString(), content);
        }
        return newsCompleTaskDTO;
    }
}
