package com.bxm.localnews.news.post.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.common.constant.DomainScene;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.PushMsgIntegService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.config.NewsProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.enums.DisplayAreaEnum;
import com.bxm.localnews.news.list.PostListService;
import com.bxm.localnews.news.model.dto.HotForumListShareDTO;
import com.bxm.localnews.news.model.param.ForumPostFillContext;
import com.bxm.localnews.news.model.param.ForumPostListQueryParam;
import com.bxm.localnews.news.model.param.HotForumListShareParam;
import com.bxm.localnews.news.model.param.HotForumsParam;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.HotForumPostVo;
import com.bxm.localnews.news.model.vo.TopicHotForumGuideInfo;
import com.bxm.localnews.news.post.HotPostService;
import com.bxm.localnews.news.recommend.RecommendPostService;
import com.bxm.localnews.news.util.PlaceHolderUtil;
import com.bxm.localnews.news.vo.HotForumDateBean;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.PageWarper;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.RedisConfig.TOPIC_HOT_POST_CACHE_KEY;

/**
 * 热门帖子相关逻辑
 *
 * @author liujia
 * @date 1/14/21 6:27 PM
 **/
@Service
@Slf4j
public class HotPostServiceImpl implements HotPostService {

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private ForumProperties forumProperties;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private PostListService postListService;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private PushMsgIntegService pushMsgIntegService;

    @Resource
    private DomainIntegrationService domainIntegrationService;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private NewsProperties newsProperties;

    private RecommendPostService recommendPostService;

    private RecommendPostService getRecommendPostService() {
        if (null == recommendPostService) {
            recommendPostService = SpringContextHolder.getBean(RecommendPostService.class);
        }
        return recommendPostService;
    }

    @Override
    public PageWarper<HotForumPostVo> hotForumPost(HotForumsParam param) {
        PageWarper<HotForumPostVo> forumPostPage = new PageWarper<>();
        //没传地区码不显示热门帖子
        if (Objects.isNull(param.getAreaCode())) {
            forumPostPage.setList(Collections.emptyList());
            return forumPostPage;
        }

        //获取缓存的榜单帖子
        List<HotForumPostVo> hotForums = getHotForumList(param.getAreaCode());

        if (CollectionUtils.isNotEmpty(hotForums)) {
            //根据入参决定集合取值范围
            hotForums = param.getViewType() == 0
                    ? hotForums
                    : hotForums.subList(0, forumProperties.getHotForumTopViewNum() < hotForums.size()
                    ? forumProperties.getHotForumTopViewNum()
                    : hotForums.size());

            int start = (param.getPageNum() - 1) * param.getPageSize();
            int end = param.getPageNum() * param.getPageSize();
            if (start < 0) {
                start = 0;
            }
            if (end > hotForums.size()) {
                end = hotForums.size();
            }

            List<HotForumPostVo> subForums = hotForums.subList(start, end);
            if (CollectionUtils.isEmpty(subForums)) {
                forumPostPage.setList(Collections.emptyList());
                forumPostPage.setIsLastPage(true);
                forumPostPage.setHasNextPage(false);
                return forumPostPage;
            }

            List<Long> hotForumIds = subForums.stream().map(HotForumPostVo::getId).collect(Collectors.toList());
            //帖子内容需要动态获取组装，所以实时获取数据
            List<HotForumPostVo> result = forumPostMapper.ForumsListByIds(hotForumIds);
            //取缓存数据中的排名和热度
            final List<HotForumPostVo> innerHotForums = hotForums;
            result.forEach(item -> innerHotForums.forEach(hotForumPostVo -> {
                if (Objects.equals(hotForumPostVo.getId(), item.getId())) {
                    item.setHotRank(hotForumPostVo.getHotRank());
                    item.setHotNum(forumHotNum(item.getDisplayDateTime(), hotForumPostVo.getHotNum()));
                }
            }));
            result = result.stream().sorted(Comparator.comparing(HotForumPostVo::getHotRank)).collect(Collectors.toList());

            forumPostPage.setList(result);
            forumPostPage.setPageNum(param.getPageNum());
            forumPostPage.setPageSize(param.getPageSize());

            //组装帖子内容详情
            postListService.fillExtInfo(ForumPostFillContext.builder()
                    .data(forumPostPage.getList())
                    .userId(param.getUserId())
                    .areaCode(param.getAreaCode())
                    .fillTitle(false)
                    .displayArea(DisplayAreaEnum.OTHER)
                    .build());

            //组装分页参数
            if (CollectionUtils.isEmpty(result) || result.size() < param.getPageSize()) {
                forumPostPage.setIsLastPage(true);
                forumPostPage.setHasNextPage(false);
            } else {
                forumPostPage.setHasNextPage(true);
            }
        }

        return forumPostPage;
    }

    /**
     * 缓存获取热帖列表原始数据
     *
     * @param areaCode 地区码
     * @return 数据集合
     */
    @Override
    public List<HotForumPostVo> getHotForumList(String areaCode) {
        TypeReference<List<HotForumPostVo>> typeReference = new TypeReference<List<HotForumPostVo>>() {
        };
        List<HotForumPostVo> hotForumPostVos = redisHashMapAdapter.get(RedisConfig.FORUM_HOT_LIST.copy(), areaCode, typeReference);
        if (CollectionUtils.isEmpty(hotForumPostVos)) {
            return this.loadFromDb(areaCode);
        }
        return hotForumPostVos;
    }

    @Override
    public List<HotForumPostVo> loadFromDb(String areaCode) {
        List<HotForumPostVo> hotForums = forumPostMapper.hotListPost(areaCode, forumProperties.getHotForumLimitNum(), forumProperties.getHotForumDaysLimit());
        for (int i = 0; i < hotForums.size(); i++) {
            hotForums.get(i).setHotRank(i + 1);
        }
        if (CollectionUtils.isNotEmpty(hotForums)) {
            //上榜推送
            this.pushMsgForHotForums(hotForums);

            redisHashMapAdapter.put(RedisConfig.FORUM_HOT_LIST.copy(), areaCode, hotForums);
            redisHashMapAdapter.expire(RedisConfig.FORUM_HOT_LIST.copy(), DateUtils.DAY_MILLISECOND / 1000);
        }
        return hotForums;
    }

    /**
     * 重新计算热门帖子热度值
     *
     * @param displayTime 发帖事件
     * @param hotNumOlder 原有的热度值
     * @return 新的热度值
     */
    private Integer forumHotNum(Date displayTime, Integer hotNumOlder) {
        List<HotForumDateBean> list = JSONObject.parseArray(forumProperties.getHotForumDate(), HotForumDateBean.class);
        int diffDays = DateUtils.getDiffDays(new Date(), displayTime, true);
        for (HotForumDateBean hotForumDateBean : list) {
            if (hotForumDateBean.getDays() == diffDays) {
                return hotForumDateBean.getOfferst().multiply(BigDecimal.valueOf(hotNumOlder)).intValue();
            }
        }
        return hotNumOlder;
    }

    private void pushMsgForHotForums(List<HotForumPostVo> hotForums) {
        hotForums.forEach(forum -> {
            if (!redisSetAdapter.exists(RedisConfig.FORUM_HOT_PUSH_KEY.copy(), forum.getId())) {
                pushMsgIntegService.pushHotForum(forum);
                savePushForumToRedis(forum.getId());
            }
        });
    }

    private void savePushForumToRedis(Long forumId) {
        redisSetAdapter.add(RedisConfig.FORUM_HOT_PUSH_KEY.copy(), forumId);
    }

    @Override
    public HotForumListShareDTO hotForumsShareBaseInfo(HotForumListShareParam param) {
        LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(param.getAreaCode());
        String shareH5host = domainIntegrationService.getOutSideShareBaseUrl(DomainScene.DomainViewScene.WX_JS_VIEW);
        String shareUrl = null;
        try {
            shareUrl = shareH5host + newsProperties.getHotForumShareUrl() + "" +
                    "?userId=" + param.getUserId() + "" +
                    "&areaCode=" + param.getAreaCode() + "" +
                    "&areaName=" + URLEncoder.encode(locationDTO.getName(), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error(e.getMessage(), e);
        }
        return HotForumListShareDTO.builder()
                .userId(param.getUserId())
                .title("新鲜出炉！" + locationDTO.getName() + "一周热门话题榜！")
                .contentDesc("我正在看" + locationDTO.getName() + "热门内容,快来一起看看吧")
                .imgUrl(newsProperties.getShareImgUrl())
                .shareUrl(shareUrl)
                .shareButtonText(newsProperties.getShareButtonText())
                .build();
    }


    @Override
    public TopicHotForumGuideInfo loadTopicHotPost(ForumPostListQueryParam param, BasicParam basicParam) {
        String subKey = param.getTopicId() + param.getAreaCode();
        TopicHotForumGuideInfo hotPost = redisHashMapAdapter.get(TOPIC_HOT_POST_CACHE_KEY, subKey, TopicHotForumGuideInfo.class);

        if (hotPost == null || hotPost.getExpiredTime() < System.currentTimeMillis()) {
            hotPost = getTopicHotPost(param, basicParam);
            hotPost.setExpiredTime(System.currentTimeMillis() + 5 * 60 * 1000L);

            redisHashMapAdapter.put(TOPIC_HOT_POST_CACHE_KEY, subKey, hotPost);
        }

        return hotPost;

    }

    /**
     * 3.10.0 需求 获取要插入的热门首图数据  逻辑：取热门类型首个带图的帖子基础信息
     * <p>
     * 当type=2 最新列表时插入 引导至热门列表项内容数据
     */
    private TopicHotForumGuideInfo getTopicHotPost(ForumPostListQueryParam param, BasicParam basicParam) {
        TopicHotForumGuideInfo topicHotForumGuideInfo = new TopicHotForumGuideInfo();

        ForumPostListQueryParam newParam = new ForumPostListQueryParam();
        BeanUtils.copyProperties(param, newParam);
        newParam.setPageNum(1);
        newParam.setPageSize(5);
        newParam.setType(0);

        boolean flag = true;
        do {
            //取最热门帖子第一条带图帖子
            List<Long> postIdList = getRecommendPostService().getRecommendPostIds(newParam, basicParam);
            if (CollectionUtils.isNotEmpty(postIdList)) {
                List<ForumPostVo> postList = forumPostMapper.listPostByIds(postIdList);
                for (ForumPostVo forumPostVo : postList) {
                    if (CollectionUtils.isNotEmpty(forumPostVo.getPostImgList())) {
                        //赋值
                        this.convent(topicHotForumGuideInfo, forumPostVo, param);
                        flag = false;
                        break;
                    }
                }
                if (Objects.isNull(topicHotForumGuideInfo.getImg())) {
                    //继续下一页
                    newParam.setPageNum(newParam.getPageNum() + 1);
                }
            } else {
                flag = false;
            }
        } while (flag);

        return topicHotForumGuideInfo;
    }

    private void convent(TopicHotForumGuideInfo topicHotForumGuideInfo, ForumPostVo forumPostVo, ForumPostListQueryParam param) {
        topicHotForumGuideInfo.setImg(forumPostVo.getPostImgList().get(0).getImgUrl());
        topicHotForumGuideInfo.setClickCount(forumPostVo.getClickCount());
        topicHotForumGuideInfo.setTitle(StringUtils.isNotBlank(forumPostVo.getTitle())
                ? forumPostVo.getTitle()
                : StringUtils.isNotBlank(forumPostVo.getTextField()) ? forumPostVo.getTextField() : "分享图片");

        //替换标题中的占位符
        LocationDTO locationDTO = new LocationDTO();
        if (StringUtils.isNotBlank(param.getAreaCode())) {
            locationDTO = locationIntegrationService.getLocationByGeocode(param.getAreaCode());
        }
        String title = PlaceHolderUtil.replace(topicHotForumGuideInfo.getTitle(), "areaname",
                locationDTO.getName(),
                "本地");

        topicHotForumGuideInfo.setTitle(title);
        topicHotForumGuideInfo.setId(forumPostVo.getId());
    }
}
