package com.bxm.localnews.news.post.impl;

import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.PostCountProperties;
import com.bxm.localnews.news.post.PostCountService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Objects;

import static java.math.RoundingMode.DOWN;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * 普通帖子阅读量 = 初始基数 * (1 + ln2 真实阅读量 / 10) + 真实阅读量
 * <p>
 * 本地用户数：20000 + 对应地区真实用户注册量
 * 初始基数：本地用户数 * 随机比例
 * 随机比例：
 * 普通帖子：(5 ~ 20) %
 * 推荐帖子：(15 ~ 25) %
 *
 * @author gonzo
 * @date 2020-10-28 20:54
 **/
@Service
@Slf4j
@AllArgsConstructor
public class PostCountServiceImpl implements PostCountService {

    private final PostCountProperties postCountProperties;

    private static final BigDecimal ONE_HUNDRED = BigDecimal.valueOf(100);

    private final UserIntegrationService userIntegrationService;

    @Override
    public long getInitialBasicNum(String areaCode, boolean isRecommend) {

        // 默认全国的基数
        BigDecimal localUserNum = postCountProperties.getDefaultGlobalUserNum();

        if (isNotBlank(areaCode)) {
            // 本地用户数：20000 + 对应地区真实用户注册量
            if (areaCode.contains(",")) {
                areaCode = areaCode.split(",")[0];
            }

            if (isNotBlank(areaCode)) {
                // 带了地区，就用地区的默认基数
                localUserNum = postCountProperties.getDefaultLocalUserNum();
                localUserNum = localUserNum.add(BigDecimal.valueOf(userIntegrationService.getRegisterUserCount(areaCode)));
            }
        }

        // 获取随机比例的范围
        int start;
        int end;
        if (isRecommend) {
            start = postCountProperties.getMinRecommendDefaultInitialBasicNumRate();
            end = postCountProperties.getMaxRecommendDefaultInitialBasicNumRate();
        } else {
            start = postCountProperties.getMinDefaultInitialBasicNumRate();
            end = postCountProperties.getMaxDefaultInitialBasicNumRate();
        }

        // 得到随机比例
        int rate = RandomUtils.nextInt(start, end + 1);

        // 本地用户数 * 随机比例
        return localUserNum.multiply(BigDecimal.valueOf(rate))
                .divide(ONE_HUNDRED, 0, DOWN)
                .longValue();
    }

    @Override
    public long getInitialBasicNum(String areaCode, Byte isRecommend) {
        return getInitialBasicNum(areaCode, Objects.equals(isRecommend, (byte) 1));
    }

    @Override
    public long getFinalClickCount(long realClickCount, String areaCode, boolean isRecommend) {
        return getFinalClickCount(realClickCount, getInitialBasicNum(areaCode, isRecommend));
    }

    @Override
    public long getFinalClickCount(long realClickCount, long initialBasicNum) {

        try {
            // 阅读量 = 初始基数 * (1 + log2 (真实阅读量) / 10) + 真实阅读量
            BigDecimal initialBasicNumCal = BigDecimal.valueOf(initialBasicNum);
            // log2 (真实阅读量)
            // log2N=logeN/loge2，logeN代表以e为底的N的对数，loge2代表以e为底的2的对数。\
            BigDecimal log2RealClickCount;
            if (realClickCount > 0) {
                log2RealClickCount = BigDecimal.valueOf(Math.log(realClickCount) / Math.log(2)).setScale(10, DOWN);
            } else {
                log2RealClickCount = BigDecimal.ZERO;
            }
            BigDecimal realClickCountCal = BigDecimal.valueOf(realClickCount);
            // 初始基数 *
            return initialBasicNumCal.multiply(
                    // (1 +
                    BigDecimal.ONE.add(
                            // log2 (真实阅读量) / 10)
                            log2RealClickCount.divide(BigDecimal.TEN, 0, DOWN)))
                    // + 真实阅读量
                    .add(realClickCountCal).longValue();
        } catch (Exception e) {
            log.error("计算阅读量失败，realClickCount: {}, initialBasicNum: {}", realClickCount, initialBasicNum, e);
        }

        return 0;

    }
}
