package com.bxm.localnews.news.recommend.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.integration.AdminMixRecommendPoolService;
import com.bxm.localnews.integration.AdvertIntegrationService;
import com.bxm.localnews.integration.NewsRecommendIntegrationService;
import com.bxm.localnews.news.config.AdvertProperties;
import com.bxm.localnews.news.config.BizConfigProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.convert.ForumParamPageConvert;
import com.bxm.localnews.news.domain.AdminMixedRecommendPoolMapper;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.enums.DisplayAreaEnum;
import com.bxm.localnews.news.enums.MixRecommendTypeEnum;
import com.bxm.localnews.news.enums.PostStatusEnum;
import com.bxm.localnews.news.list.PostListService;
import com.bxm.localnews.news.model.param.ForumPostFillContext;
import com.bxm.localnews.news.model.param.ForumPostListQueryParam;
import com.bxm.localnews.news.model.vo.*;
import com.bxm.localnews.news.post.HotPostService;
import com.bxm.localnews.news.recommend.RecommendPostService;
import com.bxm.localnews.news.recommend.TopicForumService;
import com.bxm.localnews.news.topic.ForumTopicService;
import com.bxm.localnews.news.util.ForumPostTypeUtil;
import com.bxm.localnews.news.util.VersionUtils;
import com.bxm.localnews.news.vo.AdvertIndexBean;
import com.bxm.localnews.param.ForumParam;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author liujia
 * @date 1/15/21 10:24 AM
 **/
@Service
@AllArgsConstructor
@Slf4j
public class RecommendPostServiceImpl implements RecommendPostService {

    private BizConfigProperties bizConfigProperties;

    private ForumPostMapper forumPostMapper;

    private PostListService postListService;

    private TopicForumService topicForumService;

    private HotPostService hotPostService;

    private AdvertProperties advertProperties;

    private AdvertIntegrationService advertIntegrationService;

    private ForumTopicService forumTopicService;

    private RedisHashMapAdapter redisHashMapAdapter;

    private NewsRecommendIntegrationService newsRecommendIntegrationService;

    private ForumParamPageConvert forumParamPageConvert;

    private AdminMixedRecommendPoolMapper adminMixedRecommendPoolMapper;

    private AdminMixRecommendPoolService adminMixRecommendPoolService;

    @Override
    public PageWarper<ForumPostVo> listForumPost(ForumPostListQueryParam param, BasicParam basicParam) {
        log.debug("获取推荐流,param:{}，basicParam:{}", JSONObject.toJSON(param), basicParam);

        PageWarper<ForumPostVo> forumPostPage = new PageWarper<>();
        if (isBlank(param.getAreaCode())) {
            log.warn("获取推荐流，区域编码为空，返回空数据");
            forumPostPage.setList(Collections.emptyList());
            return forumPostPage;
        }

        List<Long> postIdList;
        //如果是小程序审核
        if (PlatformEnum.APPLET.getCode() == basicParam.getPlatform()
                && bizConfigProperties.getAppletReviewEnable()
                && 3 == param.getType()) {
            postIdList = forumPostMapper.listPostRecommend(param);
        } else {
            //推荐库取帖子id
            postIdList = this.getRecommendPostIds(param, basicParam);
        }

        //组装帖子内容
        if (CollectionUtils.isNotEmpty(postIdList)) {
            List<ForumPostVo> postList = forumPostMapper.listPostByIds(postIdList);
            forumPostPage.setList(postList);
            forumPostPage.setPageNum(param.getPageNum());
            forumPostPage.setPageSize(param.getPageSize());
            //组装帖子内容详情
            ForumPostFillContext context = ForumPostFillContext.builder()
                    .data(forumPostPage.getList())
                    .userId(param.getUserId())
                    .areaCode(param.getAreaCode())
                    .fillTitle(false)
                    .loadHotReplay(true)
                    .displayArea(DisplayAreaEnum.OTHER)
                    .build();
            postListService.fillExtInfo(context);

            //若为社区首页且数量大于3且版本号大于2.5.0，则插入推荐话题
            if (3 == param.getType() && postIdList.size() >= 3) {
                //老版本推荐逻辑
                if (VersionUtils.isHighVersion(basicParam.getCurVer(), "2.5.0") > 0 && VersionUtils.isHighVersion(basicParam.getCurVer(), "3.10.0") < 0) {
                    addRecommendTopic(param, forumPostPage, postList);
                }
                //3.10.0版本及以上推荐逻辑
                if (VersionUtils.isHighVersion(basicParam.getCurVer(), "3.9.0") > 0) {
                    forumPostPage.setList((List<ForumPostVo>) topicForumService.recommendTopic(postList, 2, param.getUserId(), param.getAreaCode()));
                }
            }
        }

        //后续特殊逻辑处理
        afterLogic(param, forumPostPage, basicParam);

        //组装分页参数
        if (CollectionUtils.isEmpty(postIdList) || postIdList.size() < param.getPageSize()) {
            forumPostPage.setIsLastPage(true);
            forumPostPage.setHasNextPage(false);
        } else {
            forumPostPage.setHasNextPage(true);
        }

        log.debug("[listForumPost]拉取帖子列表完毕");
        return forumPostPage;
    }

    /**
     * 后续一些特殊处理
     *
     * @param param
     * @param forumPostPage
     */
    private void afterLogic(ForumPostListQueryParam param, PageWarper<ForumPostVo> forumPostPage, BasicParam basicParam) {

        //在固定位置插入广告信息流
        this.addAdvert(param, forumPostPage);

        //3.10.0 插入最新tab到最热tab引导数据
        //只有type=2 为“最新”类型时 且版本为3.10.0及以上时插入此内容
        if (param.getType() == 2 && VersionUtils.isHighVersion(basicParam.getCurVer(), "3.9.0") > 0) {
            //只在第一页插入
            if (param.getPageNum() == 1 && CollectionUtils.isNotEmpty(forumPostPage.getList())) {
                ForumPostVo forumPostVo = new ForumPostVo();
                forumPostVo.setTopicHotForumGuideInfo(hotPostService.loadTopicHotPost(param, basicParam));

                List<ForumPostVo> list = forumPostPage.getList();
                if (CollectionUtils.isEmpty(list)) {
                    return;
                }
                list.add(0, forumPostVo);
                forumPostPage.setList(list);
            }
        }
    }

    /**
     * 插入广告信息流
     */
    private void addAdvert(ForumPostListQueryParam param, PageWarper<ForumPostVo> forumPostPage) {
        if (isBlank(param.getAreaCode())) {
            log.warn("城市编码为空 跳过插入广告信息流");
            return;
        }

        List<ForumPostVo> list = forumPostPage.getList();
        //话题屏蔽广告
        if (CollectionUtils.isEmpty(list) || Objects.nonNull(param.getTopicId()) || Objects.nonNull(param.getChannelIds())) {
            return;
        }

        //只在第一页插入
        if (param.getPageNum() == 1) {
            //获取本地圈新闻要插入广告的位置以及所对应的广告位类型
            List<AdvertIndexBean> listIndex = JSONObject.parseArray(advertProperties.getForumIndex(), AdvertIndexBean
                    .class);

            listIndex.forEach(advertIndexBean -> {
                //根据广告类型获取广告内容
                List<AdvertVO> advertList = advertIntegrationService.getAdvertsByType(advertIndexBean.getAdvertType()
                        , param.getAreaCode(), null);

                //猛烈插入
                if (advertList.size() > 0) {
                    list.add(list.size() >= advertIndexBean.getIndex() ? advertIndexBean.getIndex() - 1 : list
                                    .size(),
                            this.convert(advertList));
                }

            });

            forumPostPage.setList(list);
        }
    }

    private ForumPostVo convert(List<AdvertVO> list) {
        ForumPostVo forumPostVo = new ForumPostVo();
        forumPostVo.setAdvertList(list);

        return forumPostVo;
    }

    private void addRecommendTopic(ForumPostListQueryParam param, PageWarper<ForumPostVo> forumPostPage, List<ForumPostVo> postList) {
        if (isBlank(param.getAreaCode())) {
            log.warn("城市编码为空，调过插入推荐话题");
            return;
        }

        //插入推荐话题
        List<TopicVo> topicVoList = forumTopicService.getTopicList(param.getAreaCode());
        if (CollectionUtils.isNotEmpty(topicVoList)) {
            if (topicVoList.size() > 6) {
                topicVoList = topicVoList.stream().limit(6).collect(Collectors.toList());
            }

            Integer order = redisHashMapAdapter.get(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), param.getUserId() + "", Integer.class);
            if (null == order || ++order >= topicVoList.size()) {
                order = 0;
            }
            redisHashMapAdapter.put(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), param.getUserId() + "", order);
            TopicVo topic = topicVoList.get(order);

            List<ForumPostVo> newPostList = new ArrayList<>();
            for (int i = 0; i < postList.size(); i++) {
                if (i == 3) {
                    ForumPostVo forumPostVo = new ForumPostVo();
                    forumPostVo.setRecommendTopicVo(topic);
                    newPostList.add(forumPostVo);
                }

                newPostList.add(postList.get(i));
            }

            forumPostPage.setList(newPostList);
        }
    }

    @Override
    public List<Long> getRecommendPostIds(ForumPostListQueryParam param, BasicParam basicParam) {
        //包装参数，从推荐引擎中获取相应的帖子id
        ForumParam forumParam = forumParamPageConvert.convert(param);
        Objects.requireNonNull(forumParam).setPlatform(basicParam.getPlatform());
        forumParam.setChannelIds(param.getChannelIds());
        return newsRecommendIntegrationService.recommendList(forumParam);
    }

    /**
     * 更新推荐池内容
     * 当目前状态是<正常显示>时，更新在推荐库中的内容
     * 否则删除在推荐库中的内容
     *
     * @param adminForumPost 帖子
     */
    @Override
    public void updateRecommendWhenChanged(AdminForumPost adminForumPost) {
        Byte isBrilliant = adminForumPost.getIsBrilliant();
        Integer status = adminForumPost.getStatus();
        AdminMixedRecommendPool result = adminMixedRecommendPoolMapper.selectByPrimaryKey(adminForumPost.getId());
        //审核通过状态
        if (Objects.equals(PostStatusEnum.NORMAL.getCode(), status)) {
            if (Objects.nonNull(result)) {
                Byte auto = result.getAuto();
                if (Objects.equals(isBrilliant, (byte) 0) && (isNull(auto) || auto == 0)) {
                    //当推荐池是自动加入时,取消精华帖删除推荐池数据
                    adminMixedRecommendPoolMapper.deleteByPrimaryKey(result.getId());
                } else {
                    AdminMixedRecommendPool adminMixedRecommendPool = AdminMixedRecommendPool.builder()
                            .id(adminForumPost.getId())
                            .title(adminForumPost.getTitle())
                            .issueTime(adminForumPost.getDisplayTime())
                            .publishTime(adminForumPost.getPublishTime())
                            .author(adminForumPost.getUserName())
                            .userId(adminForumPost.getUserId())
                            .areaDetail(isNotBlank(adminForumPost.getAreaCode()) ? adminForumPost.getAreaCode() : null)
                            .weight(result.getWeight())
                            .auto(result.getAuto())
                            .recommendTime(result.getRecommendTime())
                            .optimizationNotice(result.getOptimizationNotice())
                            .deliveryType(adminForumPost.getDeliveryType())
                            .type(ForumPostTypeUtil.getPostType(adminForumPost))
                            .origin(MixRecommendTypeEnum.MIX_POST.getCode()).build();

                    adminMixRecommendPoolService.updateMixRecommend(adminMixedRecommendPool);
                }
            } else {
                //精华帖并且审核通过加入推荐库
                if (Objects.equals(isBrilliant, (byte) 1)) {
                    adminMixRecommendPoolService.addPostToRecommendPoolAuto(adminForumPost);
                }
            }
        } else {
            //删除推荐库
            if (Objects.nonNull(result)) {
                adminMixedRecommendPoolMapper.deleteByPrimaryKey(result.getId());
            }
        }
    }

    @Override
    public void cancelRecommend(Long postId) {
        this.adminMixedRecommendPoolMapper.deleteByPrimaryKey(postId);
    }
}
