package com.bxm.localnews.news.recommend.impl;

import com.bxm.localnews.news.config.NewsProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.TopicForumInfo;
import com.bxm.localnews.news.model.vo.TopicVo;
import com.bxm.localnews.news.recommend.TopicForumService;
import com.bxm.localnews.news.topic.ForumTopicService;
import com.bxm.localnews.news.vo.News4Client;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author pf.w
 * @date 2020/12/14 14:29
 **/
@Service
@AllArgsConstructor
@Log4j2
public class TopicForumServiceImpl implements TopicForumService {

    private NewsProperties newsProperties;

    private ForumPostMapper forumPostMapper;

    private RedisHashMapAdapter redisHashMapAdapter;

    private ForumTopicService forumTopicService;

    /**
     * 头条  本地圈 推荐话题插入位置
     */
    private static final int TOUTIAO_INDEX = 5;

    private static final int BENDIQUAN_INDEX = 3;


    @Override
    public List<TopicForumInfo> loadForumInfosByTopicId(Long topicId, String areaCode) {
        List<TopicForumInfo> result = new ArrayList<>();

        String subKey = topicId.toString().trim() + "#" + areaCode.trim();
        List<ForumPostVo> forumPostVos = forumPostMapper.getForumListByTopicId(topicId, newsProperties.getTopicForumLimit(), areaCode);
        if (CollectionUtils.isEmpty(forumPostVos)) {

            TopicForumInfo topicForumInfo = TopicForumInfo.builder()
                    .newsId(0L)
                    .headImg("null")
                    .img("null")
                    .nickName("null")
                    .userId(0L)
                    .build();
            result.add(topicForumInfo);
            //防止缓存穿透 放个假数据   由定时任务去更新
            redisHashMapAdapter.put(buildKey(), subKey, result);
            return result;
        }

        forumPostVos.forEach(forumPostVo -> {
            TopicForumInfo topicForumInfo = TopicForumInfo.builder()
                    .newsId(forumPostVo.getId())
                    .userId(forumPostVo.getUserId())
                    .nickName(Objects.nonNull(forumPostVo.getUserName()) ? forumPostVo.getUserName() : null)
                    .img(forumPostVo.getPostImgList().get(0).getImgUrl())
                    .headImg(Objects.nonNull(forumPostVo.getUserImg()) ? forumPostVo.getUserImg() : null)
                    .build();
            result.add(topicForumInfo);
        });

        redisHashMapAdapter.put(buildKey(), subKey, result);
        return result;
    }

    @Override
    public List<TopicForumInfo> getForumInfosByTopicId(Long topicId, String areaCode) {
        TypeReference<List<TopicForumInfo>> typeReference = new TypeReference<List<TopicForumInfo>>() {
        };
        List<TopicForumInfo> result = redisHashMapAdapter.get(buildKey(), topicId.toString().trim() + "#" + areaCode.trim(), typeReference);

        return CollectionUtils.isNotEmpty(result) ? result : this.loadForumInfosByTopicId(topicId, areaCode);
    }

    @Override
    public Object recommendTopic(Object param, Integer type, Long userId, String areaCode) {
        log.info("areaCode : {}",areaCode);
        //插入推荐话题
        List<TopicVo> topicVoList = forumTopicService.getTopicList(areaCode);
        if (CollectionUtils.isEmpty(topicVoList)) {
            return param;
        }
        log.info("==========recommendTopic 原始话题列表 ： {}", topicVoList);
        //取权重排名前几的话题
        if (topicVoList.size() > newsProperties.getRecommendTopicOrderNum()) {
            topicVoList = topicVoList.stream().limit(newsProperties.getRecommendTopicOrderNum()).collect(Collectors.toList());
        }
        log.info("==========recommendTopic 取权重前6话题列表 ： {}", topicVoList);

        //过滤掉不满足条件话题（话题下必须有三个带图片的帖子，否则不予推荐）
        topicVoList = topicVoList.stream()
                .filter(item -> {
                    List<TopicForumInfo> list1 = this.getForumInfosByTopicId(item.getId(), areaCode);
                    log.info("==========recommendTopic 帖子信息 ，topicId : {},带图列表 list1 : {}", item.getId(),list1);
                    item.setTopicForums(list1);
                    return list1.size() >= newsProperties.getTopicForumLimit();
                })
                .collect(Collectors.toList());
        log.info("==========recommendTopic 过滤后 ： {}", topicVoList);
        if (CollectionUtils.isEmpty(topicVoList)) {
            return param;
        }

        Integer order = redisHashMapAdapter.get(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), userId + "", Integer.class);
        if (null == order || ++order >= topicVoList.size()) {
            order = 0;
        }
        log.info("==========recommendTopic 顺序 order ： {}", order);
        redisHashMapAdapter.put(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), userId + "", order);

        TopicVo topic = topicVoList.get(order);
        Date now = new Date();
        //重新判断有奖是否失效
        topic.setHasReward((DateUtils.before(now, topic.getRewardEndTime()) && DateUtils.after(now, topic.getRewardStartTime())) ? 1 : 0);

        //头条信息流推荐
        if (type == 1) {
            News4Client news4Client = new News4Client();
            news4Client.setRecommendTopicVo(topic);
            List<News4Client> news4Clients = (List<News4Client>) param;

            if (news4Clients.size() >= TOUTIAO_INDEX) {
                news4Clients.add(TOUTIAO_INDEX - 1, news4Client);
            }
            return news4Clients;
        }

        //本地圈插入
        if (type == 2) {
            List<ForumPostVo> postList = (List<ForumPostVo>) param;
            ForumPostVo forumPostVo = new ForumPostVo();
            forumPostVo.setRecommendTopicVo(topic);
            if (postList.size() >= BENDIQUAN_INDEX) {
                postList.add(BENDIQUAN_INDEX - 1, forumPostVo);
            }
            return postList;
        }

        return param;
    }

    /**
     * 缓存key
     *
     * @return key
     */
    private KeyGenerator buildKey() {
        return RedisConfig.FORUM_LIMITNUM_TOPIC_NEW.copy();
    }
}
