package com.bxm.localnews.news.statistics.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.MissionIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.ReplyConfigProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.CalculatePostDTO;
import com.bxm.localnews.news.model.vo.*;
import com.bxm.localnews.news.statistics.ForumPostStatisticService;
import com.bxm.localnews.vo.VirtualUserInfo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.sync.core.CacheHolder;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
@Log4j2
public class ForumPostStatisticServiceImpl extends BaseService implements ForumPostStatisticService {

    @Resource
    private ForumTopicMapper forumTopicMapper;

    @Resource
    private NewsReplyMapper newsReplyMapper;

    @Resource
    private UserReplyMapper userReplyMapper;

    @Resource
    private ForumPostLikeMapper forumPostLikeMapper;

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private MissionIntegrationService missionIntegrationService;

    @Resource
    private ReplyConfigProperties replyConfigProperties;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private CacheHolder cacheHolder;

    @Override
    public void doConsumeRecommendedCount(List<Long> ids, Long userId) {
        List<ForumPostStatistic> forumPostStatisticList =
                ids.stream().map(id -> ForumPostStatistic.buildRecommend(id, 1)).collect(Collectors.toList());
        MybatisBatchBuilder.create(ForumPostMapper.class, forumPostStatisticList).run(ForumPostMapper::updateStatisticByPrimaryKeySelective);
    }


    @Override
    public void doSyncPostStaticInfo() {
        List<Long> list = forumPostMapper.selectNotExitReplyAndLikeInfo();

        List<ForumPostStatistic> paramList = Lists.newArrayList();
        list.forEach(postId -> {
            ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
            forumPostStatistic.setId(postId);

            List<NewsReply> replyList = newsReplyMapper.selectPostReplyByNewsIdOrderByAddTime(postId);
            if (CollectionUtils.isNotEmpty(replyList)) {
                forumPostStatistic.setReplyInfo(toJSONString(replyList));

            }
            List<ForumPostLike> forumPostLikes = forumPostLikeMapper.selectByPostIdAndOrderByAddtime(postId);
            if (CollectionUtils.isNotEmpty(forumPostLikes)) {
                forumPostStatistic.setLikeInfo(toJSONString(forumPostLikes));
            }
            boolean flag = !(StringUtils.isBlank(forumPostStatistic.getLikeInfo()) && StringUtils.isBlank(forumPostStatistic.getReplyInfo()));
            if (flag) {
                paramList.add(forumPostStatistic);
            }
        });

        MybatisBatchBuilder.create(ForumPostMapper.class, paramList).run(ForumPostMapper::updateStatisticByPrimaryKeySelective);
    }

    @Override
    public List<CalculatePostDTO> calculatePostCount() {
        List<CalculatePostDTO> result = Lists.newArrayList();
        //最近半小时发帖统计
        List<PostCount> newPostCount = forumPostMapper.selectNewPostCount();
        //未审核帖子数量
        List<PostCount> unreviewedCounts = forumPostMapper.selectPendingReviewCount();
        newPostCount.forEach(e -> {
            CalculatePostDTO calculatePostDTO = new CalculatePostDTO();
            if (StringUtils.isNotBlank(e.getAreaCode())) {
                LocationDTO location = locationIntegrationService.getLocationByGeocode(e.getAreaCode());
                calculatePostDTO.setAreaName(location.getName());
                calculatePostDTO.setAreaCode(e.getAreaCode());
            } else {
                calculatePostDTO.setAreaName("其他地区");
                calculatePostDTO.setAreaCode(StringUtils.EMPTY);
            }
            calculatePostDTO.setNewPostCount(e.getNum());
            Integer unReviewed = unreviewedCounts.stream().filter(k -> StringUtils.equalsIgnoreCase(e.getAreaCode(), k.getAreaCode()))
                    .findFirst().map(PostCount::getNum).orElse(0);
            calculatePostDTO.setPendingReviewCount(unReviewed);
            int userCount = forumPostMapper.selectUserCount(e.getAreaCode());
            calculatePostDTO.setUserCount(userCount);
            result.add(calculatePostDTO);
        });
        return result;
    }

    @Override
    public void calcParticipantsNum() {
        logger.info("calculate participants num start. ");

        calcAllParticipantsNum();
        calcAreaParticipantsNum();

        logger.info("calculate participants num end. ");
    }

    /**
     * 计算话题全国参与人数
     */
    private void calcAllParticipantsNum() {
        //计算话题参与人数
        List<TopicVo> topicVoList = forumTopicMapper.getTopicList(null, null);
        if (CollectionUtils.isNotEmpty(topicVoList)) {
            for (TopicVo topicVo : topicVoList) {
                Integer participantsNum = getParticipantsNum(topicVo.getId(), null);
                forumTopicMapper.updateTopic(topicVo.getId(), participantsNum);
            }
        }
    }

    /**
     * 计算话题区域参与人数
     */
    private void calcAreaParticipantsNum() {
        //获取版块话题投放区域
        List<ForumTopicAreaRelationVo> areaRelationVoList = forumTopicMapper.getAreaRelationList();

        if (CollectionUtils.isNotEmpty(areaRelationVoList)) {
            //根据区域计算参与人数
            areaRelationVoList.stream()
                    .filter(e -> null != e.getType() && null != e.getRelationId() && StringUtils.isNotBlank(e.getAreaCode()))
                    .forEach(e -> {
                        if ((byte) 2 == e.getType()) {
                            Integer participantsNum = getParticipantsNum(e.getRelationId(), e.getAreaCode());
                            forumTopicMapper.updateAreaParticipantsNum(e.getId(), participantsNum);
                        }
                    });
        }
    }

    private Integer getParticipantsNum(Long topicId, String areaCode) {
        Integer participantsNum = forumPostMapper.calcPostNum(null, topicId, areaCode);
        if (null == participantsNum) {
            participantsNum = 0;
        }

        List<Long> postIdList = forumPostMapper.getPostIdList(null, topicId, areaCode);
        if (CollectionUtils.isNotEmpty(postIdList)) {
            for (Long postId : postIdList) {
                Integer replyNum = newsReplyMapper.calcReplyNum(postId);
                if (replyNum != null) {
                    participantsNum += replyNum;
                }
            }
        }

        return participantsNum;
    }

    @Async
    @Override
    public void addPostReply(ForumPostVo forumPostVo) {
        try {
            //需求：评论显示时间：用户新人贴发布当天，马上跟上3-5个评论，然后在第二天跟上3-5个评论，然后第三天回复1-2个评论
            List<String> list = newsReplyMapper.selectRandomReplyLibrary(replyConfigProperties.getNewReportId(), 24);
            // 需要添加评论的数量 多给一点
            List<VirtualUserInfo> userList = userIntegrationService.getVirtualUserListNew(24,
                    Lists.newArrayList(1, 2, 3),
                    forumPostVo.getAreaCode());

            //马上跟上3-5个评论
            int choiceNum = ThreadLocalRandom.current().nextInt(3, 6);
            addNewsReplyDo(forumPostVo,
                    System.currentTimeMillis(),
                    subList(list, 0, choiceNum),
                    subList(userList, 0, choiceNum));

            //然后在第二天跟上3-5个评论
            choiceNum = ThreadLocalRandom.current().nextInt(3, 6);
            addNewsReplyDo(forumPostVo, (System.currentTimeMillis() + 3600 * 1000 * 24),
                    subList(list, 5, choiceNum),
                    subList(userList, 5, choiceNum));

            //然后第三天回复1-2个评论
            choiceNum = ThreadLocalRandom.current().nextInt(1, 2);
            addNewsReplyDo(forumPostVo, (System.currentTimeMillis() + 3600 * 1000 * 48),
                    subList(list, 10, choiceNum),
                    subList(userList, 10, choiceNum));

        } catch (Exception e) {
            log.error("增加评论失败, forumPostVo: {}", JSON.toJSONString(forumPostVo), e);
        }
    }

    private <T> List<T> subList(List<T> list, int start, int limit) {
        return list.stream().skip(start).limit(limit).collect(Collectors.toList());
    }

    private void addNewsReplyDo(ForumPostVo forumPostVo, long currentTime, List<String> list, List<VirtualUserInfo> userList) {
        if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(userList)) {
            return;
        }

        int i = 0;
        for (String replyContent : list) {
            int nextTime = ThreadLocalRandom.current().nextInt(120000);
            VirtualUserInfo virtualUser = userList.get(i);
            NewsReply newsReply = new NewsReply();
            newsReply.setAddTime(new Date(currentTime + nextTime));
            newsReply.setDeleteFlag((byte) 0);
            newsReply.setHeadImg(virtualUser.getHeadImg());
            newsReply.setId(nextSequence());
            newsReply.setLevel((byte) 0);
            newsReply.setLikeCount(0);
            newsReply.setNewsId(forumPostVo.getId());
            newsReply.setReplyContent(replyContent);
            //如果是帖子
            newsReply.setType((byte) 3);
            newsReply.setParentId(0L);
            //都设置为待展示，等待扫描
            newsReply.setStatus((byte) 0);
            newsReply.setUserId(virtualUser.getId());
            newsReply.setUserNickname(virtualUser.getNickname());
            newsReply.setInteractiveCount(0);
            newsReply.setRootId(0L);
            // 马甲号评论
            newsReply.setReplyUserType(1);
            newsReplyMapper.insertSelective(newsReply);
            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(newsReply, userReply);
            userReplyMapper.insertSelective(userReply);
            i++;
        }
    }

    @Override
    public void addCrumbsTotal(Integer deliveryType, String areaCode, Byte postType) {
        // 如果是全国贴则增加全国帖子的数量
        if (deliveryType != null && deliveryType == 0) {
            KeyGenerator crumbsNationalTotal = getNationalAreaTotal();
            // 增加普通帖子和小纸条的全国贴数量
            if (postType != null) {
                boolean exists = redisHashMapAdapter.exists(crumbsNationalTotal, postType.toString());
                // 如果存在则自增，如果不存在则初始化为1
                if (exists) {
                    redisHashMapAdapter.increment(crumbsNationalTotal, postType.toString(), 1);
                } else {
                    //初始消息数
                    final int initNum = 1;
                    redisHashMapAdapter.put(crumbsNationalTotal, postType.toString(), initNum);
                }
            }
        }
        // 如果是区域帖子或者小纸条，直接增加区域帖子或小纸条数量
        if (deliveryType != null && deliveryType == 1) {
            if (postType != null && isNotBlank(areaCode)) {
                String[] areaCodes = areaCode.split(",");
                logger.info("新增区域帖子或小纸条，postType:{},区域编码：{}", postType, areaCode);
                for (String areaCodeItem : areaCodes) {
                    KeyGenerator crumbsAreaTotal = getCrumbsAreaTotalKey(postType);
                    boolean exists = redisHashMapAdapter.exists(crumbsAreaTotal, areaCodeItem);
                    // 如果存在则自增，如果不存在则初始化为1
                    if (exists) {
                        redisHashMapAdapter.increment(crumbsAreaTotal, areaCodeItem, 1);
                    } else {
                        //初始消息数
                        final int initNum = 1;
                        redisHashMapAdapter.put(crumbsAreaTotal, areaCodeItem, initNum);
                    }
                }
            }
        }
    }

    /**
     * 获取面包块全国贴和小纸条新增消息数量缓存key
     *
     * @return 缓存key
     */
    private KeyGenerator getNationalAreaTotal() {
        return RedisConfig.NATIONAL_AREA_TOTAL.copy();
    }

    /**
     * 获取面包块新增消息数量缓存key
     *
     * @param type 面包块类型： 1：话题广场，2：同城交友
     * @return 缓存key
     */
    private KeyGenerator getCrumbsAreaTotalKey(int type) {
        return RedisConfig.CRUMBS_AREA_TOTAL.copy().appendKey(type);
    }
}
