package com.bxm.localnews.news.task;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.news.model.vo.ForumPostClickCountVo;
import com.bxm.localnews.news.model.vo.PostClickCountVo;
import com.bxm.localnews.news.post.ForumPostFacadeService;
import com.bxm.newidea.component.schedule.task.AbstractCronTask;
import com.bxm.newidea.component.vo.PageParam;
import com.xxl.job.core.biz.model.ReturnT;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.RandomUtils.nextInt;

/**
 * 3.8.0版本帖子阅读数计算
 * 具体规则 http://10.10.1.36:81/design/WST3.8.0/index.html#g=1&p=%E8%99%9A%E6%8B%9F%E7%82%B9%E8%B5%9E%EF%BC%8C%E9%98%85%E8%AF%BB%E9%87%8F%E9%80%BB%E8%BE%91%E8%B0%83%E6%95%B4&id=1ufouk
 */
@Component
@Slf4j
public class CalcPostClickCountNewTask extends AbstractCronTask<String> {
    private static final Integer BATCH_SIZE = 50;

    @Resource
    private ForumPostFacadeService forumPostFacadeService;

    @Override
    protected ReturnT<String> service(String param) {
        log.info(".....................................新版本（子陵版）帖子阅读数相关计算 begin...................................");
        PageParam page = new PageParam();
        int pager = 1;
        page.setPageSize(BATCH_SIZE);
        List<PostClickCountVo> postClickCountVos;
        byte yes = 1;

        do {
            page.setPageNum(pager++);
            postClickCountVos = forumPostFacadeService.getUnFullClickPosts(page);

            if (log.isDebugEnabled()) {
                log.debug("page: {} 获取到的需要计算帖子的列表: {}", pager - 1, JSON.toJSONString(postClickCountVos));
            }

            List<ForumPostClickCountVo> forumPostClickList = postClickCountVos.stream()
                    .map(p -> {
                        // 判断是否 爆料，热，精华，现金奖励
                        boolean isRecommend = Objects.equals(p.getIsRecommend(), yes);

                        int clickCount;
                        if (Objects.equals(p.getIsBroke(), yes)
                                || Objects.equals(p.getIsBrilliant(), yes)
                                || Objects.equals(p.getIsRed(), yes)
                                || Objects.equals(p.getIsCash(), yes)) {
                            // 爆料，热，精华，现金奖励 的帖子 每次增加 500-1000
                            clickCount = nextInt(500, 1001);
                        } else if (isRecommend) {
                            // 推荐库帖子 每次增加 100-500
                            clickCount = nextInt(100, 500);
                        } else {
                            // 普通帖子每次40-60
                            clickCount = nextInt(40, 60);
                        }

                        long currentClickCount = (Objects.isNull(p.getClickCount()) ? 0 : p.getClickCount());
                        long currentFinalClickCount = (Objects.isNull(p.getFinalClickCount()) ? 0 : p.getFinalClickCount());
                        // 判断本次增加的是否超过最终的
                        if (clickCount + currentClickCount > currentFinalClickCount) {
                            // 是的话 就只增加剩余需要增加的数量即可
                            clickCount = (int) (currentFinalClickCount - currentClickCount);
                        }

                        ForumPostClickCountVo forumPostClickCountVo = new ForumPostClickCountVo();
                        forumPostClickCountVo.setId(p.getId());
                        forumPostClickCountVo.setClickCount(clickCount);
                        return forumPostClickCountVo;
                    }).collect(Collectors.toList());

            if (CollectionUtils.isNotEmpty(forumPostClickList)) {
                forumPostFacadeService.batchAddClick(forumPostClickList);
            }

        } while (!CollectionUtils.isEmpty(postClickCountVos));

        log.info(".....................................新版本（子陵版）帖子阅读数相关计算 end...................................");
        return ReturnT.SUCCESS;
    }

    @Override
    public String taskName() {
        return "calcPostClickCountNewTask";
    }

    @Override
    public String cron() {
        return "30 1/1 * * * ? ";
    }

    @Override
    public String description() {
        return "新版本（子陵版）定时增加帖子阅读数任务";
    }

}
