package com.bxm.localnews.news.task;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.integration.MessageService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.model.vo.NoteExtraVo;
import com.bxm.localnews.vo.UserFuns;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.schedule.task.AbstractCronTask;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Lists;
import com.xxl.job.core.biz.model.ReturnT;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 创建即发布的周期定时任务
 * 定时发送小纸条的信息给用户
 */
@Component
@Slf4j
public class NoteLeadTask extends AbstractCronTask<String> {

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private MessageService messageService;

    @Override
    protected ReturnT<String> service(String param) {
        log.info(".....................................小纸条引导任务 begin...................................");
        //获取时间段内用户发的小纸条
        Date startTime;
        Date endTime = new Date();
        Integer hour = DateUtils.getField(endTime, Calendar.HOUR_OF_DAY);
        if (10 == hour) {
            startTime = DateUtils.addField(endTime, Calendar.HOUR_OF_DAY, -16);
        } else {
            startTime = DateUtils.addField(endTime, Calendar.HOUR_OF_DAY, -8);
        }

        List<NoteExtraVo> noteList = forumPostMapper.getNoteListByTime(startTime, endTime);
        if (!CollectionUtils.isEmpty(noteList)) {
            //映射用户小纸条
            Map<Long, List<NoteExtraVo>> userNoteMap = noteList.stream().collect(Collectors.groupingBy(NoteExtraVo::getUserId));
            //获取用户粉丝
            List<UserFuns> userFunsList = userIntegrationService.getUserFuns(Lists.newArrayList(userNoteMap.keySet()));

            //计算小纸条并发送对应用户
            if (!CollectionUtils.isEmpty(userFunsList)) {
                Map<Long, NoteExtraVo> sendUserNoteMap = new HashMap<>();
                for (UserFuns userFuns : userFunsList) {
                    Set<Long> chatNoteIdSet = redisSetAdapter.getAllMembers(
                            getUserReadNoteKey(userFuns.getFunsUserId(), userFuns.getUserId()), Long.class);
                    log.debug("用户关注信息：" + JSON.toJSONString(userFuns) + "，聊过的小纸条：" + JSON.toJSONString(chatNoteIdSet));

                    List<NoteExtraVo> userNoteList = userNoteMap.get(userFuns.getUserId());
                    if (!CollectionUtils.isEmpty(userNoteList)) {
                        for (NoteExtraVo note : userNoteList) {
                            //1.同城
                            if (StringUtils.isNotBlank(note.getAreaCode()) && !note.getAreaCode().contains(userFuns.getFunsAreaCode())) {
                                continue;
                            }

                            //2.没有被聊过
                            if (!CollectionUtils.isEmpty(chatNoteIdSet) && chatNoteIdSet.contains(note.getId())) {
                                continue;
                            }

                            //3.点击量最高，点击量一样时取最新的
                            NoteExtraVo sendNote = sendUserNoteMap.get(userFuns.getFunsUserId());
                            if (sendNote != null) {
                                if (sendNote.getClickCount() > note.getClickCount()) {
                                    continue;
                                }
                                if (sendNote.getClickCount().equals(note.getClickCount())
                                        && sendNote.getDisplayTime().after(note.getDisplayTime())) {
                                    continue;
                                }
                            }

                            sendUserNoteMap.put(userFuns.getFunsUserId(), note);
                        }
                    }
                }

                //发送消息
                if (!CollectionUtils.isEmpty(sendUserNoteMap)) {
                    sendUserNoteMap.forEach((userId, noteExtraVo) -> messageService.pushLeadNote(userId, noteExtraVo));
                }
            }
        }

        log.info(".....................................小纸条引导任务 end.....................................");
        return ReturnT.SUCCESS;
    }

    @Override
    public String taskName() {
        return "noteLeadTask";
    }

    @Override
    public String cron() {
        return "0 0 10,18 * * ? *";
    }

    @Override
    public String description() {
        return "小纸条引导任务";
    }

    private KeyGenerator getUserReadNoteKey(Long userId, Long targetUserId) {
        return RedisConfig.USER_READ_NOTE.copy().appendKey(userId + "_" + targetUserId);
    }
}
