package com.bxm.localnews.news.topic;


import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.news.model.vo.TopicVo;

import java.util.List;

/**
 * 帖子话题相关的业务逻辑，针对用户端
 * 话题均进行了二级缓存处理
 *
 * @author liujia
 */
public interface ForumTopicService {

    /**
     * 获取帖子所属的话题详情
     *
     * @param topicIds 逗号分隔的话题ID
     * @param location 地区信息
     * @param userId   当前请求用户
     * @return 话题列表
     */
    List<TopicVo> loadTopicByIds(String topicIds, LocationDTO location, Long userId);

    /**
     * 获取当前地区的话题列表
     *
     * @param size     推荐的话题数量，-1表示全部，2表示随机推荐两条
     * @param areaCode 地区编码
     * @param userId   请求用户ID
     * @param postPage 是否在发帖页面，1：是，0：否
     * @return 分页结果
     */
    List<TopicVo> listTopic(String areaCode, Integer size, Long userId, Integer postPage);

    /**
     * 获取地区投放的可用的话题列表
     * 先从缓存中取列表，若为空则查数据库
     *
     * @param areaCode 地区编码，为空则返回全国的数据
     * @return 地区投放话题列表
     */
    List<TopicVo> getTopicList(String areaCode);

    /**
     * 3.10.0 获取热门推荐话题
     *
     * @param areaCode 地区码
     * @return 话题集合
     */
    List<TopicVo> getHotTopicList(String areaCode);

    /**
     * 从缓存中获取话题详情
     *
     * @param id       话题ID
     * @param userId   请求用户
     * @param areaCode 地区编码
     * @return 话题详情，处理了占位符、分享链接等动态参数
     */
    TopicVo getTopicDetail(Long id, Long userId, String areaCode);

    /**
     * 获取话题详情
     * <p>
     * 3.10.0 需求新增 （话题增加二级分类；话题弹窗文案和弹出时机）
     *
     * @param id       话题id
     * @param userId   用户id
     * @param areaCode 地区码
     * @param type     3.10.0  发帖按钮过来的type值  910001：随便聊聊    910002：本地爆料  为详情弹窗用 可为空
     * @return 话题详情实体
     */
    TopicVo getTopicDetailNew(Long id, Long userId, String areaCode, Integer type);

    /**
     * 话题信息编辑后，移除话题相关的缓存
     *
     * @param topicId 话题ID
     */
    void removeCache(Long topicId);

    /**
     * 定期重新计算话题的缓存数据和信息
     * 包括话题的参与人数、话题的参与人头像等信息
     */
    void reloadCache();
}
