package com.bxm.localnews.news.topic.impl;

import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.LogicGroupConstant;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.model.vo.TopicVo;
import com.bxm.localnews.news.topic.ForumTopicService;
import com.bxm.localnews.news.topic.context.TopicContext;
import com.bxm.localnews.news.topic.filter.*;
import com.bxm.newidea.component.filter.FilterChainExecutor;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.sync.core.CacheHolder;
import lombok.AllArgsConstructor;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.MemoryCacheKey.TOPIC_AREA_CACHE;
import static com.bxm.localnews.news.constant.MemoryCacheKey.TOPIC_CACHE;
import static com.bxm.localnews.news.constant.RedisConfig.TOPIC_DETAIL;

/**
 * @author liujia
 */
@Service
@AllArgsConstructor
public class ForumTopicServiceImpl extends BaseService implements ForumTopicService {

    private ForumTopicMapper forumTopicMapper;

    private ForumProperties forumProperties;

    private RedisHashMapAdapter redisHashMapAdapter;

    private LocationIntegrationService locationIntegrationService;

    private FilterChainExecutor filterChainExecutor;

    private CacheHolder cacheHolder;

    public static final String DEFAULT_INVALID_TOPIC_ID = "0";

    @Override
    public List<TopicVo> loadTopicByIds(String topicIds, LocationDTO location, Long userId) {
        List<TopicVo> topicVoList = new ArrayList<>();

        if (StringUtils.isNotBlank(topicIds)) {
            //如果topicId只存在一个并且是0(无效的话题id，如果未选择话题，则会有一个id为0的话题)，则返回空的话题列表
            String[] topicIdArr = topicIds.split(",");
            if (topicIdArr.length == 1 && Objects.equals(topicIdArr[0], DEFAULT_INVALID_TOPIC_ID)) {
                return Collections.emptyList();
            }
            
            for (String topicId : topicIdArr) {
                TopicVo cacheTopic = getCacheTopic(Long.valueOf(topicId));

                TopicContext context = TopicContext.builder()
                        .topic(cacheTopic)
                        .location(location)
                        .userId(userId)
                        .build();

                filterChainExecutor.choice(TopicRuntimeFilter.class)
                        .choice(TopicPlaceHolderFilter.class)
                        .doFilter(LogicGroupConstant.TOPIC_FILL_FILTER, context);

                topicVoList.add(cacheTopic);
            }
        }

        return topicVoList;
    }

    @Override
    public List<TopicVo> listTopic(String areaCode, Integer size, Long userId, Integer postPage) {
        if (StringUtils.isBlank(areaCode)) {
            logger.warn("根据地区获取话题列表 区域编码为空");
            return Collections.emptyList();
        }

        if (Objects.isNull(userId)) {
            logger.warn("根据地区获取话题列表 用户id为空");
            return Collections.emptyList();
        }

        List<TopicVo> topicVoList = getTopicList(areaCode);

        // 发帖页面屏蔽小纸条话题
        if (Objects.equals(postPage, 1)) {
            Long noteTopicId = forumProperties.getNoteTopicId();
            for (TopicVo topicVo : topicVoList) {
                if (Objects.equals(topicVo.getId(), noteTopicId)) {
                    topicVoList.remove(topicVo);
                    break;
                }
            }
        }

        //如果数量传了2，则进行随机取,扯淡的需求
        if (size != null && size == 2) {
            Collections.shuffle(topicVoList);
            return topicVoList.stream().limit(size).collect(Collectors.toList());
        }

        return topicVoList;
    }

    @Override
    public List<TopicVo> getTopicList(String areaCode) {
        if (StringUtils.isBlank(areaCode)) {
            areaCode = "0";
        }

        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        List<Long> topicIdList = cacheHolder.get(TOPIC_AREA_CACHE, areaCode);

        List<TopicVo> result = Lists.newArrayList();

        for (Long topicId : topicIdList) {
            TopicVo cacheTopic = getCacheTopic(topicId);

            TopicContext context = TopicContext.builder()
                    .topic(cacheTopic)
                    .location(location)
                    .build();

            filterChainExecutor.choice(TopicRuntimeFilter.class)
                    .choice(TopicPlaceHolderFilter.class)
                    .choice(TopicParticipantNumFilter.class)
                    .doFilter(LogicGroupConstant.TOPIC_FILL_FILTER, context);

            result.add(cacheTopic);
        }

        return result;
    }


    @Override
    public List<TopicVo> getHotTopicList(String areaCode) {
        return getTopicList(areaCode).stream()
                .filter(topicVo -> topicVo.getHot() == 1)
                .collect(Collectors.toList());
    }

    @Override
    public TopicVo getTopicDetail(Long id, Long userId, String areaCode) {
        TopicVo topicVo = getCacheTopic(id);

        TopicContext context = TopicContext.builder()
                .topic(topicVo)
                .location(locationIntegrationService.getLocationByGeocode(areaCode))
                .userId(userId)
                .build();

        filterChainExecutor.choice(TopicRuntimeFilter.class)
                .choice(TopicPlaceHolderFilter.class)
                .choice(TopicParticipantNumFilter.class)
                .doFilter(LogicGroupConstant.TOPIC_FILL_FILTER, context);

        return topicVo;
    }

    @Override
    public TopicVo getTopicDetailNew(Long id, Long userId, String areaCode, Integer type) {
        TopicContext context = TopicContext.builder()
                .topic(getCacheTopic(id))
                .location(locationIntegrationService.getLocationByGeocode(areaCode))
                .userId(userId)
                .popType(type)
                .build();

        filterChainExecutor.choice(TopicRuntimeFilter.class)
                .choice(TopicPlaceHolderFilter.class)
                .choice(TopicGuidePopFilter.class)
                .doFilter(LogicGroupConstant.TOPIC_FILL_FILTER, context);

        return context.getTopic();
    }

    @Override
    public void removeCache(Long topicId) {
        loadTopicToRedis(topicId);

        cacheHolder.sendClearCmd(TOPIC_AREA_CACHE);
        cacheHolder.sendEvictCmd(TOPIC_CACHE, topicId.toString());
    }

    @Override
    public void reloadCache() {
        Set<String> keys = redisHashMapAdapter.keys(TOPIC_DETAIL);

        for (String key : keys) {
            loadTopicToRedis(Long.valueOf(key));
        }
    }

    /**
     * 获取话题缓存信息，读取频率非常高，数量量较少，这里使用二级缓存实现
     * 占位符未替换、实时信息未填充、不同地区的参与人数未提供
     *
     * @param topicId 话题ID
     * @return 话题详情
     */
    private TopicVo getCacheTopic(Long topicId) {
        return cacheHolder.get(TOPIC_CACHE, topicId.toString());
    }

    /**
     * 初始化二级缓存
     */
    @PostConstruct
    public void init() {
        cacheHolder.set(TOPIC_CACHE, (id) -> {
            TopicVo topic = redisHashMapAdapter.get(TOPIC_DETAIL, id, TopicVo.class);

            if (topic == null) {
                topic = loadTopicToRedis(Long.valueOf(id));
            }

            return topic;
        }, 1000L, TimeUnit.MINUTES, 5);

        cacheHolder.set(TOPIC_AREA_CACHE, this::getAreaTopicList,
                500L,
                TimeUnit.MINUTES,
                2);
    }

    /**
     * 获取地区可用的话题列表
     *
     * @param areaCode 地区编码
     * @return 地区的话题列表
     */
    private List<Long> getAreaTopicList(String areaCode) {
        if (StringUtils.isBlank(areaCode) || "0".equals(areaCode)) {
            areaCode = null;
        }
        List<TopicVo> topicList = forumTopicMapper.getTopicList(areaCode, 1);
        return topicList.stream().map(TopicVo::getId).collect(Collectors.toList());
    }

    private TopicVo loadTopicToRedis(Long topicId) {
        TopicVo topic = forumTopicMapper.selectTopicDetail(topicId);

        if (null == topic) {
            topic = new TopicVo();
            topic.setId(topicId);
            return topic;
        }

        TopicContext context = new TopicContext();
        context.setTopic(topic);

        filterChainExecutor.choice(TopicJoinerFilter.class)
                .choice(TopicSecondInfoFilter.class)
                .doFilter(LogicGroupConstant.TOPIC_FILL_FILTER, context);

        redisHashMapAdapter.put(TOPIC_DETAIL, topicId.toString(), topic);
        return topic;
    }
}
