package com.bxm.localnews.news.topic.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.news.config.NewsProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.model.vo.TopicVo;
import com.bxm.localnews.news.post.ForumPostFacadeService;
import com.bxm.localnews.news.topic.TopicPopService;
import com.bxm.localnews.news.vo.TopicPopInfoBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * @author liujia
 * @date 1/21/21 7:03 PM
 **/
@Service
@AllArgsConstructor
public class TopicPopServiceImpl implements TopicPopService {

    private RedisSetAdapter redisSetAdapter;

    private NewsProperties newsProperties;

    private ForumPostFacadeService forumPostFacadeService;

    private LocationIntegrationService locationIntegrationService;

    @Override
    public void loadTopicPop(TopicVo topicInfo, Integer type, Long userId, LocationDTO location) {
        //配置取弹窗数据
        List<TopicPopInfoBean> popInfo = JSONObject.parseArray(newsProperties.getTopicPopInfo(), TopicPopInfoBean.class);
        if (Objects.isNull(popInfo)) {
            return;
        }

        Long topicId = topicInfo.getId();

        //判断是否要弹窗
        for (TopicPopInfoBean topicPopInfoBean : popInfo) {
            if (Objects.equals(type, topicPopInfoBean.getPopType())
                    || Objects.equals(topicPopInfoBean.getTopicId(), topicInfo.getId())) {
                boolean hasPopRecord = redisSetAdapter.exists(buildKeyForPop(topicId, topicPopInfoBean.getPopType()), userId);

                if (hasPopRecord) {
                    topicInfo.setGuidePopFlag(0);
                } else {
                    // 为了避免和新人弹窗冲突，这里判断有发帖纪录才弹出（其实是为了在新人帖之后）
                    boolean hasPostRecord = forumPostFacadeService.getPublishPostNum(userId) > 0;
                    topicInfo.setGuidePopFlag(hasPostRecord ? 1 : 0);
                }

                if (topicInfo.getGuidePopFlag() == 1) {
                    topicInfo.setType(topicPopInfoBean.getPopType());
                    topicInfo.setGuidePopText(this.replace(topicPopInfoBean.getPopText(), location.getCode()));
                    guidePopSaveToRedis(userId, topicId, topicPopInfoBean.getPopType());
                }
                break;
            }
        }

    }


    private String replace(String popText, String areaCode) {
        //推荐语地区占位符替换
        String areaName = "全国";
        if (StringUtils.isNotBlank(areaCode)) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);
            if (locationDTO != null && StringUtils.isNotBlank(locationDTO.getName())) {
                areaName = locationDTO.getName();
            }
        }
        return popText.replace("{areaname}", areaName);
    }

    /**
     * 记录用户是否已经进行过该类型的弹窗
     *
     * @param userId  用户ID
     * @param topicId 话题ID
     */
    private void guidePopSaveToRedis(Long userId, Long topicId, Integer type) {
        redisSetAdapter.add(buildKeyForPop(topicId, type), userId);
        redisSetAdapter.expire(buildKeyForPop(topicId, type), 24 * 60 * 60 * 30 * 12);
    }

    /**
     * 3.10.0 话题发帖弹窗key
     *
     * @param topicId 话题ID
     * @return key
     */
    private KeyGenerator buildKeyForPop(Long topicId, Integer type) {
        if (Objects.nonNull(type)) {
            return RedisConfig.TOPIC_DETAIL_GUIDE_HAS_POP_LIST.copy().appendKey(topicId.toString() + "#" + type.toString());
        } else {
            return RedisConfig.TOPIC_DETAIL_GUIDE_HAS_POP_LIST.copy().appendKey(topicId.toString() + "#" + "0");
        }
    }
}
