package com.bxm.localnews.news.util;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.news.model.vo.AdminForumPost;
import com.bxm.localnews.news.model.vo.PostImg;
import com.bxm.newidea.component.enums.FileTypeEnum;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.bxm.newidea.component.tools.HtmlContentUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author stephen
 * @desc 内容处理工具类
 */
@Slf4j
@Component
public class DealContentUtil {

    @Resource
    private AliyunOSSService aliyunOSSService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    /**
     * 解析帖子内容
     */
    public void analyzeContent(AdminForumPost adminForumPost) {
        if (StringUtils.isNotBlank(adminForumPost.getContent())) {
            //提取视频和图片
            List<PostImg> postImgList = new ArrayList<>();
            List<String> imgList = new ArrayList<>();
            String content = extractVideoAndImg(adminForumPost.getContent(), postImgList, imgList);
            adminForumPost.setContent(content);
            List<PostImg> postImgTempList = imgList.stream().map(PostImg::buildImg).collect(Collectors.toList());
            postImgList.addAll(postImgTempList);
            adminForumPost.setImgList(JSONObject.toJSONString(postImgList));
            //提取文案
            adminForumPost.setTextField(this.getForumPostTextFiled(content));
        }
    }

    /**
     * 处理帖子中的textFiled不能换行
     *
     * @param content
     * @return
     */
    private String getForumPostTextFiled(String content) {
        Document document = Jsoup.parse(content);
        Elements elements = document.select("p");
        StringBuilder sb = new StringBuilder();
        for (Element element : elements) {
            String text = element.text();
            String result = HtmlContentUtils.delHtmlTags(text);
            if (StringUtils.isNotBlank(result)) {
                if (sb.length() == 0) {
                    sb.append(result);
                } else {
                    sb.append("\n").append(result);
                }
            }
        }
        return sb.toString();
    }

    /**
     * 分析文章内容填补video标签中的封面
     *
     * @param content
     * @param postImgList
     * @return
     */
    public String extractVideoAndImg(String content, List<PostImg> postImgList, List<String> imgList) {
        Document document = Jsoup.parse(content);
        document.outputSettings().prettyPrint(true).indentAmount(1);

        //处理视频
        this.dealWithVideo(document, postImgList);

        //处理图片并提取
        this.dealWithImg(document, imgList);

        return document.body().html().replace("\n", "");
    }

    /**
     * 处理视频
     *
     * @param document    页面元素
     * @param postImgList 视频列表
     */
    private void dealWithVideo(Document document, List<PostImg> postImgList) {
        Elements elements = document.select("video");
        if (null != elements) {
            for (Element element : elements) {
                String src = element.attr("src");
                if (StringUtils.isBlank(src)) {
                    continue;
                }
                String poster = element.attr("poster");
                if (StringUtils.isEmpty(poster)) {
                    String url = src;
                    int lastIndex = src.lastIndexOf("?");
                    if (-1 != lastIndex) {
                        url = src.substring(0, lastIndex);
                    }
                    String baseName = FilenameUtils.getBaseName(url);

                    KeyGenerator videoCacheKey = DefaultKeyGenerator.build("cache", "static", "videoinfo");
                    @SuppressWarnings("unchecked")
                    Map<String, Object> videoCacheInfo = redisHashMapAdapter.get(videoCacheKey, baseName, Map.class);
                    if (videoCacheInfo != null && !videoCacheInfo.isEmpty()) {
                        String size = videoCacheInfo.get("size") == null ? "0" : videoCacheInfo.get("size").toString();
                        String duration = videoCacheInfo.get("duration") == null ? "0" : videoCacheInfo.get("duration").toString();
                        String imgUrl = videoCacheInfo.get("imgUrl") == null ? "" : videoCacheInfo.get("imgUrl").toString();
                        element.attr("poster", imgUrl);
                        element.attr("size", size);
                        element.attr("duration", duration);
                        log.debug("video标签经处理后:{}", element.html());
                    }
                }

                PostImg img = new PostImg();
                img.setType("VIDEO");
                img.setImgUrl(element.attr("poster"));
                img.setVideoUrl(element.attr("src"));
                img.setSize(element.attr("size") == null ? "0" : element.attr("size"));
                img.setDuration(element.attr("duration") == null ? 0 : NumberUtils.toLong(element.attr("duration"), 0));
                postImgList.add(img);
            }
        }
    }

    /**
     * 处理图片，将第三方图片上传至服务器，将图片收集至list
     *
     * @param document 页面元素
     * @param imgList  图片列表
     */
    public void dealWithImg(Document document, List<String> imgList) {
        Elements imgElements = document.select("img");
        if (null != imgElements) {
            for (Element element : imgElements) {
                boolean isExist = true;
                String src = element.attr("src");
                if (StringUtils.isBlank(src)) {
                    isExist = false;
                    src = element.attr("data-src");
                    if (StringUtils.isNotBlank(src)) {
                        isExist = true;
                    }
                }
                if (isExist) {
                    if (!src.contains("wstong.com")) {
                        src = this.uploadImgToOssBySrc(src);
                        if (StringUtils.isBlank(src)) {
                            element.remove();
                            continue;
                        }
                        element.attr("src", src);
                    }

                    int lastIndex = src.lastIndexOf("?");
                    if (-1 != lastIndex) {
                        src = src.substring(0, lastIndex);
                        element.attr("src", src);
                    }

                    imgList.add(src);
                } else {
                    element.remove();
                }

            }
        }
    }

    /**
     * 处理内容，将图片加上样式
     *
     * @param content 内容
     * @return 处理后内容
     */
    public String fixContent(String content) {
        Document document = Jsoup.parse(content);
        document.outputSettings().prettyPrint(true).indentAmount(1);
        Elements elements = document.select("img");
        if (null != elements) {
            for (Element element : elements) {
                String src = element.attr("src");
                if (StringUtils.isBlank(src)) {
                    src = element.attr("data-src");
                    if (StringUtils.isBlank(src)) {
                        continue;
                    }
                }
                String url = src;
                int lastIndex = src.lastIndexOf("?");
                if (-1 != lastIndex) {
                    url = src.substring(0, lastIndex);
                }
                url += "?x-oss-process=style/mcompress";
                element.attr("src", url);
            }
        }
        return document.body().html().replace("\n", "");
    }


    /**
     * 上传第三方图片
     *
     * @param src 图片链接
     * @return 上传后链接
     */
    public String uploadImgToOssBySrc(String src) {
        final RestTemplate template = new RestTemplate();
        HttpHeaders headers = new HttpHeaders();
        HttpEntity<Resource> httpEntity = new HttpEntity<>(headers);
        try {
            src = URLDecoder.decode(src, "UTF-8");

            log.debug("src:[{}]", src);

            ResponseEntity<byte[]> response = template.exchange(src, HttpMethod.GET, httpEntity, byte[].class);
            HttpHeaders responseHeaders = response.getHeaders();
            //获取文件类型
            List<String> fileTypes = responseHeaders.get("content-type");
            String fileType;
            if (!CollectionUtils.isEmpty(fileTypes)) {
                fileType = fileTypes.get(0);
                fileType = getExtNameByContentType(fileType);
                if (StringUtils.isBlank(fileType)) {
                    fileType = "jpg";
                }
            } else {
                fileType = "jpg";
            }
            String fileName = RandomStringUtils.random(8, true, false) + "." + fileType;
            return aliyunOSSService.upload(response.getBody(), fileName);
        } catch (Exception e) {
            log.error("第三方图片上传出错", e);
        }
        return null;
    }

    private String getExtNameByContentType(String contentType) {
        FileTypeEnum[] values = FileTypeEnum.values();
        for (FileTypeEnum fileType : values) {
            if (fileType.getContentType().equals(contentType)) {
                return fileType.getExtName();
            }
        }
        return null;
    }
}
