package com.bxm.localnews.news.util;

import org.apache.commons.collections.CollectionUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * @author stephen 2019年12月12日 下午1:56:28
 * @ClassName ImportUtils.java
 * @Description 功能描述：通用输出方法
 * @CopyRight 杭州微财网络科技有限公司
 */
public class ImportUtils {
    private static final Logger logger = LoggerFactory.getLogger(ImportUtils.class);

    private static final String EXCEL_2003_REGEX = "^.+\\.(?i)(xls)$";

    private static final String EXCEL_2007_REGEX = "^.+\\.(?i)(xlsx)$";

    /**
     * 根据excel文件读取信息
     *
     * @param file
     * @return
     */
    public static List<List<String>> readExcelFile(MultipartFile file) {
        String fileName = file.getOriginalFilename();
        logger.debug("导入文件名：" + fileName);
        try {
            if (null == fileName) {
                return null;
            }
            boolean isExcel2003 = fileName.matches(EXCEL_2003_REGEX);
            boolean isExcel2007 = fileName.matches(EXCEL_2007_REGEX);
            if (!isExcel2003 && !isExcel2007) {
                logger.warn("导入文件格式不是excel");
                return null;
            }
            return createExcel(file.getInputStream(), isExcel2003);
        } catch (Exception e) {
            logger.error("excel解析失败：", e);
        }
        return null;
    }

    /**
     * 根据excel文件流读取信息
     *
     * @param is          输入流
     * @param isExcel2003 excel是2003还是2007版本
     * @return
     */
    public static List<List<String>> createExcel(InputStream is, boolean isExcel2003) {
        try {
            Workbook wb;
            if (isExcel2003) {
                wb = new HSSFWorkbook(is);
            } else {
                wb = new XSSFWorkbook(is);
            }
            return readExcelValue(wb);
        } catch (IOException e) {
            logger.error("excel初始化失败：", e);
        }
        return null;
    }

    private static List<List<String>> readExcelValue(Workbook wb) {
        logger.debug("开始解析excel数据...");
        // 得到第一个sheet
        Sheet sheet = wb.getSheetAt(0);
        // 得到Excel的行数
        int totalRows = sheet.getPhysicalNumberOfRows();
        // 得到Excel的列数(前提是有行数)
        int totalCells = 0;
        if (totalRows > 1 && sheet.getRow(0) != null) {
            totalCells = sheet.getRow(0).getPhysicalNumberOfCells();
        }
        logger.debug("行数：" + totalRows + "列数：" + totalCells);
        List<List<String>> resultList = new ArrayList<>();
        // 循环Excel行数
        for (int r = 1; r < totalRows; r++) {
            Row row = sheet.getRow(r);
            if (row == null) {
                continue;
            }
            // 循环Excel的列
            List<String> list = new ArrayList<>();
            for (int c = 0; c < totalCells; c++) {
                Cell cell = row.getCell(c);
                if (null != cell) {
                    // 如果是纯数字,比如你写的是25,cell.getNumericCellValue()获得是25.0,通过截取字符串去掉.0获得25
                    if (cell.getCellType() == CellType.NUMERIC) {
                        String name = String.valueOf(cell.getNumericCellValue());
                        list.add(name.substring(0, name.length() - 2 > 0 ? name.length() - 2 : 1));
                    } else {
                        list.add(cell.getStringCellValue());
                    }
                }
            }
            // 添加到resultList
            if (CollectionUtils.isNotEmpty(list)) {
                logger.debug("单列数据：" + list);
                resultList.add(list);
            }
        }
        return resultList;
    }

}
