package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.dto.MixRecommendDTO;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.mq.common.constant.UserRecommendEnum;
import com.bxm.localnews.mq.common.model.dto.UserRecommendDTO;
import com.bxm.localnews.news.config.*;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.convert.impl.ForumBasicConvert;
import com.bxm.localnews.news.convert.impl.ForumParamPageConvert;
import com.bxm.localnews.news.convert.impl.ForumPostConvert;
import com.bxm.localnews.news.convert.impl.ForumPostCreateConvert;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.*;
import com.bxm.localnews.news.enums.*;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.*;
import com.bxm.localnews.news.service.*;
import com.bxm.localnews.news.util.ListUtil;
import com.bxm.localnews.news.util.VersionUtils;
import com.bxm.localnews.news.util.JudgeUtil;
import com.bxm.localnews.news.vo.*;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.localnews.param.ForumParam;
import com.bxm.localnews.param.VotePinParam;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.redis.HyperLogLogAdapter;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageParam;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

@Service
public class ForumPostServiceImpl extends BaseService implements ForumPostService {

    @Resource
    private ForumPostInformMapper forumPostInformMapper;

    @Resource
    private ForumPostLikeMapper forumPostLikeMapper;

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private ForumTopicMapper forumTopicMapper;

    @Resource
    private MixedRecommendPoolMapper mixedRecommendPoolMapper;

    @Resource
    private NewsCollectMapper newsCollectMapper;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private NewsRecommendIntegrationService newsRecommendIntegrationService;

    @Resource
    private BizLogIntegrationService bizLogIntegrationService;

    @Resource
    private VoteIntegrationService voteIntegrationService;

    @Resource
    private UserRecommendSourceIntegrationService userRecommendSourceIntegrationService;

    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private ForumBasicConvert forumBasicConvert;

    @Resource
    private ForumParamPageConvert forumParamPageConvert;

    @Resource
    private ForumPostCreateConvert forumPostCreateConvert;

    @Resource
    private NewsReadRewardService newsReadRewardService;

    @Resource
    private NewsCollectService collectService;

    @Resource
    private ForumTopicService forumTopicService;

    @Resource
    private ForumPostStatisticService forumPostStatisticService;

    @Resource
    private SensitiveWordService sensitiveWordService;

    @Resource
    private IUrlFactory iUrlFactory;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private UserProperties userProperties;

    @Resource
    private NewsProperties newsProperties;

    @Resource
    private ForumProperties forumProperties;

    @Resource
    private BizConfigProperties bizConfigProperties;

    @Resource
    private HyperLogLogAdapter hyperLogLogAdapter;

    @Override
    public PageWarper<ForumPostVo> listForumPost(ForumPostListQueryParam param, BasicParam basicParam) {
        logger.debug("获取推荐流,param:{}", JSONObject.toJSON(param));

        PageWarper<ForumPostVo> forumPostPage = new PageWarper<>();

        List<Long> postIdList;
        //如果是小程序审核
        if (5 == basicParam.getPlatform() && bizConfigProperties.getAppletReviewEnable() && 3 == param.getType()) {
            postIdList = forumPostMapper.listPostRecommend(param);
        } else {
            //每访问一次板块详情，对其进行记录
//            if (param.getRecommendType() == 1 && param.getForumId() != null) {
//                this.forumService.updateForum(param.getForumId(), param.getUserId());
//            }

            //包装参数，从推荐引擎中获取相应的帖子id
            ForumParam forumParam = forumParamPageConvert.convert(param);
            fixErrorParamWithIOS(forumParam, basicParam);
            Objects.requireNonNull(forumParam).setPlatform(basicParam.getPlatform());
            postIdList = newsRecommendIntegrationService.recommendList(forumParam);

        }

        //组装帖子内容
        if (CollectionUtils.isNotEmpty(postIdList)) {
            List<ForumPostVo> postList = forumPostMapper.listPostByIds(postIdList);
            forumPostPage.setList(postList);
            forumPostPage.setPageNum(param.getPageNum());
            forumPostPage.setPageSize(param.getPageSize());
            //组装帖子内容详情
            fillExtInfo(forumPostPage.getList(), param.getUserId(), param.getAreaCode(), false, (byte) 0);

            //若为社区首页且数量大于3且版本号大于2.5.0，则插入推荐话题
            if (3 == param.getType() && postIdList.size() >= 3 && VersionUtils.isHighVersion(basicParam.getCurVer(), "2.5.0") > 0) {
                addRecommendTopic(param, forumPostPage, postList);
            }
        }

        //组装分页参数
        if (CollectionUtils.isEmpty(postIdList) || postIdList.size() < param.getPageSize()) {
            forumPostPage.setIsLastPage(true);
            forumPostPage.setHasNextPage(false);
        } else {
            forumPostPage.setHasNextPage(true);
        }

        logger.debug("[listForumPost]拉取帖子列表完毕");
        return forumPostPage;
    }

    @Override
    public PageWarper<ForumPostVo> listForumPostByUser(ForumPostListUserQueryParam param) {
        PageWarper<ForumPostVo> forumPostWarper = new PageWarper<>();

        if (null == param || null == param.getType()) {
            return null;
        }
        //我的发布
        if (1 == param.getType()) {
            forumPostWarper = new PageWarper<>(forumPostMapper.listPostByIdsInUser(param));
            List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
            if (CollectionUtils.isNotEmpty(forumPostVoList)) {
                this.fillExtInfo(forumPostVoList, param.getUserId(), null, false, (byte) 0);
            }
            //我的收藏（旧版本）
        } else if (2 == param.getType()) {
            forumPostWarper = new PageWarper<>(forumPostMapper.getCollectPostList(param));
            List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
            if (CollectionUtils.isNotEmpty(forumPostVoList)) {
                this.fillExtInfo(forumPostVoList, param.getUserId(), null, true, (byte) 0);
            }
        } else if (3 == param.getType()) {
            //IM内人分享调用
            HomePagePostParam postParam = new HomePagePostParam();
            postParam.setTargetUserId(param.getUserId());
            postParam.setUserId(param.getUserId());
            BeanUtils.copyProperties(param,postParam);
            forumPostWarper = new PageWarper<>(forumPostMapper.listHomePagePostList(postParam));
            List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
            if (CollectionUtils.isNotEmpty(forumPostVoList)) {
                this.fillExtInfo(forumPostVoList, param.getUserId(), null, false, (byte) 2);
            }
        }

        return forumPostWarper;
    }

    @Override
    public ForumPostVo getForumPostDetail(Long id, Long userId, Long shareUserId, BasicParam basicParam, String areaCode, String ip) {
        ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(id);

        if (null != forumPostVo) {
            //如果是站外访问
            if (PlatformEnum.WEB.getCode() == basicParam.getPlatform() && 2 == forumPostVo.getStatus()) {
                KeyGenerator keyGenerator = RedisConfig.FORUM_RECORD_FROM_H5.copy().appendKey(forumPostVo.getId());
                Long recordNum = hyperLogLogAdapter.size(keyGenerator);
                if (null != recordNum && recordNum >= forumProperties.getH5LimitNum()) {
                    ForumPostVo emptyForumPostVo = new ForumPostVo();
                    emptyForumPostVo.setStatus(100);
                    emptyForumPostVo.setId(id);
                    return emptyForumPostVo;
                }
            }

            //下架或者删除的帖子对H5不展示
            boolean isNotClient = PlatformEnum.ANDROID.getCode() != basicParam.getPlatform()
                    && PlatformEnum.IOS.getCode() != basicParam.getPlatform();
            boolean isDeleted = null != forumPostVo.getStatus()
                    && (3 == forumPostVo.getStatus() || 4 == forumPostVo.getStatus());
            if (isNotClient && isDeleted) {
                return null;
            }

            //得到地区
            LocationDTO location = StringUtils.isNotEmpty(areaCode) ? locationIntegrationService.getLocationByGeocode(areaCode) : null;

            //填充属性（详情）
            this.completePostInfo(forumPostVo, userId, location);
            //对内容进行处理（编辑寄语，水印等）
            forumPostStatisticService.generateForumPostContent(forumPostVo, basicParam, ip, userId, areaCode);
            //判断帖子中是否开启占位符
            forumPostStatisticService.judgeEnablePlaceholder(forumPostVo, location);
            //若黑色章的开关置为【黑色章】，且帖子打上了现金奖励标，则现金奖励标志不显示
            setIsCashAndCashRewardIsZero(forumPostVo);
            //异步执行 1.记录帖子阅读记录 2.帖子分享,在在小程序中每日首次打开,分享人随机获得20-50的小红花
            SpringContextHolder.getBean(ForumPostStatisticService.class).doAsyncReadPost(userId, id, shareUserId, basicParam.getPlatform(), forumPostVo,ip);
        }

        return forumPostVo;
    }

    @Override
    public Message doCreateOrUpdatePost(ForumBasicVo forumBasicVo, BasicParam basicParam, boolean isDirectlyPassed) {
        logger.debug("用户发帖或者编辑，参数:[{}]", JSONObject.toJSONString(forumBasicVo));
        if (sensitiveWordService.contains(forumBasicVo.getTextField())) {
            return Message.build(false, "发布内容包含敏感词");
        }
        Message message = Message.build(true);

        //转换实体
        ForumPostVo forumPostVo = forumBasicConvert.convert(forumBasicVo);
        if (isDirectlyPassed) {
            forumPostVo.setStatus(1);
        }

        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        if (null == forumPostVo.getId()) {
            //设置是否用户首次发帖
            forumPostVo.setIsFirstUserPost((byte) (forumPostMapper.selectPublishPostNumByUserId(forumBasicVo.getUserId()) == 0 ? 1 : 0));

            //保存帖子至数据库
            int i = this.saveForumPost(forumPostVo);
            if (i > 0) {
                //更新话题信息
                this.forumTopicService.updateTopic(forumPostVo);
                //创建后的一些操作
                newsCompleTaskDTO = afterCreate(basicParam, forumPostVo);
            }
        } else {
            //无法修改推荐库中内容
            MixedRecommendPool mixedRecommendPool = mixedRecommendPoolMapper.selectByPrimaryKey(forumPostVo.getId());
            if (Objects.nonNull(mixedRecommendPool)) {
                return Message.build(false, "您的内容已被推荐到头条，如需修改请联系客服");
            } else {
                ForumPostVo exitsPost = forumPostMapper.selectByPrimaryKey(forumPostVo.getId());
                //增加用户修改标识
                if (exitsPost.getStatus() != 2) {
                    forumPostVo.setIsUserUpdate((byte) 1);
                }
                this.mixedRecommendPoolMapper.deleteMixRecommandPoolById(forumPostVo.getId());
                forumPostVo.setIsRecommend((byte) 0);
                forumPostVo.setModifyTime(new Date());
                this.forumPostMapper.updateByPrimaryKeySelective(forumPostVo);

                //更新话题信息
                this.forumTopicService.updateTopic(forumPostVo);
            }
        }

        //视频转码（异步）
        SpringContextHolder.getBean(ForumPostFacadeService.class).transcodePostVideo(forumPostVo.getId(), forumPostVo.getPostImgList());

        //拼装返回实体
        ForumPostCreateDTO forumPostCreateDTO = forumPostCreateConvert.generateForumPost(forumPostVo, newsCompleTaskDTO, forumBasicVo.getUserId());
        message.addParam("completeTaskAndPush", forumPostCreateDTO);

        //上报发帖信息到用户推荐数据源服务
        UserRecommendDTO userRecommendDTO = new UserRecommendDTO();
        userRecommendDTO.setUserId(forumBasicVo.getUserId());
        userRecommendDTO.setRecommendType(UserRecommendEnum.USER_BEHAVIOR.getTag());
        userRecommendDTO.setLastPostTime(new Date());
        userRecommendSourceIntegrationService.pushUserRecommendInfo(userRecommendDTO);

        return message;
    }

    @Override
    public int saveForumPost(ForumPostVo forumPostVo) {
        forumPostVo.setId(getPostId());
        UserBean user = userIntegrationService.selectUserFromCache(forumPostVo.getUserId());
        if (null != user) {
            if (StringUtils.isBlank(user.getHeadImg()) || StringUtils.isBlank(user.getNickname())) {
                logger.info("帖子保存用户头像或昵称为空,userInfo={}", JSON.toJSONString(user));
            }
            forumPostVo.setUserImg(user.getHeadImg());
            forumPostVo.setUserName(user.getNickname());
        }
        Date currentTime = new Date();
        forumPostVo.setCreateTime(currentTime);
        forumPostVo.setDisplayDateTime(currentTime);
        forumPostVo.setPublishTime(currentTime);
        forumPostVo.setModifyTime(currentTime);
        return this.forumPostMapper.insertSelective(forumPostVo);
    }

    @Override
    public Message deleteForumPost(Long id) {
        logger.debug("开始删除帖子，id：" + id);
        ForumPostVo forumPostVo = this.forumPostMapper.selectByPrimaryKey(id);
        String result = checkPostInfo(forumPostVo);
        if (StringUtils.isNotBlank(result)) {
            return Message.build(Boolean.FALSE, result);
        }
        //状态置为删除
        forumPostVo.setModifyTime(new Date());
        forumPostVo.setStatus(4);
        //更新帖子信息表
        this.forumPostMapper.updateByPrimaryKeySelective(forumPostVo);
        //删除内容推荐库的内容
        this.mixedRecommendPoolMapper.deleteMixRecommandPoolById(id);
        //在帖子被删后仍需被匹配标签
        //this.forumMapper.deletePostTag(id);
        logger.debug("开始回收金币，id：" + id);
        Integer goldNum = userAccountIntegrationService.countGoldByPostId(id, forumPostVo.getUserId());
        if (null != goldNum && goldNum > 0) {
            String content = "删除【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】";
            userAccountIntegrationService.addGold(
                    AccountGoldParam.buildPostDeleteParam(forumPostVo.getUserId(), -goldNum, id, content));
        }
        //更新用户发布数
        userIntegrationService.addPostReplyNum(forumPostVo.getUserId(), 2);
        logger.debug("帖子删除完成，id：" + id);
        return Message.build(Boolean.TRUE);
    }

    @Override
    public void fillExtInfo(List<ForumPostVo> forumPostList, Long userId, String areaCode, boolean isFillTitle, byte isInHome) {
        //填充地区
        LocationDTO location = StringUtils.isNotEmpty(areaCode) ? locationIntegrationService.getLocationByGeocode(areaCode) : null;

        //得到所有帖子id
        List<Long> postIds = forumPostList.stream().map(ForumPostVo::getId).collect(Collectors.toList());

        //得到帖子的话题和板块信息
        List<PostTopicVO> postTopicVoList = forumTopicService.listTopicByPostIds(postIds, userId);
//        List<PostForumVO> postForumVoList = forumService.listForumByPostId(postIds, userId);

        //得到点赞和收藏信息
        List<Long> postLikeIdList = null != userId ? forumPostLikeMapper.listUserPostLike(userId) : new ArrayList<>();
        List<Long> postCollectIdList = null != userId ? newsCollectMapper.selectByIds(userId, (byte) 3) : new ArrayList<>();

        if (!CollectionUtils.isEmpty(forumPostList)) {
            forumPostList.parallelStream().forEach(e -> {
                this.completePostInfo(e, userId, location, postTopicVoList, postLikeIdList, postCollectIdList, isFillTitle, isInHome);
                forumPostStatisticService.judgeEnablePlaceholder(e, location);
                setIsCashAndCashRewardIsZero(e);
            });
        }
    }

    @Override
    public NewsCompleTaskDTO doShareForumPost(Long userId, Long postId, String areaCode, Byte type, Integer platform) {
        logger.debug("用户分享帖子，参数->postId:{},userId:{},type:{},platform:{}", postId, userId, type, platform);
        bizLogIntegrationService.shareForumSuccessed(userId, postId, areaCode, platform);
        return newsReadRewardService.saveShareForums(postId, userId, type, areaCode);
    }

    @Override
    public void doLikeForumPost(Long userId, Long postId, int type, String areaCode, Integer platform) {
        logger.debug("[旧版]用户点赞帖子，参数->postId:{},userId:{},type:{},platform:{}", postId, userId, type, platform);
        ForumPostLike postLike = forumPostLikeMapper.selectByUserIdAndPostId(userId, postId);
        UserBean userBean = userIntegrationService.selectUserFromCache(userId);
        //生成一条帖子点赞记录
        this.generateForumPostLike(postLike, type, postId, userId, userBean);
        //更新帖子点赞数量
        forumPostStatisticService.doUpdateForumInfo(type, postId);
        //帖子点赞流程
        SpringContextHolder.getBean(ForumPostStatisticService.class).doAsyncLikePost(userId, postId, type, areaCode, (byte) 1, platform);
    }

    @Override
    public Message doInformForumPost(ForumPostInformParam param) {
        ForumPostInform forumPostInformParam = new ForumPostInform();
        forumPostInformParam.setPostId(param.getPostId());
        forumPostInformParam.setUserId(param.getUserId());
        List<ForumPostInform> informList = this.forumPostInformMapper.selectByModel(forumPostInformParam);
        if (CollectionUtils.isNotEmpty(informList)) {
            return Message.build(false, "您已举报过");
        }
        ForumPostInform forumPostInform = new ForumPostInform();
        BeanUtils.copyProperties(param, forumPostInform);
        forumPostInform.setId(nextSequence());
        this.forumPostInformMapper.insertSelective(forumPostInform);
        return Message.build(true);
    }

    @Override
    public List<News4Client> listPostDetailRecommend(Long postId, Long userId, Integer size, String areaCode, Integer platform) {
        ForumParam forumParam = generateForumParam(size, postId, userId, areaCode, platform);

        List<Long> idList = newsRecommendIntegrationService.recommendList(forumParam);
        logger.debug("[listPostDetailRecommend]帖子详情推荐：{}", JSONObject.toJSONString(idList));
        List<MixRecommendDTO> mixRecommendList = ForumPostConvert.convertIds2MixRecommendDTO(idList);
        List<News4Client> news4ClientList = SpringContextHolder.getBean(RecommendService.class).listNews4Client(mixRecommendList, userId, areaCode, true);
        logger.debug("[listPostDetailRecommend]帖子详情推荐结果：{}", JSONObject.toJSONString(news4ClientList));
        return news4ClientList;
    }

    @Override
    public PostLikeDto doLikeForumPost(Long postId, Long userId, String areaCode, Integer platform) {
        PostLikeDto postLikeDto = new PostLikeDto();
        UserBean userBean = userIntegrationService.selectUserFromCache(userId);
        if (StringUtils.isBlank(userBean.getHeadImg()) || StringUtils.isBlank(userBean.getNickname())) {
            logger.info("帖子点赞用户头像或昵称为空,userInfo={}", JSON.toJSONString(userBean));
        }
        ForumPostLike postLike = forumPostLikeMapper.selectByUserIdAndPostId(userId, postId);
        //判断帖子是否被用户点赞,或者为是取消点赞状态
        if (Objects.isNull(postLike) || postLike.getType() == 0) {
            //生成帖子点赞记录
            long id = this.generateForumPostLike(postLike, 1, postId, userId, userBean);

            //帖子概览数据的更新
            forumPostStatisticService.doUpdateForumInfo(1, postId);

            //判断是否是自己给自己点赞，如果是则只加点赞数
            ForumPostVo postVo = forumPostMapper.selectByPrimaryKey(postId);
            boolean flag = Objects.nonNull(userId) && Objects.nonNull(postVo.getUserId()) && userId.equals(postVo.getUserId());
            if (flag) {
                postLikeDto.setCode(PostLikeDto.SUCCESS_CODE);
                postLikeDto.setCopyWriter(null);
                postLikeDto.setShowFlower(PostLikeDto.FAILURE_CODE);
            } else {
                postLikeDto = calculateFlower(postVo, userId, id, areaCode, platform);
            }

            //  调用用户增加点赞数接口
            userIntegrationService.updateUserLikeNumByUserId(postVo.getUserId());
        } else {
            postLikeDto = null;
        }
        return postLikeDto;
    }

    @Override
    public List<ForumPostLikeDTO> getPostLikeDetail(Long id, Long userId, String devcId) {
        ArrayList<ForumPostLikeDTO> result = Lists.newArrayList();
        List<ForumPostLike> forumPostLikes = forumPostLikeMapper.selectPostById(id);
        forumPostLikes.forEach(e -> {
            ForumPostLikeDTO postLikeDTO = new ForumPostLikeDTO();
            BeanUtils.copyProperties(e, postLikeDTO);
            if (StringUtils.isBlank(postLikeDTO.getHeadImg())) {
                postLikeDTO.setHeadImg(userProperties.getDefaultHeadImgUrl());
            }
            result.add(postLikeDTO);
        });
        return result;
    }

    @Override
    public Boolean relaceSoftbankEmoji() {
        PageParam pageParam = new PageParam();
        pageParam.setPageSize(1000);

        for (int i = 1; i < 100; i++) {
            pageParam.setPageNum(i);
            PageWarper<ForumPostVo> forumPostVoPage = new PageWarper<>(forumPostMapper.selectAllPosts(pageParam));

            if (CollectionUtils.isEmpty(forumPostVoPage.getList())) {
                break;
            }
            List<ForumPostVo> updateList = new ArrayList<>();
            for (ForumPostVo forumPostVo : forumPostVoPage.getList()) {
                boolean changed = false;
                if (EmojiCodeParser.hasSoftbankEmoji(forumPostVo.getContent())) {
                    changed = true;
                    forumPostVo.setContent(EmojiCodeParser.replaceSoftbankEmoji(forumPostVo.getContent()));
                }
                if (EmojiCodeParser.hasSoftbankEmoji(forumPostVo.getTextField())) {
                    changed = true;
                    forumPostVo.setTextField(EmojiCodeParser.replaceSoftbankEmoji(forumPostVo.getTextField()));
                }
                if (EmojiCodeParser.hasSoftbankEmoji(forumPostVo.getTitle())) {
                    changed = true;
                    forumPostVo.setTitle(EmojiCodeParser.replaceSoftbankEmoji(forumPostVo.getTitle()));
                }
                if (EmojiCodeParser.hasSoftbankEmoji(forumPostVo.getUserName())) {
                    changed = true;
                    forumPostVo.setUserName(EmojiCodeParser.replaceSoftbankEmoji(forumPostVo.getUserName()));
                }
                if (changed) {
                    updateList.add(forumPostVo);
                }
            }

            if (CollectionUtils.isNotEmpty(updateList)) {
                MybatisBatchBuilder.create(ForumPostMapper.class, updateList).run(ForumPostMapper::updateByPrimaryKeySelective);
            }
        }

        return true;
    }

    @Override
    public void doCollectForumPost(ForumPostCollectParam forumPostCollectParam) {
        NewsCollect newsCollect = new NewsCollect(forumPostCollectParam.getUserId(), forumPostCollectParam.getPostId(), (byte) 3);
        this.collectService.collectNews(newsCollect, forumPostCollectParam.getPlatform(), forumPostCollectParam.getAreaCode());
    }

    @Override
    public List<ForumPostVo> getPostListByIds(String postIds) {
        List<Long> postIdList = ListUtil.convertStringToList(postIds);
        if (CollectionUtils.isEmpty(postIdList)) {
            return null;
        }

        List<ForumPostVo> forumPostVoList = forumPostMapper.listPostByIds(postIdList);
        if (CollectionUtils.isNotEmpty(forumPostVoList)) {
            for (ForumPostVo forumPostVo : forumPostVoList) {
                forumPostStatisticService.exchangeDetailPost(forumPostVo);
                completeCommonPostInfo(forumPostVo, null, null, true);
                setIsCashAndCashRewardIsZero(forumPostVo);
            }
        }

        return forumPostVoList;
    }

    @Override
    public List<ForumPostVo> getRewardPostList() {
        String rewardList = redisStringAdapter.getString(RedisConfig.FORUM_POST_REWARD_LIST.copy());

        List<ForumPostVo> forumPostVoList;
        if (StringUtils.isBlank(rewardList)) {
            forumPostVoList = forumPostMapper.getRewardPostList(50);
        } else {
            forumPostVoList = JSON.parseArray(rewardList, ForumPostVo.class);
        }

        if (CollectionUtils.isNotEmpty(forumPostVoList)) {
            forumPostVoList.forEach(e -> forumPostStatisticService.judgeEnablePlaceholder(e, null));
        }

        return forumPostVoList;
    }

    @Override
    public Boolean getContentUpdate(Date lastRequestTime, String areaCode) {
        if (Objects.isNull(lastRequestTime)) {
            return true;
        }
        Date maxPublishTime = forumPostMapper.selectMaxPublishTime(areaCode);
        if (null != maxPublishTime) {
            return maxPublishTime.compareTo(lastRequestTime) > 0;
        }
        return false;
    }

    @Override
    public PageWarper<ForumPostVo> getHomePagePostList(HomePagePostParam param) {
        PageWarper<ForumPostVo> forumPostWarper = new PageWarper<>(forumPostMapper.listHomePagePostList(param));
        List<ForumPostVo> forumPostVoList = forumPostWarper.getList();
        if (CollectionUtils.isNotEmpty(forumPostVoList)) {
            this.fillExtInfo(forumPostVoList, param.getUserId(), param.getAreaCode(), false, (byte) 0);
        }
        return forumPostWarper;
    }

    private void addRecommendTopic(ForumPostListQueryParam param, PageWarper<ForumPostVo> forumPostPage, List<ForumPostVo> postList) {
        //插入推荐话题
        List<TopicVo> topicVoList = forumTopicService.getTopicList(param.getAreaCode());
        if (CollectionUtils.isNotEmpty(topicVoList)) {
            if (topicVoList.size() > 6) {
                topicVoList = topicVoList.stream().limit(6).collect(Collectors.toList());
            }

            Integer order = redisHashMapAdapter.get(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), param.getUserId() + "", Integer.class);
            if (null == order || ++order >= topicVoList.size()) {
                order = 0;
            }
            redisHashMapAdapter.put(RedisConfig.USER_READ_RECOMMEND_TOPIC.copy(), param.getUserId() + "", order);
            TopicVo topic = topicVoList.get(order);
            //补充分享链接，紧急修复ios分享问题
            forumTopicService.fillTopicShareInfo(topic, param.getUserId());

            List<ForumPostVo> newPostList = new ArrayList<>();
            for (int i = 0; i < postList.size(); i++) {
                if (i == 3) {
                    ForumPostVo forumPostVo = new ForumPostVo();
                    forumPostVo.setRecommendTopicVo(topic);
                    newPostList.add(forumPostVo);
                }

                newPostList.add(postList.get(i));
            }

            forumPostPage.setList(newPostList);
        }
    }

    /**
     * 修复ios传递错误参数的问题
     *
     * @param forumParam 请求参数
     * @param basicParam 基础参数
     */
    private void fixErrorParamWithIOS(ForumParam forumParam, BasicParam basicParam) {
        //如果是ios并且是3.0.0版本
        if (PlatformEnum.IOS.getCode() == basicParam.getPlatform()
                && VersionUtils.isHighVersion(basicParam.getCurVer(), "3.0.0") >= 0
                && VersionUtils.isHighVersion(basicParam.getCurVer(), "3.1.0") < 0) {
            //如果存在话题ID，并且话题ID实际为帖子ID
            if (forumParam.getTopicId() != null && JudgeUtil.isPost(forumParam.getTopicId())) {
                //根据帖子ID去查询话题ID，并修改参数
                List<RelationDTO> relations = forumTopicMapper.getPostTopicRelation(ImmutableList.of(forumParam.getTopicId()));
                if (CollectionUtils.isNotEmpty(relations)) {
                    forumParam.setTopicId(relations.get(0).getBid());
                }
            }
        }
    }

    /**
     * 帖子创建后的一些操作
     *
     * @param basicParam  基础参数
     * @param forumPostVo 帖子信息
     * @return 任务完成情况
     */
    private NewsCompleTaskDTO afterCreate(BasicParam basicParam, ForumPostVo forumPostVo) {
        //记录日志是否完成发帖
        if (basicParam != null) {
            bizLogIntegrationService.forumSuccessed(forumPostVo.getUserId(), forumPostVo.getAreaCode(), basicParam.getPlatform());
        }

        //更新用户发布数（异步）
        SpringContextHolder.getBean(ForumPostStatisticService.class).doAsyncAddPublishNum(forumPostVo);

        //增加点赞评论数，如果是新人点赞直接加点赞数（异步）
        if (forumPostVo.getStatus() == 6) {
            ThreadLocalRandom random = ThreadLocalRandom.current();
            int num = random.nextInt(5, 20);
            SpringContextHolder.getBean(ForumPostStatisticService.class).addPostLike(forumPostVo, num);
            //增加马甲号评论
            SpringContextHolder.getBean(ForumPostStatisticService.class).addPostReply(forumPostVo);
        }

        //完成任务
        return forumPostStatisticService.completeTask(forumPostVo);
    }

    /**
     * 校验帖子是否能被删除
     *
     * @param forumPostVo 帖子
     * @return 删除提示
     */
    private String checkPostInfo(ForumPostVo forumPostVo) {
        String result = StringUtils.EMPTY;
        if (forumPostVo == null) {
            result = "帖子不存在";
        } else if (forumPostVo.getIsCash() == 1) {
            result = "现金奖励内容删除需要联系客服哦";
        } else if (forumPostVo.getIsBroke() == 1) {
            result = "爆料内容删除需要联系客服哦";
        } else if (forumPostVo.getIsRecommend() == 1) {
            result = "优质头条删除需要联系客服哦";
        } else if (forumPostVo.getIsBrilliant() == 1) {
            result = "精华内容删除需要联系客服哦";
        }
        return result;
    }

    /**
     * 若黑色章的开关置为【黑色章】，且帖子打上了现金奖励标，则现金奖励标志不显示
     */
    private void setIsCashAndCashRewardIsZero(ForumPostVo forumPostVo) {
        boolean flag = Objects.nonNull(forumPostVo)
                && Objects.nonNull(forumPostVo.getIsRed())
                && forumPostVo.getIsRed() == 0
                && Objects.nonNull(forumPostVo.getIsCash())
                && forumPostVo.getIsCash() == 1;
        if (flag) {
            forumPostVo.setIsCash(0);
            forumPostVo.setCashReward(new BigDecimal(0));
        }
    }

    /**
     * 为详情提供的实体属性填充
     */
    private void completePostInfo(ForumPostVo forumPostVo, Long userId, LocationDTO locationDTO) {
        if (null == forumPostVo) {
            return;
        }
        //填充板块信息
//        if (null != forumPostVo.getForumId()) {
//            forumPostVo.setForum(forumService.getForumPost(forumPostVo.getForumId(), userId));
//        }

        //填充话题信息
        forumPostVo.setTopicList(forumTopicService.listTopicByPostId(forumPostVo.getId(), userId));

        if (null != userId) {
            //填充点赞
            Integer like = this.forumPostLikeMapper.getUserPostLike(forumPostVo.getId(), userId);
            forumPostVo.setLiked(null != like && 1 == like ? 1 : 0);
            //填充收藏
            forumPostVo.setCollected(null != this.newsCollectMapper.selectUserCollect(forumPostVo.getId(), userId, (byte) 3) ? 1 : 0);
        }

        //填充详情封面
        forumPostStatisticService.exchangeDetailPost(forumPostVo);

        //填充马甲号类型
        forumPostVo.setVestType(userIntegrationService.getVirtualUserType(forumPostVo.getUserId()));

        //通用参数填充
        this.completeCommonPostInfo(forumPostVo, locationDTO, userId, true);

        setPluginInfo(forumPostVo, userId);
    }

    /**
     * 设置帖子相关插件信息，暂时只有投票插件
     *
     * @param forumPostVo 帖子详情
     * @param userId      当前查询的用户ID
     */
    private void setPluginInfo(ForumPostVo forumPostVo, Long userId) {
        logger.debug("forumpost:[{}],plugins:[{}]", forumPostVo.getId(), forumPostVo.getPlugins());
        if (forumPostVo.getPlugins() != null) {
            for (ForumPlugin plugin : forumPostVo.getPlugins()) {
                if ("VOTE".equals(plugin.getType())) {
                    VotePinParam param = new VotePinParam();
                    param.setRelationId(forumPostVo.getId());
                    param.setVoteId(plugin.getId());
                    param.setUserId(userId);

                    forumPostVo.setVoteDetailDTO(voteIntegrationService.get(param));
                    break;
                }
            }
        }
    }

    /**
     * 专为列表提供的实体属性填充
     *
     * @param forumPostVo       帖子实体
     * @param userId            用户id
     * @param locationDTO       位置名称
     * @param postTopVoList     帖子列表所在的话题列表
     * @param postLikeIdList    用户对帖子的点赞列表
     * @param postCollectIdList 用户对帖子的收藏列表
     * @param isFillTitle       是否对标题进行填充
     * @param isInHome          0 本地圈 ,1 首页, 2IM分享
     */
    private void completePostInfo(ForumPostVo forumPostVo,
                                  Long userId,
                                  LocationDTO locationDTO,
                                  List<PostTopicVO> postTopVoList,
                                  List<Long> postLikeIdList,
                                  List<Long> postCollectIdList,
                                  boolean isFillTitle,
                                  byte isInHome) {
        if (null == forumPostVo) {
            return;
        }
        //首先把内容图先保存起来
        forumPostVo.setPostContentImgList(forumPostVo.getPostImgList());

        //填充板块
//        if (null != forumPostVo.getForumId()) {
//            Optional<PostForumVO> forumVoOptional = forumVoList.parallelStream().filter(e -> e.getPostId().equals(forumPostVo.getId())).findFirst();
//            forumVoOptional.ifPresent(forumPostVo::setForum);
//        }

        //填充话题
        if (CollectionUtils.isNotEmpty(postTopVoList)) {
            List<TopicVo> topicVoList = postTopVoList.parallelStream().filter(e -> e.getPostId().equals(forumPostVo.getId())).collect(Collectors.toList());
            forumPostVo.setTopicList(topicVoList);
        }

        //填充点赞和收藏
        if (null != userId) {
            forumPostVo.setLiked(postLikeIdList.parallelStream().anyMatch(e -> e.equals(forumPostVo.getId())) ? 1 : 0);
            forumPostVo.setCollected(postCollectIdList.parallelStream().anyMatch(e -> e.equals(forumPostVo.getId())) ? 1 : 0);
        }

        //填充剩余评论数
        forumPostVo.setLeftRelyNum(CollectionUtils.isNotEmpty(forumPostVo.getReplyInfo()) ? forumPostVo.getCommentCount() - forumPostVo.getReplyInfo().size() : forumPostVo.getCommentCount());

        //是否在没有标题的情况下用内容补充标题
        if (isFillTitle) {
            forumPostVo.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
        }

        if (isInHome == 0) {
            //本地圈的封面图设置
            forumPostStatisticService.exchangeDetailPost(forumPostVo);
        } else if (isInHome == 1) {
            //首页的封面图设置
            forumPostStatisticService.exchangeCoverPost(forumPostVo);
        } else if (isInHome == 2) {
            //IM 内分享图设置
            this.getImCoverList(forumPostVo);
        }

        //填充通用参数
        this.completeCommonPostInfo(forumPostVo, locationDTO, userId, isInHome != 2);
    }

    private void getImCoverList(ForumPostVo forumPostVo) {
        List<PostImgVo> coverFromPost = forumPostStatisticService.getCoverFromPost(forumPostVo.getCoverList(), forumPostVo.getPostImgList(), forumPostVo.getCoverSelect());
        forumPostVo.setPostImgList(coverFromPost);
        if (CollectionUtils.isNotEmpty(coverFromPost)) {
            forumPostVo.setShareImg(coverFromPost.get(0));
        } else {
            forumPostVo.setShareImg(PostImgVo.buildImg(newsProperties.getDefaultImShareImg()));
        }
    }

    /**
     * 帖子列表和详情->通用属性的实体填充
     * 地区
     * 文本
     * 内容
     * 展示时间
     * 分享地址
     * 内容标识
     *
     * @param forumPostVo 帖子
     * @param locationDTO 地区
     * @param userId      用户id
     * @param isFillNote  是否填充小纸条
     */
    private void completeCommonPostInfo(ForumPostVo forumPostVo, LocationDTO locationDTO, Long userId, boolean isFillNote) {
        //如果没有地区就填充当前地区
        if (StringUtils.isBlank(forumPostVo.getLocation())) {
            forumPostVo.setLocation(null != locationDTO ? locationDTO.getName() : null);
        }

        //填充文本
        forumPostVo.setTextField(StringUtils.isNotBlank(forumPostVo.getTextField()) ? StringEscapeUtils.unescapeHtml(forumPostVo.getTextField()) : null);

        //填充内容
        forumPostVo.setContent(StringUtils.isNotBlank(forumPostVo.getContent()) ? StringEscapeUtils.unescapeHtml(forumPostVo.getContent()) : null);

        //填充展示时间
        forumPostVo.setDisplayTime(null != forumPostVo.getDisplayDateTime() ? DateUtils.timeConvertString(forumPostVo.getDisplayDateTime()) : null);

        //填充分享地址
        forumPostVo.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.FORUM, UrlTypeEnum.SHARE, forumPostVo.getId(), userId));

        //填充小程序分享地址
        forumPostVo.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.POST, forumPostVo.getId(), userId));

        //如果不是精华，不是爆料，不是现金奖励且是优质头条就设置为红色章
        if (0 == forumPostVo.getIsBrilliant() && 0 == forumPostVo.getIsBroke() && 0 == forumPostVo.getIsCash() && 1 == forumPostVo.getIsRecommend()) {
            forumPostVo.setIsRed(1);
        }

        if (PostTypeEnum.NOTE.getCode() == forumPostVo.getPostType()) {
            //小纸条填充标题，需求如此，无能为力
            if (isFillNote) {
                forumPostVo.setTitle("我发布了一个纸条");
            }
            forumPostVo.setShareImg(PostImgVo.buildImg(forumProperties.getNoteShareImg()));
        }
    }

    /**
     * 生成一条帖子点赞记录
     *
     * @param postLike 点赞信息
     * @param type     类型
     * @param postId   帖子id
     * @param userId   用户id
     * @param userBean 用户信息
     */
    private Long generateForumPostLike(ForumPostLike postLike, int type, Long postId, Long userId, UserBean userBean) {
        String updateHeadImg = StringUtils.isBlank(userBean.getHeadImg()) ? userProperties.getDefaultHeadImgUrl() : userBean.getHeadImg();
        String updateNickname = StringUtils.isBlank(userBean.getNickname()) ? null : userBean.getNickname();
        long id;
        if (Objects.isNull(postLike)) {
            id = nextId();
            ForumPostLike forumPostLike = ForumPostLike.buildPostLike(postId, userId);
            forumPostLike.setId(id);
            forumPostLike.setType((byte) type);
            forumPostLike.setHeadImg(updateHeadImg);
            forumPostLike.setUserNickname(updateNickname);
            //插入点赞表
            forumPostLikeMapper.insertSelective(forumPostLike);
        } else {
            id = postLike.getId();
            postLike.setType((byte) type);
            postLike.setHeadImg(updateHeadImg);
            postLike.setUserNickname(updateNickname);
            forumPostLikeMapper.updateByPrimaryKeySelective(postLike);
        }
        return id;
    }

    /**
     * 组装帖子详情请求参数
     *
     * @param size     数量
     * @param postId   帖子id
     * @param userId   用户id
     * @param areaCode 区域编码
     * @param platform 平台
     * @return 请求参数
     */
    private ForumParam generateForumParam(Integer size, Long postId, Long userId, String areaCode, Integer platform) {
        if (null == size) {
            size = 5;
        }
        ForumParam forumParam = new ForumParam();
        //帖子详情类型
        forumParam.setOperationId(7);
        forumParam.setPostId(postId);
        forumParam.setUserId(userId);
        forumParam.setAreaCode(areaCode);
        forumParam.setPlatform(platform);
        if (size == 5) {
            forumParam.setPostNum(3);
            forumParam.setNewsNum(2);
        } else if (size == 15) {
            forumParam.setPostNum(10);
            forumParam.setNewsNum(5);
        }
        return forumParam;
    }

    /**
     * 点赞之后对鲜花的增减以及返回包装类
     *
     * @param postVo   帖子
     * @param userId   用户id
     * @param id       帖子id
     * @param areaCode 区域编码
     * @param platform 平台
     * @return 包装类
     */
    private PostLikeDto calculateFlower(ForumPostVo postVo, Long userId, Long id, String areaCode, int platform) {
        //根据是否精华帖获得赞赏鲜花的数量
        final int postLikeConsume = forumPostStatisticService.getPostLikeConsume(postVo.getIsBrilliant());
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        //判断帖子是否开启占位符
        forumPostStatisticService.judgeEnablePlaceholder(postVo, location);
        //点赞用户减少红花
        String consumeContent = "点赞【" + ExtendFactory.getPostContent(postVo.getTitle(), postVo.getTextField()) + "】 消耗小红花";
        Boolean consume = userAccountIntegrationService.addGold(AccountGoldParam.buildPostLikeExpendParam(userId, postLikeConsume, id, consumeContent));

        PostLikeDto postLikeDto = new PostLikeDto();
        if (consume) {
            //帖子作者增加红花
            String addContent = "你发布的【" + ExtendFactory.getPostContent(postVo.getTitle(), postVo.getTextField()) + "】被赞";
            userAccountIntegrationService.addGold(AccountGoldParam.buildPostLikeAddParam(postVo.getUserId(), postLikeConsume, id, addContent));
            //红花增减成功，填充返回实体
            postLikeDto.setAmount(postLikeConsume);
            postLikeDto.setCode(PostLikeDto.SUCCESS_CODE);
            //自己给自己点赞没有文案
            forumPostStatisticService.fillPostLikeDto(postLikeDto, userId, postVo.getUserId(), postLikeConsume);
            //异步推送信息
            SpringContextHolder.getBean(ForumPostStatisticService.class).doAsyncLikePost(userId, postVo.getId(), 1, areaCode, (byte) 0, platform);
        } else {
            //扣除红花失败,回滚数据
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            //扣款失败
            postLikeDto.setCode(PostLikeDto.FAILURE_CODE);
            postLikeDto.setShowFlower(PostLikeDto.FAILURE_CODE);
            postLikeDto.setCopyWriter("你的小红花余额不足，快转发文章赚小红花");
        }
        return postLikeDto;
    }

    /**
     * 获取帖子生成id，默认从4000000000000000开始
     *
     * @return id
     */
    private Long getPostId() {
        KeyGenerator postKeyGenerator = RedisConfig.SEQ_POST_ID.copy();
        return redisStringAdapter.incrementWithDefault(postKeyGenerator, 4000000000000000L, 1);
    }

}
