package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.news.config.BizConfigProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumMapper;
import com.bxm.localnews.news.dto.RelationDTO;
import com.bxm.localnews.news.enums.UrlCategoryEnum;
import com.bxm.localnews.news.enums.UrlPositionEnum;
import com.bxm.localnews.news.enums.UrlTypeEnum;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.service.ForumService;
import com.bxm.localnews.news.vo.ForumVo;
import com.bxm.localnews.news.vo.PostForumVO;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableList;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class ForumServiceImpl extends BaseService implements ForumService {

    @Resource
    private ForumMapper forumMapper;

    @Resource
    private IUrlFactory iUrlFactory;

    @Resource
    private BizConfigProperties bizConfigProperties;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private RedisListAdapter redisListAdapter;

    @Override
    public List<PostForumVO> listForumByPostId(List<Long> postIds, Long userId) {
        List<PostForumVO> postForumVoList = new ArrayList<>();

        List<RelationDTO> relationList = forumMapper.getPostForumRelation(postIds);
        if (CollectionUtils.isNotEmpty(relationList)) {
            for (RelationDTO relation : relationList) {
                PostForumVO postForum = new PostForumVO();
                postForum.setPostId(relation.getAid());

                if (null != relation.getBid()) {
                    ForumVo forumVo = getForumPost(relation.getBid(), userId);
                    if (null != forumVo) {
                        BeanUtils.copyProperties(forumVo, postForum);
                    }
                }

                postForumVoList.add(postForum);
            }
        }

        return postForumVoList;
    }

    @Override
    public void fillForumShareInfo(ForumVo forumVo, Long userId) {
        if (forumVo != null) {
            forumVo.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.BLOCK, forumVo.getId(), userId));
            forumVo.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.FORUM, UrlTypeEnum.BLOCK, forumVo.getId(), userId));
        }
    }

    @Override
    public ForumVo getForumPost(Long forumId, Long userId) {
        ForumVo forumVo = getForumById(forumId, null);
        fillForumShareInfo(forumVo, userId);
        return forumVo;
    }

    @Override
    public List<ForumVo> listForum(String areaCode, Integer size, Long userId, BasicParam basicParam) {
        List<ForumVo> forumVoList = getForumList(areaCode);

        if (CollectionUtils.isNotEmpty(forumVoList)) {
            //如果在首页展示(且用户不为空)，则需要加入最近浏览
            if (null != userId) {
                this.sortForm(forumVoList, userId);
                forumVoList = forumVoList.stream().distinct().collect(Collectors.toList());
            }

            //如果传了数量，则做限制
            if (size != null) {
                forumVoList = forumVoList.stream().limit(size).collect(Collectors.toList());
            }

            //小程序过审时隐藏一些敏感板块
            forumVoList = this.filterForum(forumVoList, basicParam);

            //填补分享链接
            forumVoList.forEach(e -> fillForumShareInfo(e, userId));
        }

        return forumVoList;
    }

    /**
     * 保存用户最近浏览的板块，保存3天
     * 用户每次浏览某个板块，都将其置入redis中，并按时间顺序倒排
     * 等用户刷新板块列表时，最近浏览的板块都会显示在最前面
     *
     * @param forumId
     * @param userId
     */
    @Override
    public void updateForum(Long forumId, Long userId) {
        List<Long> ids = new ArrayList<>();
        ids.add(forumId);
        KeyGenerator forumKey = RedisConfig.FORUM_INFO.copy().appendKey(userId).appendKey("forum").appendKey("recent");
        if (redisListAdapter.size(forumKey) > 0) {
            List<Long> forumList = redisListAdapter.leftIndex(forumKey, redisListAdapter.size(forumKey), Long.class);
            ids.addAll(forumList);
        }
        ids = ids.stream().distinct().collect(Collectors.toList());
        logger.debug("根据id进入板块详情获取推荐流更新板块顺序,id:{},userId:{},redisKey:{}", forumId, userId, forumKey.gen());
        redisListAdapter.remove(forumKey);
        redisListAdapter.rightPush(forumKey, ids.toArray(new Long[0]));
        redisListAdapter.expire(forumKey, 3 * 24 * 60 * 60);
    }

    @Override
    public ForumVo getForumDetail(Long id, Long userId, String areaCode) {
        ForumVo forumVo = getForumById(id, areaCode);
        if (null != forumVo) {
            this.updateForum(id, userId);
        }
        fillForumShareInfo(forumVo, userId);
        return forumVo;
    }

    private ForumVo getForumById(Long id, String areaCode) {
        List<ForumVo> forumVoList = getForumList(areaCode);

        if (CollectionUtils.isNotEmpty(forumVoList)) {
            for (ForumVo forumVo : forumVoList) {
                if (forumVo.getId().equals(id)) {
                    return forumVo;
                }
            }
        }

        return forumMapper.selectForumById(id, areaCode);
    }

    /**
     * 先从缓存中取列表，若为空则查数据库
     *
     * @param areaCode
     * @return
     */
    private List<ForumVo> getForumList(String areaCode) {
        String areaCodeKey = areaCode;
        //若不传areaCode则取全国
        if (StringUtils.isBlank(areaCode)) {
            areaCodeKey = "0";
        }

        List<ForumVo> forumVoList = redisHashMapAdapter.get(RedisConfig.FORUM.copy(), areaCodeKey, new TypeReference<List<ForumVo>>() {
        });

        if (CollectionUtils.isEmpty(forumVoList)) {
            forumVoList = forumMapper.getForumList(areaCode, 1);
            if (CollectionUtils.isNotEmpty(forumVoList)) {
                redisHashMapAdapter.put(RedisConfig.FORUM.copy(), areaCodeKey, forumVoList);
            }
        }
        return forumVoList;
    }

    /**
     * 个性化板块显示
     * 将用户在redis中保存的最近浏览板块信息进行处理
     * 将用户最新的点击的板块排在前面
     *
     * @param forumVoList
     * @param userId
     */
    private void sortForm(List<ForumVo> forumVoList, Long userId) {
        KeyGenerator forumKey = RedisConfig.FORUM_INFO.copy().appendKey(userId).appendKey("forum").appendKey("recent");
        if (redisListAdapter.size(forumKey) > 0) {
            List<Long> forumList = redisListAdapter.leftIndex(forumKey, redisListAdapter.size(forumKey), Long.class);
            if (CollectionUtils.isNotEmpty(forumList)) {
                List<ForumVo> existence = new ArrayList<>();

                for (Long id : forumList) {
                    for (ForumVo forumVo : forumVoList) {
                        if (forumVo.getId().equals(id)) {
                            existence.add(forumVo);
                        }
                    }
                }
                //将用户最近访问的放在第三个位置
                if (forumVoList.size() >= 2) {
                    forumVoList.addAll(2, existence);
                } else {
                    forumVoList.addAll(existence);
                }
            }
        }
    }

    /**
     * 过滤板块(小程序过审时用)
     * 有些板块内容比较敏感，不做展示
     * 感觉没卵用，所以写死了
     *
     * @param forumVoList
     * @param basicParam
     */
    private List<ForumVo> filterForum(List<ForumVo> forumVoList, BasicParam basicParam) {
        if (5 == basicParam.getPlatform() && bizConfigProperties.getAppletReviewEnable()) {
            List<Long> postIds = ImmutableList.of(
                    72317665317008L,
                    79585788760192L,
                    80775388106880L,
                    81892185086080L,
                    77973991071872L,
                    80669901684864L,
                    79211746459776L,
                    79211241126016L
            );

            return forumVoList.stream().filter(e -> !postIds.contains(e.getId())).collect(Collectors.toList());
        } else {
            return forumVoList;
        }
    }

}
