package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.AppVersionIntegrationService;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.enums.SensitiveFlagEnum;
import com.bxm.localnews.news.domain.NewsKindMapper;
import com.bxm.localnews.news.domain.UserKindMapper;
import com.bxm.localnews.news.dto.KindDTO;
import com.bxm.localnews.news.service.NewsKindService;
import com.bxm.localnews.news.vo.NewsKind;
import com.bxm.localnews.news.vo.UserKind;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.ctrip.framework.apollo.spring.annotation.ApolloJsonValue;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.io.*;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.bxm.localnews.news.constant.RedisConfig.NEWS_KIND;

/**
 * Created by mars on 2018/3/12.
 */
@Service
@RefreshScope
public class NewsKindServiceImpl extends BaseService implements NewsKindService {

    private final static Logger     logger = LoggerFactory.getLogger(NewsKindServiceImpl.class);

    private NewsKindMapper          newsKindMapper;

    private UserKindMapper          userKindMapper;

    private RedisStringAdapter      redisStringAdapter;

    private RedisHashMapAdapter     redisHashMapAdapter;

    private RedisSetAdapter         redisSetAdapter;

    private AppVersionIntegrationService appVersionIntegrationService;

    private LocationIntegrationService locationIntegrationService;

    /**
     * 推广地区需要默认显示的频道列表
     */
    @ApolloJsonValue("${news.config.promotion:[]}")
    private List<NewsKind> promotion;

    @Autowired
    public NewsKindServiceImpl(NewsKindMapper newsKindMapper,
                               UserKindMapper userKindMapper,
                               RedisStringAdapter redisStringAdapter,
                               RedisHashMapAdapter redisHashMapAdapter,
                               RedisSetAdapter redisSetAdapter,
                               AppVersionIntegrationService appVersionIntegrationService,
                               LocationIntegrationService locationIntegrationService
                              ){
        this.newsKindMapper = newsKindMapper;
        this.userKindMapper = userKindMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.redisSetAdapter = redisSetAdapter;
        this.appVersionIntegrationService = appVersionIntegrationService;
        this.locationIntegrationService = locationIntegrationService;
    }

    /**
     * 是否处于提包状态
     *
     * @param basicParam
     * @return
     */
    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionIntegrationService.getPublishState(basicParam);
    }

    @Override
    public List<NewsKind> selectAll(BasicParam basicParam, String areaName,String areaCode) {
        List<NewsKind> allKinds = this.selectAllToRedis();
        // 将本地标签换成对应用户定位地区
        this.replaceLocalAreaName(areaCode,areaName, allKinds);
        if (getPublishState(basicParam)) {
            allKinds.removeIf(newsKind -> SensitiveFlagEnum.IS_SENSITIVE.getState().equals(newsKind.getSensitiveFlag()));
        }
        return allKinds;
    }

    @Override
    public Message createUserDefaultKinds(Long userId) {
        return Message.build(this.newsKindMapper.initUserDefaultKinds(userId));
    }

    @Override
    public Message createUserDefaultKindsForPublish(Long userId, BasicParam basicParam) {
        List<Integer> ids;

        List<NewsKind> allKinds = this.selectAllToRedis();
        ids = allKinds.stream().filter(newsKind -> newsKind.getIsDefault() == 1).map(NewsKind::getId).collect(Collectors.toList());

        List<Integer> userKindList = getFixKindId();
        if (CollectionUtils.isNotEmpty(ids)) {
            userKindList.addAll(ids);
        }
        userKindList = userKindList.stream().distinct().collect(Collectors.toList());

        List<UserKind> userKinds = Lists.newArrayList();
        for (int i = 0; i < userKindList.size(); i++) {
            UserKind userKind = new UserKind();
            userKind.setUserKindId(nextSequence());
            userKind.setUserId(userId);
            userKind.setKindId(userKindList.get(i));
            userKind.setSortNo(i);
            userKinds.add(userKind);
        }

        userKindMapper.batchInsert(userKinds);
        return Message.build(true);
    }


    @Override
    public KindDTO getMyKindAndAllKind(Long userId, String areaName, BasicParam basicParam,String areaCode) {
        //得到全部的频道
        List<NewsKind> allKinds = selectAllToRedis();

        KindDTO kindDTO = new KindDTO(allKinds, null);
        if (null != userId) {
            List<NewsKind> myKinds;
            //如果地区已经推广，并且用户第一次进推广地区
            if (checkForumPostEnable(areaCode) && !redisSetAdapter.exists(RedisConfig.FIRST_ENTER_PROMOTION_AREA, userId)) {
                myKinds = generatePromotionKinds(userId);
                logger.debug("[getMyKindAndAllKind]用户第一次进推广频道，频道列表:{}", JSONObject.toJSONString(myKinds));
                redisSetAdapter.add(RedisConfig.FIRST_ENTER_PROMOTION_AREA, userId);
            } else {
                //得到用户的频道
                myKinds = getMyKind(userId);
            }

            if (CollectionUtils.isEmpty(myKinds)) {
                myKinds = generateCommonMyKinds(userId);
            }
            kindDTO = new KindDTO(allKinds, myKinds);
        }

        this.replaceLocalAreaName(areaCode,areaName, kindDTO.getAllKinds());
        if (getPublishState(basicParam)) {
            kindDTO.getAllKinds().removeIf(newsKind -> SensitiveFlagEnum.IS_SENSITIVE.getState().equals(newsKind.getSensitiveFlag()));
        }

        if (CollectionUtils.isNotEmpty(kindDTO.getMyKinds())) {
            this.replaceLocalAreaName(areaCode,areaName, kindDTO.getMyKinds());
            if (getPublishState(basicParam)) {
                kindDTO.getMyKinds().removeIf(newsKind -> SensitiveFlagEnum.IS_SENSITIVE.getState().equals(newsKind.getSensitiveFlag()));
            }
        }

        return kindDTO;
    }

    /**
     * 包装我的频道( 推广地区) 配置中的+数据库的
     * @param userId
     */
    private List<NewsKind> generatePromotionKinds(Long userId){
        List<NewsKind> result = new ArrayList<>();

        List<NewsKind> fixKindList = new ArrayList<>();
        //获取固定频道（7个）
        try {
            fixKindList =deepCopy(promotion);
        } catch (IOException | ClassNotFoundException e) {
            logger.error("推广地区默认频道复制失败，原因：{}",e.getMessage());
            e.printStackTrace();
        }

        result.addAll(fixKindList);
        //去重
        result = result.stream().distinct().collect(Collectors.toList());

        NewsKindService newsKindService = SpringContextHolder.getBean(NewsKindService.class);
        newsKindService.saveKind(userId,result);

        return result;

    }

    /**
     * 包装我的频道(非推广地区)
     * @param userId
     */
    private List<NewsKind> generateCommonMyKinds(Long userId){
        //获取默认频道(全部)
        List<NewsKind> fixKindList = newsKindMapper.selectAll();
        //去重
        fixKindList = fixKindList.stream().distinct().collect(Collectors.toList());

        NewsKindService newsKindService = SpringContextHolder.getBean(NewsKindService.class);
        newsKindService.saveKind(userId,fixKindList);

        return fixKindList;
    }


    @Async
    @Override
    public void saveKind(Long userId, List<NewsKind> fixKindList){
        //排序
        AtomicInteger i= new AtomicInteger(0);
        fixKindList.forEach(x->{
            x.setSortNo(i.get());
            i.getAndIncrement();
        });

        //保存用户频道
        Integer[] myKindArray = fixKindList.stream().map(NewsKind::getId).toArray(Integer[]::new);
        this.resortUserKind(userId,myKindArray,"0");
    }

    /**
     * @param areaName
     * @param myKinds
     * @Description 方法描述：将本地标签换成对应用户定位地区
     * @author leon 2019年1月23日 下午7:17:01
     * @CopyRight 杭州微财网络科技有限公司
     */
    private void replaceLocalAreaName(String areaCode,String areaName, List<NewsKind> myKinds) {
        if (StringUtils.isNotBlank(areaName) && myKinds != null && myKinds.size() > 0) {
            boolean isPostEnable = checkForumPostEnable(areaCode);
            Iterator iterator = myKinds.iterator();
            while (iterator.hasNext()) {
                NewsKind newsKind = (NewsKind) iterator.next();
                logger.debug("newsKind.getName:{}", newsKind.getName());
                //如果频道中有本地频道，则判断是否时推广地区，如果是非推广地区就删除，如果是推广地区则显示XX社区
                if (StringUtils.equals(newsKind.getName(), "本地")) {
                    if (isPostEnable) {
                        newsKind.setName(areaName + "社区");
                    } else {
                        myKinds.remove(newsKind);
                    }
                    break;
                }
            }

        }
    }

    /**
     * 根据地区编号去查询该编号所在的地区是否开启社区
     * 旧版本不传areaCode字段，在此判断
     * @param areaCode
     * @return
     */
    private boolean checkForumPostEnable(String areaCode) {
        if (StringUtils.isNotEmpty(areaCode)) {
            LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);
            if (locationDTO != null) {
                int enableCommunityContent = locationDTO.getEnableCommunityContent()==null ?0:locationDTO.getEnableCommunityContent();
                return 1 == enableCommunityContent;
            }
        }
        return false;
    }

    @Override
    public Message addUserKind(Long userId, Integer kindId) {
        List<NewsKind> myKinds = getMyKind(userId);
        if (myKinds.stream().anyMatch(item -> item.getId().equals(kindId))) {
            return Message.build(false, "重复添加");
        }

        List<NewsKind> allKinds = selectAllToRedis();
        Optional<NewsKind> newsKindOptional = allKinds.stream().filter(item -> item.getId().equals(kindId)).findFirst();
        if (newsKindOptional.isPresent()) {
            NewsKind newsKind = newsKindOptional.get();
            myKinds.add(newsKind);
        }

        syncMyKind(myKinds, userId);
        return Message.build();
    }

    @Override
    public void resortUserKind(Long userId, Integer[] kindIds, String firstOpenType) {
        if (ArrayUtils.isEmpty(kindIds)) {
            return;
        }

        List<Integer> userKindIds = Stream.of(kindIds).collect(Collectors.toList());
        List<Integer> userKindList = getFixKindId();
        userKindList.addAll(userKindIds);
        userKindList = userKindList.stream().distinct().collect(Collectors.toList());

        List<NewsKind> allKinds = selectAllToRedis();
        List<NewsKind> myKinds = Lists.newArrayList();
        userKindList.forEach(id -> {
            allKinds.forEach(newsKind -> {
                if (newsKind.getId().equals(id)) {
                    myKinds.add(newsKind);
                }
            });
        });

        syncMyKind(myKinds, userId);
    }

    /**
     * 获取用户频道信息
     *
     * @param userId
     * @return
     */
    private List<NewsKind> getMyKind(Long userId) {
        TypeReference<List<NewsKind>> typeReference = new TypeReference<List<NewsKind>>() {
        };
        List<NewsKind> myKinds = this.redisHashMapAdapter.get(RedisConfig.NEWS_USER_KIND, userId.toString(),
                typeReference);

        if (CollectionUtils.isEmpty(myKinds)) {
            myKinds = this.userKindMapper.selectMyKinds(userId);
            syncMyKind(myKinds, userId);
        }

        return myKinds;
    }

    /**
     * 更新redis的用户频道数据
     *
     * @param myKinds
     * @param userId
     */
    private void syncMyKind(List<NewsKind> myKinds, Long userId) {
        if (CollectionUtils.isNotEmpty(myKinds)) {
            this.redisHashMapAdapter.put(RedisConfig.NEWS_USER_KIND, userId.toString(), myKinds);
            // 用户id存到set里---调度任务把数据同步到数据库
            this.redisSetAdapter.add(RedisConfig.NEWS_USER_KIND_SET, userId);
        }
    }

    /**
     * 获取所有的频道信息
     *
     * @return
     */
    private List<NewsKind> selectAllToRedis() {
        String str = this.redisStringAdapter.get(NEWS_KIND, String.class);
        List<NewsKind> newsKindList;
        if (StringUtils.isNotEmpty(str)) {
            newsKindList = JSONArray.parseArray(str, NewsKind.class);
        } else {
            newsKindList = this.newsKindMapper.selectAll();
            try {
                if (null != newsKindList) {
                    this.redisStringAdapter.set(NEWS_KIND, JSONArray.toJSONString(newsKindList));
                }
            } catch (Exception e) {
                logger.error("newsKind set redis error");
            }
        }
        return newsKindList;
    }

    /**
     * 获取固定频道的id
     *
     * @return
     */
    private List<Integer> getFixKindId() {
        List<NewsKind> allKinds = this.selectAllToRedis();
        return allKinds.stream().filter(newsKind -> newsKind.getIsFix() == 1).map(NewsKind::getId).collect(Collectors.toList());
    }

    /**
     * 使用序列化的方式克隆一个List对象（之后会移到工具方法中）
     * 主要解决由于mabatis一级缓存导致的对象循环引用问题
     *
     * @param src 原数据源
     * @param <T>
     * @return
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public <T> List<T> deepCopy(List<T> src) throws IOException, ClassNotFoundException {
        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(byteOut);
        out.writeObject(src);

        ByteArrayInputStream byteIn = new ByteArrayInputStream(byteOut.toByteArray());
        ObjectInputStream in = new ObjectInputStream(byteIn);
        @SuppressWarnings("unchecked")
        List<T> dest = (List<T>) in.readObject();
        return dest;
    }
}
