package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.MissionIntegrationService;
import com.bxm.localnews.integration.NewsStatisticIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.NewsSharEnum;
import com.bxm.localnews.news.param.ExecGoldCalParam;
import com.bxm.localnews.news.param.ExecGoldParam;
import com.bxm.localnews.news.service.NewsReadRewardService;
import com.bxm.localnews.news.strategy.GoldenStrategyContext;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.news.constant.AppConst.UN_GOLD;
import static com.bxm.localnews.news.constant.RedisConfig.USER_SHARE_NEWS_RECORD;

@Service
public class NewsReadRewardServiceImpl extends BaseService implements NewsReadRewardService {

    private UserIntegrationService userIntegrationService;

    private RedisStringAdapter redisStringAdapter;

    private NewsStatisticsMapper newsStatisticsMapper;

    private NewsCollectMapper newsCollectMapper;

    private NewsMapper newsMapper;

    private NewsShareMapper newsShareMapper;

    private GoldenStrategyContext goldenStrategyContext;

    private MissionIntegrationService missionIntegrationService;

    private NewsStatisticIntegrationService newsStatisticIntegrationService;

    private ForumPostMapper forumPostMapper;

    private ForumPostShareMapper forumPostShareMapper;

    private DistributedLock distributedLock;

    @Autowired
    public NewsReadRewardServiceImpl(UserIntegrationService userIntegrationService,
                                     RedisStringAdapter redisStringAdapter,
                                     NewsStatisticsMapper newsStatisticsMapper,
                                     NewsCollectMapper newsCollectMapper,
                                     NewsMapper newsMapper, NewsShareMapper newsShareMapper,
                                     GoldenStrategyContext goldenStrategyContext,
                                     MissionIntegrationService missionIntegrationService,
                                     NewsStatisticIntegrationService newsStatisticIntegrationService,
                                     ForumPostMapper forumPostMapper,
                                     ForumPostShareMapper forumPostShareMapper,
                                     DistributedLock distributedLock) {
        this.userIntegrationService = userIntegrationService;
        this.redisStringAdapter = redisStringAdapter;
        this.newsStatisticsMapper = newsStatisticsMapper;
        this.newsCollectMapper = newsCollectMapper;
        this.newsMapper = newsMapper;
        this.newsShareMapper = newsShareMapper;
        this.goldenStrategyContext = goldenStrategyContext;
        this.missionIntegrationService = missionIntegrationService;
        this.newsStatisticIntegrationService = newsStatisticIntegrationService;
        this.forumPostMapper = forumPostMapper;
        this.forumPostShareMapper = forumPostShareMapper;
        this.distributedLock = distributedLock;
    }

    @Override
    public Json<NewsCompleTaskDTO> saveShareNews(Long newsId, Long userId,Byte type) {
        if (newsId == null || userId == null) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }
        UserBean user = this.userIntegrationService.selectUserFromCache(userId);

        //用户未正式注册前不能分享新闻获得金币
        if (user == null || user.getState() != 1) {
            return ResultUtil.genSuccessResult(new NewsCompleTaskDTO());
        }

        NewsCompleTaskDTO shareGoldDTO = null;
        KeyGenerator shareRecordKey = USER_SHARE_NEWS_RECORD.copy()
                .appendKey(userId.toString())
                .appendKey(DateUtils.formatDate(new Date()))
                .appendKey(newsId.toString());

        String requestId = String.valueOf(nextSequence());
        String key = "share_"+ userId+"_"+newsId;

        if (distributedLock.lock(key, requestId)) {
            if (!redisStringAdapter.hasKey(shareRecordKey)) {
                //发放分享文章奖励
                shareGoldDTO = missionIntegrationService.compleTask(userId, "TASK_NEWS_SHARE", newsId.toString());
                //记录是否分享过该新闻
                redisStringAdapter.set(shareRecordKey, "exist");
                redisStringAdapter.expire(shareRecordKey, DateUtils.getCurSeconds());

                //分享成功获取金币记录
                NewsShare share = new NewsShare();
                share.setAddTime(new Date());
                share.setNewsId(newsId);
                share.setUserId(userId);
                share.setId(nextSequence());
                share.setReward(Objects.isNull(shareGoldDTO.getGoldNum()) ? 0 : shareGoldDTO.getGoldNum().intValue());
                share.setType(type);
                this.newsShareMapper.insertSelective(share);
            }
            this.newsStatisticIntegrationService.pushShareData(userId, newsId);
            //解锁
            this.distributedLock.unlock(key, requestId);
        }
        return ResultUtil.genSuccessResult(shareGoldDTO);
    }


    @Override
    public Json<NewsCompleTaskDTO> saveShareForums(Long postId, Long userId,Byte type) {
        if (postId == null || userId == null) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误");
        }
        UserBean user = this.userIntegrationService.selectUserFromCache(userId);

        //用户未正式注册前不能分享新闻获得金币
        if (user == null || user.getState() != 1) {
            return ResultUtil.genSuccessResult(new NewsCompleTaskDTO());
        }

        NewsCompleTaskDTO shareGoldDTO = null;
        KeyGenerator shareRecordKey = USER_SHARE_NEWS_RECORD.copy()
                .appendKey(userId.toString())
                .appendKey(DateUtils.formatDate(new Date()))
                .appendKey(postId.toString());
        if (!redisStringAdapter.hasKey(shareRecordKey)) {
            //发放分享文章奖励
            shareGoldDTO = missionIntegrationService.compleTask(userId, "TASK_NEWS_SHARE", postId.toString());
            //记录是否分享过该新闻
            redisStringAdapter.set(shareRecordKey, "exist");
            redisStringAdapter.expire(shareRecordKey, DateUtils.getCurSeconds());

            //分享成功获取金币记录
            ForumPostShare forumPostShare = new ForumPostShare();
            forumPostShare.setAddTime(new Date());
            forumPostShare.setId(nextSequence());
            forumPostShare.setPostId(postId);
            forumPostShare.setReward(Objects.isNull(shareGoldDTO.getGoldNum())?0:shareGoldDTO.getGoldNum().intValue());
            forumPostShare.setUserId(userId);
            forumPostShare.setType(type);
            forumPostShareMapper.insertSelective(forumPostShare);
        }

        //更新分享数量
        ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
        forumPostStatistic.setShareCount(1);
        forumPostStatistic.setId(postId);
        this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);
        return ResultUtil.genSuccessResult(shareGoldDTO);
    }


    @Override
    public Json<NewsGoldMeta> execGetGold(ExecGoldParam execGoldParam) {
        //首先判断有无此新闻
        if (null != execGoldParam.getUserId() && null != execGoldParam.getNewsId()) {
            News news = this.newsMapper.selectByPrimaryKey(execGoldParam.getNewsId());
            if (isNewsExsit(news, execGoldParam.getViewTime())) {

                //填充一些额外参数
                NewsGoldMeta newsGoldMeta = new NewsGoldMeta();
                ExecGoldCalParam execGoldCalParam = new ExecGoldCalParam();
                BeanUtils.copyProperties(execGoldParam, execGoldCalParam);
                execGoldCalParam.setNewsGoldMeta(newsGoldMeta);
                execGoldCalParam.setNews(news);

                //默认策略
                goldenStrategyContext.chooseCalulator(execGoldCalParam);

                return ResultUtil.genSuccessResult(execGoldCalParam.getNewsGoldMeta());
            }
        }
        //否则返回默认的，防止出现参数错误
        NewsGoldMeta newsGoldMeta = new NewsGoldMeta();
        newsGoldMeta.setGoldNum(0);
        newsGoldMeta.setTaskName("阅读新闻");
        newsGoldMeta.setGoldType(UN_GOLD);
        newsGoldMeta.setTotalGold(BigDecimal.ZERO);
        return ResultUtil.genSuccessResult(newsGoldMeta);
    }

    private boolean isNewsExsit(News news, Integer viewTime) {
        return news != null && viewTime != null;
    }
}
