package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.news.constant.AppConst;
import com.bxm.localnews.news.constant.UrlCategoryEnum;
import com.bxm.localnews.news.constant.UrlPositionEnum;
import com.bxm.localnews.news.constant.UrlTypeEnum;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.enums.TaskEnum;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.param.NewsParam;
import com.bxm.localnews.news.service.AssemblyService;
import com.bxm.localnews.news.service.NewsService;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.HtmlContentUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service("newsService")
public class NewsServiceImpl extends BaseService implements NewsService {

    @Resource
    private NewsMapper newsMapper;

    @Resource
    private NewsRecordMapper newsRecordMapper;

    @Resource
    private NewsStatisticsMapper newsStatisticsMapper;

    @Resource
    private NewsSearchWordMapper newsSearchWordMapper;

    @Resource
    private NewsKindMapper newsKindMapper;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private NewsRecommendIntegrationService newsRecommendIntegrationService;

    @Resource
    private NewsSearchIntegrationService newsSearchIntegrationService;

    @Resource
    private NewsStatisticIntegrationService newsStatisticIntegrationService;

    @Resource
    private NewsCollectMapper newsCollectMapper;

    @Resource
    private AssemblyService assemblyService;

    @Resource
    private MissionIntegrationService missionIntegrationService;

    @Resource
    private MessageIntegrationService messageIntegrationService;

    @Resource
    private IUrlFactory iUrlFactory;

    @Override
    public Json<NewsUrlVo> getUrlAddress(Long newsId, Long userId, Byte type) {
        //判断是否是新闻
        if (isNews(type)) {
            News news = this.newsMapper.selectByPrimaryKey(newsId);
            if (null != news) {
                String url = getNewsUrlBegin(type, news.getType()) + "newsId=" + newsId + "&userId=" + userId + "&type=" + type;
                NewsUrlVo newsUrlVo = new NewsUrlVo(url);
                return ResultUtil.genSuccessResult(newsUrlVo);
            }
        }

        logger.error("获得详情地址时新闻不存在-newsId:{},userId:{},type:{}", newsId, userId, type);
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "获得详情地址时新闻不存在");
    }

    private boolean isEssayShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.ESSAY;
    }

    private boolean isIconEssayShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.ICON_ESSAY;
    }

    private boolean isVideoShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.VIDEO;
    }

    private boolean isEssayDetail(Byte type) {
        return type == 1 || type == 3;
    }

    private boolean isNews(Byte type) {
        return type != 4;
    }

    /**
     * 组装新闻url
     */
    private String getNewsUrlBegin(byte type, byte newsType) {
        String url = "";
        if (isEssayShare(type, newsType)) {
            url = "shareNewsDetail.html?";
        }
        if (isIconEssayShare(type, newsType)) {
            url = "sharePicDetail.html?";
        }
        if (isVideoShare(type, newsType)) {
            url = "shareVideoDetail.html?";
        }
        if (isEssayDetail(type)) {
            url = "newsDetail.html?";
        }
        return url;
    }

    @Override
    public Json<NewsDetailVO> getNewsDetail(NewsParam newsParam) {
        long begin = System.currentTimeMillis();
        if (null == newsParam.getNewsId()) {
            return ResultUtil.genSuccessResult(new NewsDetailVO());
        }

        List<NewsVO> newsVOList = newsSearchIntegrationService.multipleGet(new Long[]{newsParam.getNewsId()});

        if (newsVOList.isEmpty()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "获取详情时新闻缺失");
        }

        NewsVO newsVO = newsVOList.get(0);
        if (newsVO.getStatus() != 1) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "新闻已下架");
        }
        //填充额外信息
        this.fillDetailExtInfo(newsVO,newsParam);

//        newsVO.setContent(assemblyService.deleteNbspFromVideo(newsVO.getContent()));
        NewsDetailVO newsDetailVO = new NewsDetailVO(newsVO);

        //TODO 用户每一次阅读均进行记录,这里会影响金币的发放，通过消息队列记录阅读记录[liujia]

        logger.debug("[getNewsDetail]新闻详情es拉取完毕, param:[{}] ,content:[{}],耗时:[{}]",
                newsParam, StringUtils.substring(JSONObject.toJSONString(newsDetailVO),0,50), System.currentTimeMillis() - begin);

        //将用户行为记录上报消息中心
        if (null != newsParam.getUserId()) {
            newsStatisticIntegrationService.pushNewClickData(newsParam.getUserId(), newsParam.getNewsId());
            newsStatisticIntegrationService.pushViewData(newsParam.getUserId(), newsParam.getNewsId());
        }



        return ResultUtil.genSuccessResult(newsDetailVO);
    }

    /**
     * 填充列表数据
     *
     * @param newsVO
     * @param newsParam
     */
    private void fillDetailExtInfo(NewsVO newsVO, NewsParam newsParam) {
        newsVO.setCollect(0);
        if (newsParam.getUserId() != null) {
            newsVO.setCollect(countCollect(newsParam.getUserId(), newsParam.getNewsId()));
        }
        //添加水印和图片压缩参数
        newsVO.setContent(assemblyService.fixContent(newsVO.getContent()));

        newsVO.setLocal((byte) 1);
        if (StringUtils.isNotEmpty(newsVO.getAreaDetail())) {
            newsVO.setLocal((byte) 2);
        }

        //添加分享链接
        newsVO.setLinkUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.DETAIL, newsParam.getNewsId(), newsParam.getUserId()));
        newsVO.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.SHARE, newsParam.getNewsId(), newsParam.getUserId()));
        newsVO.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.NEWS, newsParam.getNewsId(), newsParam.getUserId()));

        //添加视频信息
        if (StringUtils.isNotEmpty(newsVO.getVideoStr())) {
            newsVO.setVideoList(JSONArray.parseArray(newsVO.getVideoStr(), PostImgVo.class));
        }

        //如过审则将敏感信息设置为空
        if (appVersionIntegrationService.getPublishState(newsParam)) {
            newsVO.setAuthor(StringUtils.EMPTY);
            newsVO.setAuthorImg(StringUtils.EMPTY);
        }

        //强制引导下载为空时给默认值 0
        if (Objects.isNull(newsVO.getIsBootDownload())) {
            newsVO.setIsBootDownload((byte) 0);
        }
    }

    private int countCollect(Long userId, Long newsId) {
        NewsCollect newsCollect = newsCollectMapper.selectUserCollect(newsId, userId, (byte) 1);
        return null != newsCollect ? 1 : 0;
    }

    @Override
    @Async
    public void completeTaskAndPush(NewsParam newsParam, NewsVO newsVO) {
        if (5 == newsParam.getPlatform()
                && Objects.nonNull(newsParam.getShareUserId())
                && !newsParam.getShareUserId().equals(newsParam.getUserId())) {
            //完成任务
            NewsCompleTaskDTO completeTask = missionIntegrationService.compleTask(newsParam.getShareUserId(), TaskEnum.TASK_FISRT_BROWSE.name(), newsParam.getUserId().toString());
            if (Objects.nonNull(completeTask) && completeTask.getGoldNum() > 0) {

                //提取必要信息
                String extendContent = null;
                String extendUrl = getExtendUrl(newsVO.getImgUrl());
                if (StringUtils.isEmpty(extendUrl)) {
                    extendContent = getExtendContent(newsVO.getTitle(), newsVO.getContent());
                }

                //推送
                messageIntegrationService.addMessage(
                        newsParam.getShareUserId(),
                        newsParam.getUserId(),
                        completeTask,
                        newsParam.getNewsId(),
                        ReplyTypeEnum.NEWS_REPLY.getCode(),
                        extendUrl,
                        extendContent,
                        newsVO.getTitle());
            }
        }
    }

    /**
     * 提取新闻中的第一张图片
     * @param imgStr
     * @return
     */
    private String getExtendUrl(String imgStr){
        //视频封面暂不需要
        List<String> imageList = JSONObject.parseArray(imgStr, String.class);
        if(CollectionUtils.isNotEmpty(imageList)){
            return  imageList.get(0);
        }
        return null;
    }

    /**
     * 提取新闻中的前10个内容
     * @param title
     * @param content
     * @return
     */
    private String getExtendContent(String title, String content) {
        if (StringUtils.isNotBlank(title)) {
            return title;
        } else {
            return HtmlContentUtils.getHtmlDesc(content, 10);
        }
    }

    @Override
    public Json doRecordNews(Long newsId, Long userId, Integer viewTime, String tabType, String advertShow) {
        NewsRecord nRecord = this.newsRecordMapper.getByIds(newsId, userId, null);
        Date d = new Date();
        if (nRecord != null) {
            nRecord.setUpdateTime(d);
            //阅读时间大于上次阅读时间则更新
            if (viewTime > nRecord.getCheckTime()) {
                nRecord.setCheckTime(viewTime);
            }
            this.newsRecordMapper.updateByPrimaryKeySelective(nRecord);
        } else {
            NewsRecord nr = new NewsRecord();
            nr.setId(nextSequence());
            nr.setNewsId(newsId);
            nr.setUserId(userId);
            nr.setUpdateTime(d);
            nr.setCheckTime(viewTime);
            nr.setGetGold(AppConst.UN_GOLD);
            nr.setAddTime(d);
            nr.setLastLocation(tabType);
            nr.setAdViewType(Byte.valueOf(advertShow));
            this.newsRecordMapper.insertSelective(nr);
        }
        return ResultUtil.genSuccessResult();
    }

    @Override
    public Json<List<News4Client>> listRecommendNews(Long newsId, Long userId, Integer size) {
        if (newsId == null || userId == null) {
            logger.error("新闻详情推荐列表接口----参数有误,newsId:{},userId:{}", newsId, userId);
        }

        if (size == null) {
            size = 5;
        }
        List<Long> ids = newsRecommendIntegrationService.recommendByNewsDetail(newsId, userId, size);

        List<Long> nonNullIds = ids.stream().filter(Objects::nonNull).collect(Collectors.toList());
        if (!nonNullIds.isEmpty()) {
            nonNullIds = nonNullIds.stream().limit(size).collect(Collectors.toList());
            Long[] s = nonNullIds.toArray(new Long[0]);
            List<News> list = newsSearchIntegrationService.listRecommendedNews(s);
            fillExtInfo(userId, list);
            List<News4Client> news4ClientList = list.stream().map(news -> new News4Client(news, null)).collect(Collectors.toList());
            return ResultUtil.genSuccessResult(news4ClientList);
        } else {
            return ResultUtil.genSuccessResult(new ArrayList<>());
        }

    }

    @Override
    public Json<List<News4Client>> listRecommendNews4ImgNews(Long newsId) {
        //组图末尾推荐的组图
        List<News> list = this.newsMapper.queryRecommendNewsList(newsId, 8, null);
        List<News4Client> news4ClientList = new ArrayList<>();
        for (News aList : list) {
            news4ClientList.add(new News4Client(aList, null));
        }
        return ResultUtil.genSuccessResult(news4ClientList);
    }

    @Override
    public Json<List<NewsSearchWord>> listHotTitle() {
        return ResultUtil.genSuccessResult(this.newsSearchWordMapper.queryList());
    }

    @Override
    public Json<List<NewsKind>> listNewsKinds() {
        return ResultUtil.genSuccessResult(this.newsKindMapper.selectVideoKinds());
    }

    @Deprecated
    @Override
    public Json<FailVideo> getFailVideo(Long newsId) {
        //新闻视频无法播放的处理
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "无效视频");
    }

    @Override
    public Json doSaveNewsClick(Long newsId) {
        this.newsStatisticsMapper.searchRecommended(newsId);
        return ResultUtil.genSuccessResult();
    }

    @Override
    public void fillExtInfo(Long userId, List<News> result) {
        List<NewsCollect> newsCollect = new ArrayList<>();
        if (userId != null) {
            newsCollect = newsCollectMapper.selectByIds(userId, (byte) 1);
        }

        List<NewsCollect> finalNewsCollect = newsCollect;
        result.forEach(item -> {
            item.setLinkUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.DETAIL, item.getId(), userId));
            item.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.SHARE, item.getId(), userId));
            item.setCollect(0);
            if (CollectionUtils.isNotEmpty(finalNewsCollect)) {
                for (NewsCollect collect : finalNewsCollect) {
                    if (item.getId().equals(collect.getNewsId())) {
                        item.setCollect(1);
                        break;
                    }
                }
            }
            item.setLocal((byte) 1);
            if (StringUtils.isNotEmpty(item.getAreaDetail())) {
                item.setLocal((byte) 2);
            }

            if (StringUtils.isNotEmpty(item.getVideoStr())) {
                item.setVideoList(JSONArray.parseArray(item.getVideoStr(), PostImgVo.class));
            }
        });

    }

    @Override
    public Json<NewsDetailVO> getNewsDetailById(NewsParam newsParam,BasicParam basicParam) {
        Json<NewsDetailVO> data = this.getNewsDetail(newsParam);
        NewsDetailVO result = data.getResult();
        NewsVO news = result.getNews();
        String content = news.getContent();
        if (StringUtils.isNotEmpty(content)) {
            if (5 == basicParam.getPlatform()) {
                String appletsContent = assemblyService.deleteVideoLabel(content);
                news.setContent(appletsContent);
            } else {
                String htmlContent = assemblyService.assemblyContent(content);
                news.setContent(htmlContent);
            }
        }
        String shareDesc = HtmlContentUtils.getHtmlDesc(content,50);
        news.setShareDesc(shareDesc);
        result.setNews(news);
        data.setResult(result);
        //Android,ios,H5 拼接编辑寄语到content
        int platform = basicParam.getPlatform();
        boolean flag = (platform == 1 || platform == 2 || platform == 3) && StringUtils.isNotBlank(news.getEditorMessage());
        if (flag){
            news.setContent(news.getContent() + news.getEditorMessage());
        }
        //帖子分享,在在小程序中每日首次打开,分享人随机获得20-50的小红花
        //异步调用 用户首次打开小程序完成分享任务

        NewsService newsService = SpringContextHolder.getBean(NewsService.class);
        newsService.completeTaskAndPush(newsParam,news);
        return data;
    }

}
