package com.bxm.localnews.news.service;

import java.util.List;

import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.news.dto.*;
import com.bxm.localnews.news.param.ForumPostInformParam;
import com.bxm.localnews.news.param.ForumPostListQueryParam;
import com.bxm.localnews.news.param.ForumPostListUserQueryParam;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;

/**
 * 社区相关服务
 */
public interface ForumService {

    /**
     * 查询版块,enable已弃用
     *
     * @param areaCode 地区编码
     * @param enable   查询版块范围：0全部 1已上架
     * @return 分页结果
     */
    List<ForumVo> listForum(String areaCode, Integer enable, Long userId, Integer size, BasicParam basicParam);

    /**
     * 获取版块详情
     *
     * @param id
     * @return
     */
    ForumVo getForumDetail(Long id, Long userId);

    /**
     * 查询话题,enable已弃用，若size=2，则随机取两条
     *
     * @param areaCode 地区编码
     * @param enable   查询话题范围：0全部 1已上架
     * @return 分页结果
     */
    List<TopicVo> listTopic(String areaCode, Integer enable, Integer size, Long userId);

    /**
     * 获取话题详情
     *
     * @param id
     * @return
     */
    TopicVo getTopicDetail(Long id, Long userId);

    /**
     * 根据版块或话题获取帖子列表
     *
     * @param param
     * @return
     */
    PageWarper<ForumPostVo> listForumPost(ForumPostListQueryParam param, BasicParam basicParam);

    /**
     * 根据用户及类型获取帖子列表
     *
     * @param param
     * @return
     */
    PageWarper<ForumPostVo> listForumPostByUser(ForumPostListUserQueryParam param);

    /**
     * 根据id获取帖子详情
     *
     * @param id
     * @param userId
     * @param deviceId
     * @return
     */
    ForumPostVo getForumPostDetail(Long id, Long userId, String deviceId, Long shareUserId, BasicParam basicParam, String areaCode);

    /**
     * 新增或编辑帖子
     *
     * @param forumBasicVo
     * @return
     */
    Message doCreateOrUpdatePost(ForumBasicVo forumBasicVo);

    /**
     * 用户需要在发帖之后的一小时内，看到自己的帖子是置顶的
     * 把用户发布的帖子加入到redis用户帖子置顶中，并调用定时任务在一小时后进行清除置顶
     *
     * @param forumPostVo
     */
    void createPostUserTopTaskJob(ForumPostVo forumPostVo);

    /**
     * 删除帖子
     *
     * @param id
     * @return
     */
    Boolean deleteForumPost(Long id);

    /**
     * 分享帖子，在分享帖子后可以获得金币，每天会有次数限制
     *
     * @param userId
     * @param newsId
     * @param areaCode
     * @return
     */
    NewsCompleTaskDTO doShareForumPost(Long userId, Long newsId, String areaCode, Byte type, Integer platform);

    /**
     * 填充其余参数
     *
     * @param forumPostList
     * @param userId
     */
    void fillExtInfo(List<ForumPostVo> forumPostList, Long userId, String areaCode, boolean isFillTitle);

    /**
     * 记录帖子阅读信息
     *
     * @param userId
     * @param postId
     * @param shareUserId
     * @param platform
     * @param forumPostVo
     */
    void doRecordForumPost(Long userId, Long postId, Long shareUserId, Integer platform, ForumPostVo forumPostVo);

    /**
     * 对某篇帖子进行点赞操作
     *
     * @param userId
     * @param postId
     * @param type
     * @param areaCode
     */
    void doLikeForumPost(Long userId, Long postId, int type, String areaCode, Integer platform);

    /**
     * 举报某篇帖子
     *
     * @param forumPostInformParam
     */
    Message doInformForumPost(ForumPostInformParam forumPostInformParam);

    /**
     * 收藏某篇帖子
     *
     * @param userId
     * @param postId
     */
    void doCollectForumPost(Long userId, Long postId);

    /**
     * 计算参与人数
     */
    void calcParticipantsNum();

    /**
     * 根据帖子id获取与帖子具有相关性的新闻和帖子
     *
     * @param postId
     * @param userId
     * @param size     数量
     * @param areaCode 地区编号
     * @return
     */
    List<News4Client> listPostDetailRecommend(Long postId, Long userId, Integer size, String areaCode);

    /**
     * 获得推荐库中的一个热门帖子或者新闻用于补签
     *
     * @param userId
     * @param areaCode
     * @return
     */
    ReplenishmentDTO getHotPostOrNews(Long userId, String areaCode, Byte type);

    /**
     * 对帖子的消费次数进行消费，对帖子的次数进行增加
     *
     * @param ids
     */
    void doConsumeRecommendedCount(List<Long> ids, Long userId);

    /**
     * 对帖子的点击次数进行消费，对帖子的次数进行增加
     *
     * @param postId
     */
    void doConsumeClickCount(Long postId, Long userId);

    /**
     * 对帖子的有效阅读次数进行消费，对帖子的次数进行增加
     *
     * @param postId
     */
    void doConsumeActiveViewCount(Long postId, Long userId);

    /**
     * 更新帖子中用户头像以及昵称
     *
     * @param userId
     * @param nickname
     * @param headImg
     */
    void triggerUpdateInfo(Long userId, String nickname, String headImg);

    /**
     * 异步调用
     *
     * @param forumPostVo
     */
    void doAsyncCallService(ForumPostVo forumPostVo);

    /**
     * 增加帖子点击次数
     *
     * @param postId
     * @param clickCount
     */
    void doSavePostClick(Long postId, Long clickCount);

    /**
     * 帖子点赞异步执行
     *
     * @param userId
     * @param postId
     * @param type
     * @param areaCode
     */
    void doAsyncLikePost(Long userId, Long postId, int type, String areaCode, byte isOld, Integer platform);

    /**
     * 查看详情时异步调用
     *
     * @param userId
     * @param postId
     * @param shareUserId
     * @param platform
     * @param forumPostVo
     */
    void doAsyncReadPost(Long userId, Long postId, Long shareUserId, Integer platform, ForumPostVo forumPostVo);

    /**
     * 小程序帖子点赞
     *
     * @param postId   帖子id
     * @param userId   用户id
     * @param areaCode 地区编码
     * @return
     */
    PostLikeDto doLikeForumPost(Long postId, Long userId, String areaCode, String userNickname, String headImg, Integer platform);

    /**
     * 查看详情中的点赞列表
     *
     * @param id
     * @param userId
     * @param devcId
     * @return
     */
    List<ForumPostLikeDTO> getPostLikeDetail(Long id, Long userId, String devcId);

    /**
     * 得到帖子鲜花数量
     *
     * @param isBrilliant
     * @return
     */
    int getPostLikeConsume(Integer isBrilliant);

    /**
     * 定时修改帖子为显示的点赞数据
     *
     * @return
     */
    int doRefreshLikeInfo();

    /**
     * 帖子点赞用户头像,用户昵称同步
     */
    void doSyncPostLikeInfo();

    /**
     * 帖子点赞概览,评论概览信息同步
     */
    void doSyncPostStaticInfo();

    /**
     * 统计客户端最近半小时发帖情况
     *
     * @return
     */
    CalculatePostDTO calculatePostCount();

    /**
     * 用户更改信息后同步帖子点赞表的用户头像,用户昵称字段
     *
     * @param userId
     * @param nickname
     * @param headImg
     */
    void doTriggerUpdateForumpostLikeInfo(Long userId, String nickname, String headImg);

    /**
     * 根据帖子id列表获取帖子列表
     *
     * @param postIds
     * @return
     */
    List<ForumPostVo> getPostListByIds(String postIds);

    /**
     * 生成帖子
     *
     * @param forumBasicVo
     */
    Boolean createPost(ForumBasicVo forumBasicVo);
}
