package com.bxm.localnews.news.service.impl;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.common.config.NewsProperties;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.feign.NewsRecommendFeignService;
import com.bxm.localnews.integration.feign.NewsSearchFeignService;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.ESNewsContentDTO;
import com.bxm.localnews.news.dto.VideoDto;
import com.bxm.localnews.news.enums.ShowLevelEnum;
import com.bxm.localnews.news.param.VideoQueryParam;
import com.bxm.localnews.news.service.RecommendService;
import com.bxm.localnews.news.service.VideoRecommendService;
import com.bxm.localnews.news.service.WeightService;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.gexin.fastjson.JSON;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

@Service("recommendService")
public class RecommendServiceImpl extends BaseService implements RecommendService {

    /**
     * 每页获取新闻数量上限
     */
    private static final int MAX_PAGE_SIZE = 20;

    /**
     * 用户缓存警戒值，达到此值就进行用户缓存的追加
     */
    private static final int CACHE_ALARM_VALUE = MAX_PAGE_SIZE * 3;

    private NewsMapper newsMapper;

    private NewsRecommendedMapper newsRecommendedMapper;

    private MarketingActivitiesMapper marketingActivitiesMapper;

    private MarketingActivitiesRecommendedMapper marketingActivitiesRecommendedMapper;

    private AppVersionSupplyService appVersionSupplyService;

    private NewsProperties newsProperties;

    private VideoRecommendService videoRecommendService;

    private RedisListAdapter redisListAdapter;

    private NewsPoolMapper newsPoolMapper;

    private WeightService weightService;

    private NewsRecommendFeignService newsRecommendFeignService;


    private NewsSearchFeignService newsSearchFeignService;

    @Autowired
    public RecommendServiceImpl(NewsMapper newsMapper,
                                NewsRecommendedMapper newsRecommendedMapper,
                                NewsProperties newsProperties,
                                AppVersionSupplyService appVersionSupplyService,
                                VideoRecommendService videoRecommendService,
                                RedisListAdapter redisListAdapter,
                                NewsPoolMapper newsPoolMapper,
                                WeightService weightService,
                                NewsRecommendFeignService newsRecommendFeignService,
                                NewsSearchFeignService newsSearchFeignService,
                                MarketingActivitiesMapper marketingActivitiesMapper,
                                MarketingActivitiesRecommendedMapper marketingActivitiesRecommendedMapper) {
        this.newsMapper = newsMapper;
        this.newsRecommendedMapper = newsRecommendedMapper;
        this.newsProperties = newsProperties;
        this.appVersionSupplyService = appVersionSupplyService;
        this.videoRecommendService = videoRecommendService;
        this.redisListAdapter = redisListAdapter;
        this.newsPoolMapper = newsPoolMapper;
        this.weightService = weightService;
        this.newsRecommendFeignService = newsRecommendFeignService;
        this.newsSearchFeignService = newsSearchFeignService;
        this.marketingActivitiesMapper = marketingActivitiesMapper;
        this.marketingActivitiesRecommendedMapper = marketingActivitiesRecommendedMapper;
    }

    @Override
    public Json<NewsMeta> execRecommend(NewsRecommendParam param) {
        NewsMeta meta = new NewsMeta();
        //调用推荐系统提供的推荐服务
        ResponseEntity<List<Long>> responseEntity = newsRecommendFeignService.recommendList(param.getUserId(),
                param.getKindId() == newsProperties.getRecommendKindId() ? null : param.getKindId().intValue(),
                param.getPagesize(), param.getKindId() == newsProperties.getRecommendKindId() ? param.getAreaCode() : null);
        List<Long> idList = responseEntity.getBody();
        logger.info("推荐引擎返回:{}", responseEntity.getBody());
        if (CollectionUtils.isEmpty(idList)) {
            return ResultUtil.genSuccessResult(meta);
        }
        //获取新闻信息
        List<News> list = newsMapper.findNewsByIds(idList);


        Long[] ids = list.stream().mapToLong(News::getId).boxed().toArray(Long[]::new);
        //获取新闻内容
        ResponseEntity<List<ESNewsContentDTO>> newsRspEntity = newsSearchFeignService.multipleGet(ids);
        List<ESNewsContentDTO> dtos = newsRspEntity.getBody();
        for (News news : list) {
            if (dtos != null && !dtos.isEmpty()) {
                Optional<ESNewsContentDTO> optional = dtos.stream().filter(dto -> dto.getId().longValue() == news.getId().longValue()).findFirst();
                news.setContent(optional.map(ESNewsContentDTO::getContent).orElse(null));
            }
        }
        List<News> result = new ArrayList<>();
        for (Long id : idList) {
            Optional<News> optionalNews = list.stream().filter(news -> news.getId().longValue() == id.longValue()).findFirst();
            optionalNews.ifPresent(result::add);
        }
        //获取营销活动
        List<News> activityNews = addMarketingActivities(param.getUserId(), param.getAreaCode());
        if (CollectionUtils.isNotEmpty(activityNews)) {
            result.addAll(activityNews);
        }
        result = warp(result, param.getPagesize());

        List<News4Client> news4ClientList = result.stream().map(news -> new News4Client(news, null)).collect(Collectors.toList());
        //添加小视频
        if (param.getKindId() == null || this.newsProperties.getRecommendKindId() == param.getKindId()) {
            addRecommendVideo(param, news4ClientList);
        }
        meta.setList(news4ClientList);
        meta.setNewsCount(meta.getList().size());
        //隐藏敏感信息
        hideInfo(param, meta);
        logger.info("推荐完成：{}", meta);
        return ResultUtil.genSuccessResult(meta);
    }

    /**
     * 推荐小视频，每2页显示一次小视频
     *
     * @param param           推荐参数
     * @param news4ClientList 推荐返回结果元对象
     */
    private void addRecommendVideo(NewsRecommendParam param, List<News4Client> news4ClientList) {
        logger.info("NewsRecommendParam：{}", param);
        if (param.getCurPage() != 0 && param.getCurPage() % 2 == 0) {
            VideoQueryParam videoParam = new VideoQueryParam();
            videoParam.setUserId(param.getUserId());
            //在新闻列表中间插入
            List<VideoDto> videoDtoList = videoRecommendService.execRecommend(videoParam);
            if (CollectionUtils.isNotEmpty(videoDtoList)) {
                int size = news4ClientList.size();
                news4ClientList.add(size / 2, new News4Client(videoDtoList));
            }
        }
    }

    /**
     * 插入营销活动
     *
     * @param userId
     * @return
     */
    private List<News> addMarketingActivities(Long userId, String areaCode) {
        List<MarketingActivities> list = marketingActivitiesMapper.listForRecommend(userId, areaCode);
        List<News> newsList = new ArrayList<>();
        for (MarketingActivities activities : list) {
            News news = new News();
            news.setId(activities.getId());
            news.setTitle(activities.getTitle());
            news.setAuthor(activities.getAuthor());
            news.setContent(activities.getLinkUrl());
            news.setKindId(activities.getDeliveryChannel());
            news.setActivity((byte) 2);//1否2是 这个奇葩数值兼容前人挖的坑
            news.setIssueTime(activities.getShowTime());
            String[] coverUrls = StringUtils.split(activities.getCoverUrl(), ",");
            news.setImgNum(coverUrls.length);
            news.setImgUrl(JSON.toJSONString(coverUrls));
            news.setHot((byte) 1);//1否
            news.setTop((byte) 1);//1否
            news.setType((byte) 1);//'新闻类型  1：文章  2：组图  3：视频',
            news.setShowLevel(activities.getShowLevel());
            news.setShowLevelDetail(activities.getShowLevelDetail());
            news.setDeliveryType(activities.getDeliveryType());
            news.setAreaDetail(activities.getAreaDetail());
            news.setLinkUrl(activities.getLinkUrl());
            newsList.add(news);
            marketingActivitiesRecommendedMapper.save(userId, activities.getId());
        }
        return newsList;
    }

    /**
     * 对标签逻辑进行包装
     *
     * @param list
     * @return
     */
    private List<News> warp(List<News> list, Integer pageSize) {

        list = list.stream().sorted(Comparator.comparing(News::getShowLevel)).collect(Collectors.toList());
        List<News> result = new ArrayList<>();
        News news;
        for (int i = 0; i < (pageSize > list.size() ? list.size() : pageSize); i++) {
            news = list.get(i);
            logger.info("title:{},getShowLevelDetail:{}", news.getTitle(), news.getShowLevelDetail());
            logger.info("news.getShowLevelDetail().indexOf(ShowLevelEnum.TOP.getCode().toString())>-1:{}", news.getShowLevelDetail().indexOf(ShowLevelEnum.TOP.getCode().toString()) > -1);
            if (news.getShowLevelDetail() != null && news.getShowLevelDetail().indexOf(ShowLevelEnum.TOP.getCode().toString()) > -1) {
                news.setTop((byte) 2);
            }
            if (news.getShowLevelDetail() != null && news.getShowLevelDetail().indexOf(ShowLevelEnum.HOT.getCode().toString()) > -1) {
                news.setHot((byte) 2);
            }
            if (news.getShowLevelDetail() != null && news.getShowLevelDetail().indexOf(ShowLevelEnum.ACTIVITY.getCode().toString()) > -1) {
                news.setActivity((byte) 2);
            }
            if (news.getShowLevelDetail() != null && news.getShowLevelDetail().indexOf(ShowLevelEnum.LOCAL.getCode().toString()) > -1) {
                news.setLocal((byte) 2);
            }
            result.add(news);
        }
        return result;
    }

    /**
     * 处于提包状态时，隐藏部分敏感信息，用于上包
     * 去除作者信息
     *
     * @param param 推荐参数
     * @param meta  返回结果
     */
    private void hideInfo(NewsRecommendParam param, NewsMeta meta) {
        if (appVersionSupplyService.getPublishState(param)) {
            meta.getList().removeIf(client -> client.getVideos() != null);
            for (News4Client client : meta.getList()) {
                if (client.getNews() != null) {
                    client.getNews().setAuthor(null);
                    client.getNews().setAuthorImg(null);
                }
            }
        }
    }

}
