package com.bxm.localnews.payment.proxy;

import java.util.Collection;
import java.util.Map;

import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.service.PayModeService;
import com.bxm.localnews.payment.service.impl.DefaultPayModeServiceImpl;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.google.common.collect.Maps;

import org.springframework.stereotype.Component;

/**
 * @author zhaoyadong 2019/1/2 14:01
 * @desc
 */
@Component
public class PayProxyService {

    private Map<String, PayModeService> payModeMap;

    private PayModeService defaultPayModeService = new DefaultPayModeServiceImpl();

    /**
     * 下单
     *
     * @param paymentOrderDetail
     */
    public void create(PaymentOrderDetail paymentOrderDetail) {
        getService(PayTypeEnum.getNameByType(paymentOrderDetail.getPayType())).create(paymentOrderDetail);
    }

    public void query(PaymentOrder paymentOrder, PayTypeEnum payType) {
        getService(payType.getName()).query(paymentOrder);
    }

    /**
     * 支付完成后的回调处理
     *
     * @param data    回调内容
     * @param payType 支付方式
     * @return 处理结果
     */
    public String notifyPay(String data, PayTypeEnum payType) {
        return getService(payType.getName()).callBack(data);
    }

    /**
     * 获取支付方式对应的支付服务提供接口，如果没有配置服务则返回默认的支付服务接口
     *
     * @param payType 支付方式
     * @return 支付服务提供接口
     */
    private PayModeService getService(String payType) {
        PayModeService service = getPayModeMap().get(payType);

        if (service == null) {
            service = defaultPayModeService;
        }

        return service;
    }

    /**
     * 从spring context中加载PayModeService的子类，根据支付类型进行匹配
     *
     * @return 支付类型和对应支付方式服务提供类的map
     */
    private Map<String, PayModeService> getPayModeMap() {
        if (null == payModeMap) {
            payModeMap = Maps.newHashMap();
            Collection<PayModeService> payModeServices = SpringContextHolder.getBeans(PayModeService.class);

            for (PayModeService payModeService : payModeServices) {
                if (payModeService.support() != null) {
                    payModeMap.put(payModeService.support().getName(), payModeService);
                }
            }
        }

        return payModeMap;
    }
}
