package com.bxm.localnews.payment.proxy;

import com.bxm.localnews.payment.config.WxPayProperties;
import com.bxm.localnews.payment.request.WxPayQueryWithdrawRequest;
import com.bxm.localnews.payment.request.WxPayWithdrawRequest;
import com.bxm.localnews.payment.result.QueryWithdrawResult;
import com.bxm.localnews.payment.result.WxWithdrawResult;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.github.binarywang.wxpay.bean.notify.WxPayOrderNotifyResult;
import com.github.binarywang.wxpay.bean.order.WxPayAppOrderResult;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.bean.request.WxPayUnifiedOrderRequest;
import com.github.binarywang.wxpay.bean.result.WxPayOrderQueryResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import com.github.binarywang.wxpay.service.impl.WxPayServiceImpl;
import com.thoughtworks.xstream.XStream;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 基于第三方实现的微信提现代理实现
 * @author zhaoyadong 2019/3/11 20:38
 */
@Component
@Slf4j
public class WxWithdrawProxyService extends WxPayServiceImpl {

    private final WxPayService wxPayService;

    private final WxPayService wxAppletPayService;

    private final WxPayProperties wxPayProperties;

    @Autowired
    public WxWithdrawProxyService(WxPayService wxPayService,
                                  WxPayService wxAppletPayService,
                                  WxPayProperties wxPayProperties) {
        this.wxPayService = wxPayService;
        this.wxAppletPayService = wxAppletPayService;
        this.wxPayProperties = wxPayProperties;
    }

    @Override
    public byte[] postForBytes(String url, String requestStr, boolean useKey) {
        return new byte[0];
    }

    @Override
    public String post(String url, String requestStr, boolean useKey) {
        return null;
    }

    private WxPayService getWxPayServiceByType(Byte withType){
        if (null != withType && 2 == withType) {
            return wxAppletPayService;
        }
        return wxPayService;
    }

    /**
     * 提现
     */
    public WxWithdrawResult transfers(WxPayWithdrawRequest request,Byte withType) throws WxPayException {
        WxPayService wxPayService = getWxPayServiceByType(withType);

        request.checkAndSign(wxPayService.getConfig());
        String url = wxPayService.getPayBaseUrl() + "/mmpaymkttransfers/promotion/transfers";
        String responseContent = wxPayService.post(url, request.toXML(), Boolean.TRUE);

        XStream xstream = new XStream();
        XStream.setupDefaultSecurity(xstream);
        xstream.processAnnotations(WxWithdrawResult.class);
        xstream.allowTypes(new Class[]{WxWithdrawResult.class});
        return (WxWithdrawResult) xstream.fromXML(responseContent);
    }

    /**
     * 查询企业付款
     */
    public QueryWithdrawResult queryWithdraw(WxPayQueryWithdrawRequest request,Byte withType) throws WxPayException {
        WxPayService wxPayService = getWxPayServiceByType(withType);

        request.checkAndSign(wxPayService.getConfig());
        String url = this.getPayBaseUrl() + "/mmpaymkttransfers/gettransferinfo";
        String responseContent = super.post(url, request.toXML(), Boolean.TRUE);

        XStream xstream = new XStream();
        XStream.setupDefaultSecurity(xstream);
        xstream.processAnnotations(QueryWithdrawResult.class);
        xstream.allowTypes(new Class[]{QueryWithdrawResult.class});
        return (QueryWithdrawResult) xstream.fromXML(responseContent);
    }

    /**
     * 微信支付结果通知
     */
    @Override
    public WxPayOrderNotifyResult parseOrderNotifyResult(String data) {
        try {
            WxPayService wxPayService = getWxPayServiceByType(null);
            return wxPayService.parseOrderNotifyResult(data);
        } catch (WxPayException e) {
            log.error("微信支付回调异常:", e);
            return null;
        }
    }

    /**
     * 查询微信订单
     */
    public WxPayOrderQueryResult queryOrder(String paymentNum, Byte withType) {
        try {
            WxPayService wxPayService = getWxPayServiceByType(withType);
            return wxPayService.queryOrder(null, paymentNum);
        } catch (WxPayException e) {
            log.error("查询订单[{}]异常: [{}]", paymentNum, e);
            return null;
        }
    }

    /**
     * APP下单返回信息
     */
    public WxPayAppOrderResult createOrder(PaymentOrderDetail paymentOrder, Byte withType) {
        try {
            WxPayService wxPayService = getWxPayServiceByType(withType);
            WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest = generateOrder(paymentOrder);
            return wxPayService.createOrder(wxPayUnifiedOrderRequest);

        } catch (WxPayException e) {
            log.error("微信下单失败！订单号：{},原因:{}", paymentOrder.getPaymentNum(), e.getMessage());
            log.error("下单异常：", e);
        }

        return null;
    }

    /**
     * 统一下单
     */
    private WxPayUnifiedOrderRequest generateOrder(PaymentOrder paymentOrder) {
        WxPayUnifiedOrderRequest wxPayUnifiedOrderRequest = new WxPayUnifiedOrderRequest();

        wxPayUnifiedOrderRequest.setDeviceInfo("WEB");
        wxPayUnifiedOrderRequest.setOutTradeNo(paymentOrder.getPaymentNum());
        wxPayUnifiedOrderRequest.setBody(wxPayProperties.getBody());
        wxPayUnifiedOrderRequest.setSceneInfo(wxPayProperties.getSceneInfo());
        wxPayUnifiedOrderRequest.setTotalFee(BaseWxPayRequest.yuanToFen(paymentOrder.getAmount().toString()));
        wxPayUnifiedOrderRequest.setSpbillCreateIp(paymentOrder.getClientIp());

        return wxPayUnifiedOrderRequest;
    }
}
